//    

/*++ ---------------- diracInt.mu ---------------------
 
Description:
This file contains functions for integrating generalized functions containing
Dirac's delta function.
							  
Functions: 

 - used Parameter:
    ++ f,a,b :DOM_EXPR
    ++ x     :DOM_IDENT				  
    ++ f     = (complex) generalized function (distribution)
    ++ x     = indeterminate of f					  
    ++ a,b   = constant expression w.r.t. x, representing lower and upper limit

 - intlib::diracInt(f,x)
    ++ returns hold(int)(f,x) or an (formal) antiderivative of the generalized 
    ++ function f w.r.t. x, wherby the antiderivative F is defined as follows:
    ++
    ++              F:=int(g(x)*dirac(p(x)),x)
    ++    and
    ++              F= sum( g(x0)/abs(diff(p,x))(x0)*heaviside(x-x0) with p(x0)=0 )
    ++
    ++ For integrating of derivatives of the dirac function the following
    ++ (generalized) rule is used:
    ++
    ++ f(x)*dirac(a*x+b,n)=f(-b/a)*dirac(a*x+b,n)
    ++                     -1/a*D(f)(-b/a)*dirac(a*x+b,n-1)
    ++                     +1/a^2*(D@@2)(f)(-b/a)*dirac(a*x+b,n-2)
    ++                     +...+(-1)^n/a^n*(D@@n)(f)(-b/a)*dirac(a*x+b)
    ++
    ++ NOTE: p(x) needs to be solvable by the method linsolve.
                               
 - intlib::diracDef(f,x,a,b)
    ++ returns hold(int)(f,x=a..b) or the definite integral of the generalized
    ++ function f w.r.t. x in the range of a to b, wherby the integral F
    ++ is defined as follows:
    ++
    ++              F:=int(g(x)*dirac(x-x0),x=a..b)
    ++    and
    ++             { g(x0)       a<x0<b and g continuous in x0
    ++        F =  { undefined   a=x0 or b=x0
    ++             { 0           x0 not in (a,b)
    ++
    ++ NOTE: Currently this usual definition is slightly modified into Fmod
    ++        Fmod  =  { f(x0)*heaviside(0)  a=x0 or b=x0
    ++                 { F                   otherwise
    ++
    ++ For integrating of derivatives of the dirac function the following
    ++ (generalized) rule is used:
    ++
    ++              F':=int(g(x)*dirac(a*x+b,n),x=r1..r2), x0:=-b/a
    ++    and
    ++             { (-1)^n/a^(n+1)*
    ++             { g^(n)(-b/a)            r1<x0<r2 and g continuous in x0
    ++       F' =  { heaviside(0)*
    ++             { (-1)^n/a^(n+1)*
    ++             { g^(n)(-b/a)            r1=x0 or r2=x0
    ++             { 0                      x0 not in (r1,r2)

 - intlib::linArgSolve(f,x)
    ++ solves the linear expression f w.r.t. x and returns a sequence of this 
    ++ soution and may manipulated expression for f (currently simply f) or
    ++ FAIL. 
 - intlib::isContinuousAt(f,x=x0)
    ++ tests, if f is continuous at x=x0 and returns either FALSE or f(x0).
// - intlib::diff2D(f,x)
    ++ replaces as much as possible "diff"-expressions in f w.r.t. x. The rules
    ++ are described near the code of this function.
 - intlib::contDiffAt(f,x=x0,n)
    ++ computes the n-th derivative of f w.r.t. x, replaces as much as 
    ++ possible "diff"-expressions and tests if the resulting expression is
    ++ is continuous at x=x0. The return value is a sequence of the computed
    ++ derivative and either (D@@n)(f)(x0) or FAIL. 

See:
  [1] Foellinger, O. (1986). Laplace- und Fourier-Transformation. Huethig, 
      4. Auflage.

Examples:
++*/

intlib::linArgSolve:=
  proc(f,x)
    local pf,x0;
  begin
    if degree((pf:=poly(f,[x])))=1 then 
      x0:=-coeff(pf,x,0)/coeff(pf,x,1);
      x0, f
    else FAIL
    end_if;
  end_proc: 

intlib::isContinuousAt:=
  proc(f,place)
    local s,x,x0;
  begin
    x:=op(place,1); 
    x0:=op(place,2);

    // if f contains an unknown function
    if hastype(f,"function") then
      //if traperror((s:=eval(subs(f,x=x0))))=0 then
      if traperror((s:=limit(f,x=x0)))=0 then
        return(s)
      else
        return(FALSE)
      end_if;
    end_if;

    s:=discont(f,x=x0..x0); // only over Dom::Real?
    if s={} then
      subs(f,x=x0,EvalChanges)
    else
      FALSE
    end_if;
  end_proc: 

// R1: diff(f(x),x)      -> D(f)(x)
// R2: diff(f(x),x$n)    -> (D@@n)(f)(x)
// R3: diff(f(x,y,..),x) -> D(f)(x,y,..)
// R4: diff(f(x),x,y)    -> diff(D(f)(x),y)
/*
intlib::diff2D:=
  proc(f,x)
  begin
    eval(subs(f,hold(diff)=proc(ff)
                             local xx,n,r;
                           begin
                             xx:=split([args(2..args(0))],has,x);
                             n:=nops(xx[1]);
                             if n=0 then //print("n=0"); 
                                         return(diff(args())) end_if;
                             if nops(xx[2])<>0 then //print("other vars"); 
                               // R4
                               return(diff(intlib::diff2D(
                                           diff(ff,op(xx[1])),x),op(xx[2])))
                             end_if;
                             case type(ff)
                               of "diff" do //print("case diff");
                                 r:=intlib::diff2D(op(ff,1),op(ff,2));
                                 (D@@n)(op(r,0))(op(r)); break;
                               of "function" do //print("case function");
                                 // R1 & R2 & R3
                                 (D@@n)(op(ff,0))(op(ff)); break;
                               otherwise //print("case otherwise");
                                 (D@@(n-1))(map(ff,intlib::diff2D,x))(x)
                             end_case
                           end_proc
              ))
  end_proc: 
*/
// Series::diff2D: convert diffs in an expression into D's
//
// Applying the multivariate chain rule,
// diff(function(y1, y2, ...), x1, x2, ...)) is rewritten as
// diff(  D([1], function)(y1, y2, ...) * diff(y1, x1)
//      + D([2], function)(y1, y2, ...) * diff(y2, x1)
//      + ..., x2, ...)
// and the diff's are converted into D's recursively

/*
intlib::diff2D :=
  proc(f)
    local function, y, x1, otherVariables;
  begin
    if not has(f, diff) then
      return(f);
    end_if;
    userinfo(15, "diff2D called for ".expr2text(f));
    case type(f)
      of "_plus" do
      of "_mult" do
      of "_power" do
        // step recursively into expressions
        return(map(f, intlib::diff2D))
      of "diff" do
        // apply multivariate chain rule and proceed recursively
        function := op(f, [1, 0]);
        y := [op(op(f, 1))];
        x1 := op(f, 2);
        otherVariables := op(f, 3..nops(f));
        return(intlib::diff2D(diff(
                _plus(D([i], function)(op(y)) * intlib::diff2D(diff(y[i], x1))
                      $ i = 1..nops(y)),
                                   otherVariables)))
    end_case;
    // default: do nothing
    f
  end_proc:
*/

intlib::contDiffAt:=
  proc(f,place,n)
    local x,x0,df,dfx0;
  begin
    x:=op(place,1); x0:=op(place,2);
    //df:=intlib::diff2D(diff(f,x$n),x);
    df:=rewrite(diff(f,x$n),D);
    //df:=diff(f,x$n);
    dfx0:=intlib::isContinuousAt(df,x=x0);                      
    df,(if dfx0=FALSE then FAIL else dfx0 end_if)
  end_proc:

intlib::diracInt:=
  proc(f,x) 
    local options, g,delta,s,r,x0,gx0,p,n,linarg,i,dg;
  begin
    //print("entering diracInt with ",f,x);
    options:=intlib::getOptions(args(3..args(0)));                  
    f:= expand(f, ArithmeticOnly):
    case type(f)
      of "_plus" do //print("case _plus");
        return( map(f, e->if hastype(e,"dirac") then
                               intlib::diracInt(e,x) 
                          else intlib::int(e,x) 
                          end_if)); break;
      of "dirac" do //print("case dirac");
        g:=1; 
        delta:=f; 
        break;
      of "_mult" do //print("case _mult");
        s:=split(f, i -> hastype(i,"dirac") and has(i, x));
        if type(s[1])="dirac" then 
           delta:=s[1]; 
           g:=s[2]*s[3] 
        else 
	   if s[1] = 1 then
	      // there were diracs, but they did not depend on x.
	      // Extract them from the integral:
              s:=split(f, i -> hastype(i,"dirac"));
	      if s[1] <> 1 then 
                return(s[1]*int(s[2]*s[3], x)):
	      end_if:
	   end_if;
	   // there were strange entries with diracs (such as
	   // products of diracs). No idea, what to do with them:
	   g:= simplify::dirac(f);
	   if f <> g then
             return(int(g,x,intlib::printOptions(options)));
	   else
             return(hold(int)(g,x,intlib::printOptions(options)));
	   end_if:
        end_if; 
        break;
      otherwise //print("case otherwise");
        //return(FAIL);
        return(hold(int)(f,x));
    end_case;

    // now f(x)=g(x)*delta(p(x)) or f(x)=g(x)*delta(p(x),n)
    p:=op(delta,1);
    if nops(delta)=1 then
      n:=null()
    else
      n:=op(delta,2); 
      if not(domtype(n)=DOM_INT and n>0) then return(FAIL)
      end_if;
    end_if;
    if (linarg:=intlib::linArgSolve(p,x))<>FAIL then
      r:=0;
      // p:= s*(x - x0);
      x0:=linarg[1];
      s:= diff(p, x);
      if not has(x0,I) then //heaviside at most nonzero for real values
        case nops(delta)
          of 1 do
            if (gx0:=intlib::isContinuousAt(g/s,x=x0))<>FALSE then
                            // g continuous in x0
              r:=r+gx0*heaviside(linarg[2]); 
            else r:=FAIL; end_if; break;
          of 2 do
            dg:=intlib::contDiffAt(g/s,x=x0,0);
            r:=r+dg[2]*dirac(linarg[2],n-1);
            for i from 1 to n-1 do
              dg:=map([intlib::contDiffAt(dg[1],x=x0,1)],_divide,s);
              r:=r+(-1)^i*binomial(n, i)*dg[2]*dirac(linarg[2],n-1-i); 
              if r=FAIL then break; end_if;
            end_for;
            dg:=map([intlib::contDiffAt(dg[1],x=x0,1)],_divide,s);
            r:=r+(-1)^n*dg[2]*heaviside(linarg[2]);
            break;
        otherwise
          r:=FAIL
        end_case;
        r;
      else r:=FAIL
      end_if
    else r:=FAIL
    end_if;
    //r
    if r=FAIL then 
       hold(int)(f,x) 
    else 
       r 
    end_if
  end_proc:

        
intlib::diracDef:=
  proc(f,x,a,b) // options 
    local g,delta,s,r,x0,gx0,p,linarg,options,n, i;
  begin
    options:=intlib::getOptions(args(5..args(0)));                  
  // print("entering diracDef with ",f,x);
    f:= expand(f, ArithmeticOnly);
    case type(f)
      of "_plus" do //print("case _plus");
        return( map(f, e->if hastype(e,"dirac") then
                            intlib::diracDef(e,x,a,b,options) 
                          else 
                            // Walter 7.6.06: call defInt rather than
                            // defInt_intern. Example:
                            // int(dirac(x) + heaviside(x)/x, x = 1..3)
                            // did not work:
                            // Old:
                            // intlib::defInt_intern(e,x=a..b,options) 
                            // New:
                               intlib::defInt(e,x=a..b,options) 
                          end_if)); break;
      of "dirac" do //print("case dirac");
        g:=1; 
        delta:=f; 
        break;
      of "_mult" do //print("case _mult");
        s:=split(f, i -> hastype(i,"dirac") and has(i, x));
        if type(s[1])="dirac" then 
           delta:=s[1]; 
           g:=s[2]*s[3] 
        else 
	   if s[1] = 1 then
	      // there were diracs, but they did not depend on x.
	      // Extract them from the integral:
              s:=split(f, i -> hastype(i,"dirac"));
	      if s[1] <> 1 then 
                return(s[1]*int(s[2]*s[3], x = a..b,intlib::printOptions(options)))
	      end_if:
	   end_if;
	   // there were strange entries with diracs (such as
	   // products of diracs). No idea, what to do with them:
	   g:= simplify::dirac(f);
	   if f <> g then
             return(int(g,x=a..b,intlib::printOptions(options)))
	   else
             return(hold(int)(g,x=a..b,intlib::printOptions(options)))
	   end_if:
        end_if; 
	break;
      otherwise //print("case otherwise");
        //return(FAIL);
        return(hold(int)(f,x=a..b,intlib::printOptions(options)))
    end_case;

    // now f(x)=g(x)*delta(p(x)) or f(x)=g(x)*delta(p(x),n)
    p:=op(delta,1);
    if nops(delta)=1 then
      n:=null()
    else
      n:=op(delta,2); 
      //if not(is(n,Type::Integer)=TRUE and is(n>0)=TRUE) then return(FAIL)
      if not(domtype(n)=DOM_INT and n>0) then return(FAIL)
      end_if;
    end_if;
    if (linarg:=intlib::linArgSolve(p,x))<>FAIL then
      r:=0;
      x0:=linarg[1];
      s:= diff(p, x);
      // use abs(s), because dirac(-a*x) = dirac(a*x)=dirac(x)/abs(s)
      s:= abs(s);
      case nops(delta)
        of 1 do
          gx0:= intlib::isContinuousAt(g/s,x=x0);
          if gx0 <> FALSE then
             // heaviside does not react to properties,
             // so branch here into the following cases:
             if ((a = -infinity or is(a-x0 < 0)=TRUE) and
                 (b =  infinity or is(b-x0 > 0)=TRUE)
                ) // x0 element in range (a,b)
                then
                  r:= r + gx0;
             elif (a = -infinity or is(a-x0<0)=TRUE) then
                  r:= r+gx0*heaviside(b - x0);
             elif (b = infinity or is(b-x0>0)=TRUE) then
                  r:= r + gx0*heaviside(x0 - a);
             else r:= r + gx0*heaviside(b - x0) - gx0*heaviside(a - x0);
             end_if;
          else 
             r:=FAIL            
          end_if;
          break;

        of 2 do
          if iszero(a - x0) or iszero(b - x0) then
               r:= r +(-1)^n/(s)^n*
                             intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(b - x0) 
                     -(-1)^n/(s)^n*
                             intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(a - x0);
               if not iszero(a - x0) then
                 r:= r - _plus( (-1)^i*binomial(n, i)/s^n*
                                intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(a - x0, n - 1 - i)
                               $ i = 0..n - 1);
               end_if;
               if not iszero(b - x0) then
                 r:= r + _plus( (-1)^i*binomial(n, i)/s^n*
                                intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(b - x0, n - 1 - i)
                               $ i = 0..n - 1);
               end_if;
          elif ((a = -infinity or is(a-x0 < 0)=TRUE) and
                (b =  infinity or is(b-x0 > 0)=TRUE)
               ) then // x0 element in range (a,b)
            r:= r + (-1)^n/(s)^n*
                        intlib::contDiffAt(g/s,x=x0,n)[2];
          elif (a = -infinity or is(a-x0<0)=TRUE) then
            r:= r + _plus( (-1)^i*binomial(n, i)/s^n*
                           intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(b - x0, n - 1 - i)
                          $ i = 0..n - 1)
                  +(-1)^n/(s)^n*
                        intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(b - x0);
          elif (b = infinity or is(b-x0>0)=TRUE) then
            r:= r - _plus( (-1)^i*binomial(n, i)/s^n*
                           intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(a - x0, n - 1 - i)
                          $ i = 0..n - 1)
                  +(-1)^n/(s)^n*
                        intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(x0 - a);
          else 
            r:= r + _plus( (-1)^i*binomial(n, i)/s^n*
                           intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(b - x0, n - 1 - i)
                          $ i = 0..n - 1)
                  - _plus( (-1)^i*binomial(n, i)/s^n*
                           intlib::contDiffAt(g/s,x=x0,i)[2]*dirac(a - x0, n - 1 - i)
                          $ i = 0..n - 1)
                  +(-1)^n/(s)^n*
                        intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(b - x0) 
                  -(-1)^n/(s)^n*
                        intlib::contDiffAt(g/s,x=x0,n)[2]*heaviside(a - x0);
          end_if; 
          break;
      otherwise
        r:=FAIL;
      end_case;
      if r=FAIL then 
        hold(int)(f,x=a..b,intlib::printOptions(options))
      else
         r 
      end_if;
    else
      hold(int)(f,x=a..b,intlib::printOptions(options))
    end_if;
  end_proc:
