/*++ ---------------- specfunc.mu ---------------------

Description:
This file contains functions for integrating generalized functions as well as
special functions and "functions" like "big Oh".
							  
Functions: 

 - used Parameter:
    ++ f,a,b :DOM_EXPR
    ++ x     :DOM_IDENT				  
    ++ f     = (complex) generalized function 
    ++ x     = indeterminate of f					  
    ++ a,b   = constant expression w.r.t. x, representing lower and upper limit

 - intlib::specialFunctionInt(f,x)
    ++ returns FAIL or an (formal) antiderivative of the generalized function
    ++ f w.r.t. x.                               
 - intlib::specialFunctionDef(f,x,a,b, options) 
    ++ returns FAIL or the definite integral of the generalized function f 
    ++ w.r.t. x in the range of a to b.
 - intlib::absInt(f,x)
    ++ rewrites occuring absolute values with the rule |u|=u/sign(u) and
    ++ tries to integrate the resulting expression. If f does not contain an
    ++ absolute value or a sign-function FAIL is returned.
 - intlib::bigOhInt(f,x)
    ++ returns the integral unevaluated or an formal antiderivative of f
    ++ w.r.t. x, if it contains an order term.

Examples:
++*/

intlib::specialFunctionDef:=
  proc(f,x,a,b) // options
    local options;
  begin
    options:=intlib::getOptions(args(5..args(0)));
    
    userinfo(4,"looking for special functions");
    // integrand contains absolute value
    if has(f,{abs,sign}) then
      return(intlib::antiderivative(f,x=a..b,options))
    end_if;
     // big Oh
    if hastype(f,O) then
      return(hold(int)(f,x=a..b,intlib::printOptions(options)))
    end_if;
    // generalized functions
    if has(f,dirac) then return(intlib::diracDef(f,x,a,b,options));
      // heaviside case is treated by indefinite integration 
    end_if;

    // otherwise
    FAIL;
  end_proc:

intlib::specialFunctionInt:=
  proc(f,x,options=table())
    local T: DOM_TABLE,
    substitute: DOM_PROC,
    g, i, oldg, fn;
  begin

    // local method substitute
    // replaces one call ceil(..), floor(..), trunc(..) by an identifier X
    // and adds an entry the argument = X to table T
    substitute:=
    proc(subExpression): DOM_IDENT
      local id: DOM_IDENT;
    begin
      if contains(T, subExpression) then
        T[subExpression]
      else  
        id:= genident("CeilTruncFloor");
        T[subExpression]:= id;
        id
      end_if
    end_proc;
    
    userinfo(4,"looking for special functions");
    // generalized functions
    if has(f, dirac) then
      return(intlib::diracInt(f,x,options));
    elif has(f, heaviside) then
      return(intlib::heavisideInt(f,x,options));
    end_if;
    // integrand contains absolute value or a sign function
    if has(f,{abs,sign}) then
      return(intlib::absInt(f,x,options))
    end_if;
    // big Oh
    if hastype(f,O) then
      return(intlib::bigOhInt(f,x,options))
    end_if;
    // piecewise constant functions
    if has(f, {ceil, floor, trunc}) then
      // replace each call to a constant function by
      // an identifier, integrate, then substitute back
      T:= table();
      g:= misc::maprec(f, {"floor", "ceil", "trunc"} = substitute);
      g:= int(g, x, options);
      if type(g)="int" then
        return(hold(int)(f, x, intlib::printOptions(options)))
      else
        oldg := FAIL;
        while g <> oldg do
          oldg := g;
          g := subs(g, op(i, 2)=op(i, 1) $i in T);
        end_while;
        return(g);
        end_if;  
    end_if;
    // functions that are integrals, such as Si, Ci,
    // and products involving them
    if contains({"Ci", "Si", "erf", "erfi"}, type(f)) then
      return(eval(intlib::byparts(hold(int)(f,x,options), 1)))
    end_if;
    if type(f) = "_mult" then
      for i from 1 to nops(f) do
        if contains({"Ci", "Si", "erf", "erfi"}, type(op(f, i))) then
          g:= eval(intlib::byparts(hold(int)(f,x,options), f / op(f, i)));
          if not hastype(g, "int") then
            return(g)
          end_if
        end_if
      end_for
    end_if;
    
    // {ln, arctan, ...}(f(x)) may be easier to handle by partial integration.
    for fn in {["ln", g -> x/g*diff(g, x)],
               ["arctan", g -> x*diff(g, x)/(g^2+1)],
               ["arcsin", g -> x*diff(g, x)/sqrt(1-g^2)],
               ["arccos", g -> -x*diff(g, x)/sqrt(1-g^2)],
               ["arcsinh", g -> x*diff(g, x)/sqrt(g^2+1)],
               ["arccosh", g -> x*diff(g, x)/sqrt(g^2-1)],
               ["arctanh", g -> x*diff(g, x)/(1-g^2)]} do
      if type(f) = fn[1] then
        g := op(f);
        g := normal(fn[2](g));
        if intlib::Simplify::defaultValuation(g) < intlib::Simplify::defaultValuation(f) then
          g := int(g, x, options);
          if not hastype(g, "int") then
            return(x*f - g);
          end_if;
        end_if;
      end_if;
    end_for;
    
    // otherwise
    FAIL;
  end_proc:

intlib::absInt:=
  proc(f,x,options=table())
    local fs,fr,F,sI;
  begin
    // avoiding infinite loops between abs and sign
    if intlibDepthLevel <> hold(intlibDepthLevel) and intlibDepthLevel > 4 then
      return(hold(int)(f,x,intlib::printOptions(options)));
    end_if;
    
    fs:=subs(f,hold(abs)=(u -> if has(u, x) then u*conjugate(sign(u)) else abs(u) end_if),EvalChanges);
    fr:=rationalize(fs, DescendInto = proc(ex) 
                                      begin
                                        _lazy_and(domtype(ex)=DOM_EXPR,
                                                  has(ex, x),
                                                  _lazy_or(type(ex)<>"sign",
                                                    is(op(ex) in R_, Goal=TRUE) <> TRUE))
                                      end_proc);
    // substituting I away is a bad idea, since sign(x+I)
    // is not piecewise constant:
    sI := select(fr[2], eq -> op(eq, 2) = I);
    if sI <> {} then
      fr[1] := subs(fr[1], sI);
      fr[2] := fr[2] minus sI;
    end_if;
    if fr[2]<>{} then
      F:=intlib::int(fr[1],x,options);
      if type(F)<>"int" then
        subs(F,fr[2],EvalChanges);
      else
        hold(int)(f,x,intlib::printOptions(options));
      end_if;
    else
      fs := rewrite(fr[1], piecewise);
      if fs <> fr[1] then
        return(intlib::int(fs, x, options));
      end_if;
      FAIL
    end_if;
  end_proc:

intlib::bigOhInt:=
  proc(f,x,options) 
    local g;
  begin
    //print("entering bigOhInt with ",f,x);
    case type(f)
      of "_plus" do //print("case _plus");
        return( map(f, e->if hastype(e,O) then
                            intlib::bigOhInt(e,x,options) 
                          else intlib::int(e,x,options)
                          end_if)); break;
      of O do break; //print("case O");
      otherwise //print("case otherwise");
        return(hold(int)(f,x,intlib::printOptions(options))); 
    end_case;

    // now f(x)=O(g(x))
    g:=op(f,1);
    O::create(int(g,x,options),f::dom::points(f));
  end_proc:

