//      

/* ------------------------------------------------------------- */
/* --- plot                                                  --- */
/* ---                                                       --- */
/* --- The library domain for 2- and 3-dimensional graphics. --- */
/* ------------------------------------------------------------- */

plot:= newDomain("plot"):

plot::create_dom:=hold(plot):
plot::info_str:= FAIL:
plot::info:= "Library 'plot': graphical primitives and functions for two- and three-dimensional plots":

// slot used by info:
plot::_exported := FAIL:
plot::exported := {}:


plot::callCounter := proc()
                       option escape;
                       local count;
                     begin
                       count := 0;
                       () -> (count := count + 1)
                     end():

/*--
    plot::new -- function to put precreated graphics on the screen

    Syntax:

    plot(g1 <, g2, ...>  <, op1, op2, ...>)

    g1, g2, ...   : graphical objects
    op1, op2, ... : scene options, see plot2d and plot3d for details

    show either creates a canvas of type plot::Canvas out of the given
    graphical objects and puts it on the screen or if a complete canvas
    of type plot::Canvas is given, it is shown on the screen.

    Example:

    >> plot(plot::Function2d(sin(t), t = 1..3))

    >> plot(plot::Function3d(sin(t + u), t = 0..2*PI, u = -1..1))

    >> plot(plot::Point2d(1,2))
--*/
plot::new :=
proc() local tmp, fileName, exportOpts, i,
             exportMode,   // bool - ususal plot or export?
             exportByVcam, // bool - do we need to call VCam for export?
             exportSTL,    // bool - export to STL file?
             Args, Opts, dummy, plotFileName, call, plotArg, res;
  name plot;
begin
  exportMode   := FALSE;
  exportByVcam := FALSE;
  exportSTL    := FALSE;
  plotFileName := "";

  // first get default filename
  fileName   := plot::getDefault(OutputFile)();

  // look for export options OutputFile and OutputOptions
  [exportOpts, Args, dummy] := split([eval(args())], x -> type(x) = "_equal" and
                                     (op(x, 1) = OutputFile or
                                      op(x, 1) = OutputOptions));

  // check export options
  exportOpts := table(exportOpts);
  if contains(exportOpts, OutputFile) then
    if testtype(exportOpts[OutputFile], Type::Function)  or
      testtype(exportOpts[OutputFile], DOM_STRING) then
      fileName := (exportOpts[OutputFile])();
    else
      error("option 'OutputFile': string expected")
    end_if;
  end_if;
  if contains(exportOpts, OutputOptions) then
    if domtype(exportOpts[OutputOptions]) = DOM_LIST then
      //  merge with defaults
      exportOpts := table(plot::getDefault(OutputOptions),
                          op(plot::checkOptions("Default", [OutputOptions=
                                      exportOpts[OutputOptions]]), [1,2]));
    else
      error("option 'OutputOptions': list of output options expected")
    end_if;
  end_if;

  if fileName <> "" then
    exportMode := TRUE;
    if WRITEPATH <> hold(WRITEPATH) and
      // make sure fileName contains no path delimiters
      {fileName[i] $ i = 1..length(fileName)} intersect
      {"\\", "/", ":"} = {} then
      fileName := WRITEPATH.fileName
    end_if;

    if length(fileName) > 3 and
      stringlib::lower(fileName[-4..-1]) = ".xvc" then
      // library can export directly
      plotFileName := fileName;
    elif length(fileName) > 3 and
      stringlib::lower(fileName[-4..-1]) = ".stl" then
      // library can export directly
      exportSTL := TRUE;
      plotFileName := fileName;
    elif not strmatch(fileName, "\\.") then
      // fileName has no extension
      plotFileName := fileName.".xvc";
    else
      // call vcam for conversion
      exportByVcam := TRUE;
      // build call of vcam
      call := null();

      if exportOpts[ReduceTo256Colors] = TRUE then
        call := call, hold(ReduceColors)
      end_if;
      if contains(exportOpts, DotsPerInch) then
        call := call, hold(Resolution) = exportOpts[DotsPerInch]
      end_if;
      if contains(exportOpts, FramesPerSecond) then
        call := call, hold(Frames) = exportOpts[FramesPerSecond]
      end_if;
      if contains(exportOpts, Quality) then
        call := call, hold(Quality) = exportOpts[Quality]
      end_if;
      if contains(exportOpts, JPEGMode) then
        call := call, hold(Mode) = exportOpts[JPEGMode]
      end_if;
      if contains(exportOpts, EPSMode) then
        call := call, hold(Mode) = exportOpts[EPSMode]
      end_if;
      if contains(exportOpts, AVIMode) then
        call := call, hold(Mode) = exportOpts[AVIMode]
      end_if;
      if contains(exportOpts, WMFMode) then
        call := call, hold(Mode) = exportOpts[WMFMode]
      end_if;
      plotFileName := "";
    end_if;
  end_if;

  // Preprocess input objects with plot::easy, if available. 
  // andi, 2007-07-03
  if plot::easy <> FAIL then
     Args:= [ plot::easy(op(Args)) ];
  end_if;

  [Args, Opts, dummy] := split(Args, x -> type(x) = "plotObject");

  if nops(Args) = 1 and (Args[1])::dom::objType = "Canvas" then
    // argument was a Canvas
    plotArg := Args[1];
    if Opts <> [] then
      // arguments are 1 Canvas and some other attributes
      plotArg := plot::modify(plotArg, op(Opts))
    end_if;
  elif map({op(Args)}, x -> x::dom::objType) intersect {"Canvas"} = {} then
    plotArg := plot::Canvas(op(Args), op(Opts));
  else
    // there was  more than 1 Canvas or a mixture of a Canvas and
    // other plot primitives
    error("Only one Canvas allowed")
  end_if;

  // check if object is consistent
  plotArg::dom::checkObjectRecursive(plotArg):

  if exportSTL then
    plot::STL::openOutputFile(plotFileName);
    res := plot::STL(plotArg);
    tmp := plot::STL::closeOutputFile();
  else
    plot::MuPlotML::openOutputFile(plotFileName);
    res := plot::MuPlotML(plotArg);
    tmp:= plot::MuPlotML::closeOutputFile();
  end_if;

  // 2003-09-17, tonner+andi
  // undocumented feature for web applications
  // plot returns the list of created files if
  // plot::verbose was set to TRUE

  if exportMode then
    if exportByVcam then
      if contains(exportOpts, PlotAt) then
        fileName := fileName[1..-5]."-%n".fileName[-4..-1];
        tmp := stdlib::vcam([[tmp = fileName, exportOpts[PlotAt]]],
                            hold(Remove), call);
      else
        tmp := stdlib::vcam([tmp = fileName], hold(Remove), call);
      end_if;
    end_if;
    if plot::verbose() then
      if domtype(tmp) <> DOM_LIST then
        return(["##PLOTOUTPUTFILELIST##",op(tmp)])
      else
        return(["##PLOTOUTPUTFILELIST##"].tmp);
      end_if;
    end_if;
  else
    if sysname() = "UNIX" then
      if package::which("javaview") <> FAIL then  // javaview loaded
        javaview::plot(tmp, res, exportOpts)
      else // not package javaview
	    clientMessage(clientObject(), "plot_result_file", tmp);
      end_if;
    else
      clientMessage(clientObject(), "plot_result_file", tmp);
    end_if;
  end_if;
  null();
end_proc:


// other name for plot::new:
display := subsop(plot::new, 6=NIL):

/*--
    plot::replot -- function used for interacive replots
--*/
plot::inReplot := FALSE:
plot::replot :=
proc()
  local tmp, Args, plotArg, res;
  option hold;
begin
  sysassign(plot::inReplot, TRUE);
  Args := [context(args())];


  // usually not needed, just in case
  if nops(Args) = 1 and type(Args[1]) = "plotObject"
    and (extop(Args[1], 1))::objType = "Canvas" then
    plotArg := Args[1];
  else
    plotArg := plot::Canvas(op(Args));
  end_if;

  res := traperror((// check if object is consistent
                    plotArg::dom::checkObjectRecursive(plotArg):

                    plot::MuPlotML::openOutputFile("");
                    plot::MuPlotML(plotArg);
                    tmp:= plot::MuPlotML::closeOutputFile();
                    clientMessage(clientObject(), "plot_result_file", tmp);
                    ));

  sysassign(plot::inReplot, FALSE);
  if res <> 0 then
    lasterror()
  end_if;

  null();
end_proc:

plot::interface :=
  {
   // functions
   hold(getDefault),
   hold(setDefault),
   hold(copy),
   hold(modify),
   hold(easy),
   hold(hull),
   hold(delaunay),

   // technical domains
   hold(Canvas),
   hold(Scene2d),
   hold(Scene3d),
   hold(CoordinateSystem2d),
   hold(CoordinateSystem3d),
   hold(Camera),
   hold(AmbientLight),
   hold(PointLight),
   hold(DistantLight),
   hold(SpotLight),
   hold(ClippingBox),

   // Transformations
   hold(Translate2d),
   hold(Reflect2d),
   hold(Rotate2d),
   hold(Scale2d),
   hold(Transform2d),
   hold(Translate3d),
   hold(Reflect3d),
   hold(Rotate3d),
   hold(Scale3d),
   hold(Transform3d),

   // Groups
   hold(Group2d),
   hold(Group3d),

   // Graphical Primitives
   hold(Arc2d),
   hold(Arc3d),
   hold(Arrow2d),
   hold(Arrow3d),
   hold(Bars2d),
   hold(Bars3d),
   hold(Box),
   hold(Boxplot),
   hold(Circle2d),
   hold(Circle3d),
   hold(Cone),
   hold(Conformal),
   hold(Curve2d),
   hold(Curve3d),
   hold(Cylinder),
   hold(Cylindrical),
   hold(Density),
   hold(Dodecahedron),
   hold(Ellipse2d),
   hold(Ellipse3d),
   hold(Ellipsoid),
   hold(Function2d),
   hold(Function3d),
   hold(Hatch),
   hold(Hexahedron),
   hold(Histogram2d),
   hold(Icosahedron),
   hold(Implicit2d),
   hold(Implicit3d),
   hold(Inequality),
   hold(Integral),
   hold(Iteration),
   hold(Line2d),
   hold(Line3d),
   hold(Listplot),
   hold(Lsys),
   hold(Matrixplot),
   hold(MuPADCube),
   hold(Octahedron),
   hold(Ode2d),
   hold(Ode3d),
   hold(Parallelogram2d),
   hold(Parallelogram3d),
   hold(Piechart2d),
   hold(Piechart3d),
   hold(Plane),
   hold(Point2d),
   hold(Point3d),
   hold(PointList2d),
   hold(PointList3d),
   hold(Polar),
   hold(Polygon2d),
   hold(Polygon3d),
   hold(Prism),
   hold(Pyramid),
   hold(QQplot),
   hold(Raster),
   hold(Rectangle),
   hold(Rootlocus),
   hold(Scatterplot),
   hold(Sequence),
   hold(SparseMatrixplot),
   hold(Sphere),
   hold(Spherical),
   hold(Streamlines2d),
   hold(Sum),
   hold(Surface),
   hold(SurfaceSet),
   hold(SurfaceSTL),
   hold(Sweep),
   hold(Tetrahedron),
   hold(Text2d),
   hold(Text3d),
   hold(Tube),
   hold(Turtle),
   hold(VectorField2d),
   hold(VectorField3d),
   hold(Waterman),
   hold(XRotate),
   hold(ZRotate)
   }:

// 2003-09-17, tonner+andi
// undocumented feature for web applications
// plot returns the list of created files if
// plot::verbose was set to TRUE
plot::verbose := proc()
                   option escape;
                   local i, old;
                 begin
                   i := FALSE;
                   proc(j)
                   begin
                     if args(0) = 1 then
                       if contains({TRUE, FALSE}, j) then
                         old := i;
                         i := j
                       else
                         error("TRUE or FALSE expected")
                       end_if
                     else
                       return(i)
                     end_if;
                     old
                   end_proc;
                 end_proc():


//  Up to how many points should be visible in the object inspector?
//  Used in Polygon2/3d, PointList2/3d
plot::pointsInObjectInspector := proc(i)
                                 option escape;
                               begin
                                 proc(j)
                                 begin
                                   if args(0) = 1 then
                                     if domtype(j) = DOM_INT then
                                       i := j
                                     else
                                       error("integer expected")
                                     end_if
                                   else
                                     i
                                   end_if
                                 end
                                 end(10^3):

//  treshhold for automatic scaling switch
plot::scalingRatio := proc(i)
                        option escape;
                      begin
                        proc(j)
                        begin
                          if args(0) = 1 then
                            if testtype(j, Type::Positive) then
                              i := float(j)
                            else
                              error("positive number expected")
                            end_if
                          else
                            i
                          end_if
                        end
                      end(3.5):

// adds an opacity component to a RGB color
plot::opacity := proc(color : Type::ListOf(Type::Arithmetical, 3, 3),
                      opacity : Type::Arithmetical)
                 begin
                   color.[opacity]
                 end_proc:


// set one or more hints in object
plot::setHint := proc(obj)
                   local eq;
                 begin
                   for eq in [args(2..args(0))] do
                     obj::dom::setValue(extop(obj, 1), eq);
                   end_for;
                   TRUE
                 end_proc:

// event dispatcher
plot::onEvent :=
  proc(event : DOM_STRING, canvasID : DOM_STRING, objectID : DOM_STRING)
    local rep, canvas, object;
  begin
    // resolve canvas and onject
    rep := newDomain(canvasID);
    if rep::objType <> FAIL then
      canvas := new(slot(plot, rep::objType), rep);
    else
      return()
    end_if;
    rep := newDomain(objectID);
    if rep::objType <> FAIL then
      object := new(slot(plot, rep::objType), rep);
    else
      object := FAIL
    end_if;

    if canvas::onEvent <> FAIL then
      // call method onEvent of Canvas
      canvas::onEvent(event, canvas, object, args(4..args(0)));
    else
      // canvas has no onEvent slot, silently ignore
    end_if;
  end_proc:

/*
 *  Auxiliary domains (not graphical primitive)
 */
alias(path = pathname("PLOT", "AUXILIARY")):

plot::Camera      := loadproc(plot::Camera,      path, "Camera"):
plot::Canvas      := loadproc(plot::Canvas,      path, "Canvas"):
plot::ClippingBox := loadproc(plot::ClippingBox, path, "ClippingBox"):
plot::CoordinateSystem2d :=
              loadproc(plot::CoordinateSystem2d, path, "CoordinateSystem2d"):
plot::CoordinateSystem3d :=
              loadproc(plot::CoordinateSystem3d, path, "CoordinateSystem3d"):
plot::AmbientLight:= loadproc(plot::AmbientLight,path, "AmbientLight"):
plot::PointLight  := loadproc(plot::PointLight,  path, "PointLight"):
plot::DistantLight:= loadproc(plot::DistantLight,path, "DistantLight"):
plot::SpotLight   := loadproc(plot::SpotLight,   path, "SpotLight"):
plot::Scene2d     := loadproc(plot::Scene2d,     path, "Scene2d"):
plot::Scene3d     := loadproc(plot::Scene3d,     path, "Scene3d"):

plot::Translate2d := loadproc(plot::Translate2d, path, "Translate2d"):
plot::Reflect2d   := loadproc(plot::Reflect2d,   path, "Reflect2d"):
plot::Rotate2d    := loadproc(plot::Rotate2d,    path, "Rotate2d"):
plot::Scale2d     := loadproc(plot::Scale2d,     path, "Scale2d"):
plot::Transform2d := loadproc(plot::Transform2d, path, "Transform2d"):
plot::Translate3d := loadproc(plot::Translate3d, path, "Translate3d"):
plot::Reflect3d   := loadproc(plot::Reflect3d,   path, "Reflect3d"):
plot::Rotate3d    := loadproc(plot::Rotate3d,    path, "Rotate3d"):
plot::Scale3d     := loadproc(plot::Scale3d,     path, "Scale3d"):
plot::Transform3d := loadproc(plot::Transform3d, path, "Transform3d"):

plot::Group2d     := loadproc(plot::Group2d,     path, "Group2d"):
plot::Group3d     := loadproc(plot::Group3d,     path, "Group3d"):

plot::View2d      := loadproc(plot::View2d,      path, "View2d"):
plot::View3d      := loadproc(plot::View3d,      path, "View3d"):

unalias( path ):


/*
 *  The graphical primitives
 */
alias(path = pathname("PLOT", "PRIMITIV")):

plot::Arc2d        := loadproc(plot::Arc2d,        path, "Arc2d"):
plot::Arc3d        := loadproc(plot::Arc3d,        path, "Arc3d"):
plot::Arrow2d      := loadproc(plot::Arrow2d,      path, "Arrow2d"):
plot::Arrow3d      := loadproc(plot::Arrow3d,      path, "Arrow3d"):
plot::Bars2d       := loadproc(plot::Bars2d,       path, "Bars2d"):
plot::Bars3d       := loadproc(plot::Bars3d,       path, "Bars3d"):
plot::Box          := loadproc(plot::Box,          path, "Box"):
plot::Boxplot      := loadproc(plot::Boxplot,      path, "Boxplot"):
plot::Circle2d     := loadproc(plot::Circle2d,     path, "Circle2d"):
plot::Circle3d     := loadproc(plot::Circle3d,     path, "Circle3d"):
plot::Cone         := loadproc(plot::Cone,         path, "Cone"):
plot::Conformal    := loadproc(plot::Conformal,    path, "Conformal"):
plot::Curve2d      := loadproc(plot::Curve2d,      path, "Curve2d"):
plot::Curve3d      := loadproc(plot::Curve3d,      path, "Curve3d"):
plot::Cylinder     := loadproc(plot::Cylinder,     path, "Cylinder"):
plot::Cylindrical  := loadproc(plot::Cylindrical,  path, "Cylindrical"):
plot::Density      := loadproc(plot::Density,      path, "Density"):
plot::Dodecahedron := loadproc(plot::Dodecahedron, path, "Dodecahedron"):
plot::Ellipse2d    := loadproc(plot::Ellipse2d,    path, "Ellipse2d"):
plot::Ellipse3d    := loadproc(plot::Ellipse3d,    path, "Ellipse3d"):
plot::Ellipsoid    := loadproc(plot::Ellipsoid,    path, "Ellipsoid"):
plot::Function2d   := loadproc(plot::Function2d,   path, "Function2d"):
plot::Function3d   := loadproc(plot::Function3d,   path, "Function3d"):
plot::Hatch        := loadproc(plot::Hatch,        path, "Hatch"):
plot::Hexahedron   := loadproc(plot::Hexahedron,   path, "Hexahedron"):
plot::Histogram2d  := loadproc(plot::Histogram2d,  path, "Histogram2d"):
// internal and undocumented:
plot::HonestFunction2d
                   := loadproc(plot::HonestFunction2d, path, "HonestFunction2d"):
plot::Implicit2d   := loadproc(plot::Implicit2d,   path, "Implicit2d"):
plot::Implicit3d   := loadproc(plot::Implicit3d,   path, "Implicit3d"):
plot::Inequality   := loadproc(plot::Inequality,   path, "Inequality"):
plot::Icosahedron  := loadproc(plot::Icosahedron,  path, "Icosahedron"):
plot::Integral     := loadproc(plot::Integral,     path, "Integral"):
plot::Iteration    := loadproc(plot::Iteration,    path, "Iteration"):
plot::Line2d       := loadproc(plot::Line2d,       path, "Line2d"):
plot::Line3d       := loadproc(plot::Line3d,       path, "Line3d"):
plot::Listplot     := loadproc(plot::Listplot,     path, "Listplot"):
plot::Lsys         := loadproc(plot::Lsys,         path, "Lsys"):
plot::Matrixplot   := loadproc(plot::Matrixplot,   path, "Matrixplot"):
plot::MuPADCube    := loadproc(plot::MuPADCube,    path, "MuPADCube"):
plot::Octahedron   := loadproc(plot::Octahedron,   path, "Octahedron"):
plot::Ode2d        := loadproc(plot::Ode2d,        path, "Ode2d"):
plot::Ode3d        := loadproc(plot::Ode3d,        path, "Ode3d"):
plot::Parallelogram2d :=
                  loadproc(plot::Parallelogram2d,  path, "Parallelogram2d"):
plot::Parallelogram3d:=
                  loadproc(plot::Parallelogram3d,  path, "Parallelogram3d"):
plot::Piechart2d   := loadproc(plot::Piechart2d,   path, "Piechart2d"):
plot::Piechart3d   := loadproc(plot::Piechart3d,   path, "Piechart3d"):
plot::Plane        := loadproc(plot::Plane,        path, "Plane"):
plot::Point2d      := loadproc(plot::Point2d,      path, "Point2d"):
plot::Point3d      := loadproc(plot::Point3d,      path, "Point3d"):
plot::PointList2d  := loadproc(plot::PointList2d,  path, "PointList2d"):
plot::PointList3d  := loadproc(plot::PointList3d,  path, "PointList3d"):
plot::Polar        := loadproc(plot::Polar,        path, "Polar"):
plot::Polygon2d    := loadproc(plot::Polygon2d,    path, "Polygon2d"):
plot::Polygon3d    := loadproc(plot::Polygon3d,    path, "Polygon3d"):
plot::Prism        := loadproc(plot::Prism,        path, "Prism"):
plot::Pyramid      := loadproc(plot::Pyramid,      path, "Pyramid"):
plot::QQplot       := loadproc(plot::QQplot,       path, "QQplot"):
plot::Raster       := loadproc(plot::Raster,       path, "Raster"):
plot::Rectangle    := loadproc(plot::Rectangle,    path, "Rectangle"):
plot::Rootlocus    := loadproc(plot::Rootlocus,    path, "Rootlocus"):
plot::Scatterplot  := loadproc(plot::Scatterplot,  path, "Scatterplot"):
plot::Sequence     := loadproc(plot::Sequence,     path, "Sequence"):
plot::SparseMatrixplot := loadproc(plot::SparseMatrixplot,
                                                   path, "SparseMatrixplot"):
plot::Sphere       := loadproc(plot::Sphere,       path, "Sphere"):
plot::Spherical    := loadproc(plot::Spherical,    path, "Spherical"):
plot::Streamlines2d := loadproc(plot::Streamlines2d, path, "Streamlines2d"):
plot::Surface      := loadproc(plot::Surface,      path, "Surface"):
plot::Sum          := loadproc(plot::Sum,          path, "Sum"):
plot::SurfaceSet   := loadproc(plot::SurfaceSet,   path, "SurfaceSet"):
plot::SurfaceSTL   := loadproc(plot::SurfaceSTL,   path, "SurfaceSTL"):
plot::Sweep        := loadproc(plot::Sweep,        path, "Sweep"):
plot::Tetrahedron  := loadproc(plot::Tetrahedron,  path, "Tetrahedron"):
plot::Text2d       := loadproc(plot::Text2d ,      path, "Text2d"):
plot::Text3d       := loadproc(plot::Text3d ,      path, "Text3d"):
plot::Tube         := loadproc(plot::Tube,         path, "Tube"):
plot::Turtle       := loadproc(plot::Turtle,       path, "Turtle"):
plot::VectorField2d:= loadproc(plot::VectorField2d,path, "VectorField2d"):
plot::VectorField3d:= loadproc(plot::VectorField3d,path, "VectorField3d"):
plot::Waterman     := loadproc(plot::Waterman,     path, "Waterman"):
plot::XRotate      := loadproc(plot::XRotate,      path, "XRotate"):
plot::ZRotate      := loadproc(plot::ZRotate,      path, "ZRotate"):

unalias( path ):


/*
 *  manipulation functions
 */
alias (path = pathname( "PLOT" )):

plot::modify   := loadproc(plot::modify,   path, "copy"):
plot::copy     := loadproc(plot::copy,     path, "copy"):
plot::easy     := loadproc(plot::easy,     path, "easy"):

// type checking and attribute handling
plot::objectHierarchy:= loadproc(plot::objectHierarchy, path, "model"):
plot::allObjectTypes := loadproc(plot::allObjectTypes,  path, "model"):
plot::animatedObjects:= loadproc(plot::animatedObjects, path, "model"):
plot::allLights      := loadproc(plot::allLights,       path, "model"):
plot::allGraphPrim   := loadproc(plot::allGraphPrim,    path, "model"):
plot::allGraphPrim2d := loadproc(plot::allGraphPrim2d,  path, "model"):
plot::allGraphPrim3d := loadproc(plot::allGraphPrim3d,  path, "model"):
plot::all2dObjects   := loadproc(plot::all2dObjects,    path, "model"):
plot::all3dObjects   := loadproc(plot::all3dObjects,    path, "model"):
plot::allTransformations2d :=
                loadproc(plot::allTransformations2d,    path, "model"):
plot::allTransformations3d :=
                loadproc(plot::allTransformations3d,    path, "model"):
plot::allTransformations :=
                  loadproc(plot::allTransformations,    path, "model"):
plot::addToObjectHierarchy:=
                   loadproc(plot::addToObjectHierarchy, path, "model"):

plot::createPlotDomain :=
                 loadproc(plot::createPlotDomain, path, "createPlotDomain"):

plot::attributes     := loadproc(plot::attributes,      path, "attributes"):
plot::extraValues    := loadproc(plot::extraValues,     path, "attributes"):
plot::hintAllowed    := loadproc(plot::hintAllowed,     path, "attributes"):
plot::standardHints  := loadproc(plot::standardHints,   path, "attributes"):
plot::checkColorList := loadproc(plot::checkColorList,  path, "attributes"):
plot::elementOptFunctionExpr:=
                 loadproc(plot::elementOptFunctionExpr, path, "attributes"):
plot::elementOptTextExpr:=
                     loadproc(plot::elementOptTextExpr, path, "attributes"):
plot::varNames       := loadproc(plot::varNames,        path, "attributes"):
plot::functions      := loadproc(plot::functions,       path, "attributes"):
plot::colorFunctions := loadproc(plot::colorFunctions,  path, "attributes"):
plot::allInheritedAttributes:=
                 loadproc(plot::allInheritedAttributes, path, "attributes"):
plot::allOptionalAttributes:=
                  loadproc(plot::allOptionalAttributes, path, "attributes"):
plot::allLibraryAttributes:=
                   loadproc(plot::allLibraryAttributes, path, "attributes"):
plot::allMandatoryAttributes:=
                 loadproc(plot::allMandatoryAttributes, path, "attributes"):
plot::outputSizeAttributes:=
                   loadproc(plot::outputSizeAttributes, path, "attributes"):
plot::libRangeOfXYZExpr:=
                      loadproc(plot::libRangeOfXYZExpr, path, "attributes"):
plot::readRangeOfXYZExpr:=
                     loadproc(plot::readRangeOfXYZExpr, path, "attributes"):
plot::libRangeOfOptExpr:=
                      loadproc(plot::libRangeOfOptExpr, path, "attributes"):
plot::readRangeOfOptExpr:=
                     loadproc(plot::readRangeOfOptExpr, path, "attributes"):
plot::libListOfOptExprXYZ :=
                loadproc(plot::libListOfOptExprXYZ,     path, "attributes"):
plot::readListOfOptExprXYZ:=
               loadproc(plot::readListOfOptExprXYZ,     path, "attributes"):
plot::libListOfExpr :=
                      loadproc(plot::libListOfExpr,     path, "attributes"):
plot::readListOfExpr:=
                     loadproc(plot::readListOfExpr,     path, "attributes"):
plot::libOneToMore :=
                      loadproc(plot::libOneToMore,      path, "attributes"):
plot::readOneToMore:=
                     loadproc(plot::readOneToMore,      path, "attributes"):
plot::libAlias :=       loadproc(plot::libAlias,        path, "attributes"):
plot::readAlias:=       loadproc(plot::readAlias,       path, "attributes"):
plot::checkAttributesTable:=
                   loadproc(plot::checkAttributesTable, path, "attributes"):
plot::legalAttributes:= loadproc(plot::legalAttributes, path, "attributes"):
plot::computeLegalAttributes:=
                 loadproc(plot::computeLegalAttributes, path, "attributes"):

plot::checkOptions   := loadproc(plot::checkOptions,    path, "checkOptions"):
plot::getInherited   := loadproc(plot::getInherited,    path, "checkOptions"):
plot::setDefault     := loadproc(plot::setDefault,      path, "checkOptions"):
plot::setDefaultInternal:=
                     loadproc(plot::setDefaultInternal, path, "checkOptions"):
plot::getDefault     := loadproc(plot::getDefault,      path, "checkOptions"):
plot::getAttributeNames :=
                      loadproc(plot::getAttributeNames, path, "checkOptions"):
plot::getOwnAttributeNames :=
                   loadproc(plot::getOwnAttributeNames, path, "checkOptions"):
plot::getOptionAndValuesNames :=
               loadproc(plot::getOptionAndValuesNames,  path, "checkOptions"):
plot::getOptionNames := loadproc(plot::getOptionNames,  path, "checkOptions"):
plot::getOptionals   := loadproc(plot::getOptionals,    path, "checkOptions"):

plot::ExprAttribute  := loadproc(plot::ExprAttribute,   path, "exprAttribute"):
plot::inspector      := loadproc(plot::inspector,       path, "inspector"):
plot::buildInspectorEntry :=
                    loadproc(plot::buildInspectorEntry, path, "inspector"):
plot::StyleSheetEntry:= loadproc(plot::StyleSheetEntry, path, "styles"):

unalias( path ):

/*
 * backends
 */
 
alias(path = pathname("PLOT", "OUTPUT")):
plot::GenericOutput := loadproc(plot::GenericOutput, path, "GenericOutput"):
plot::MuPlotML := loadproc(plot::MuPlotML, path, "MuPlotML"):
plot::STL := loadproc(plot::STL, path, "STL"):

unalias(path):

/*
 *  collection of internal tools
 */
alias (path = pathname("PLOT", "TOOLS")):

plot::functionEval    := loadproc(plot::functionEval,   path, "functionEval"):
plot::curveEval       := loadproc(plot::curveEval,      path, "curveEval"):
plot::surfaceEval     := loadproc(plot::surfaceEval,    path, "surfaceEval"):
plot::adaptiveSurfaceEval
           := loadproc(plot::adaptiveSurfaceEval, path, "adaptiveSurfaceEval"):
plot::listAttribute2table
           := loadproc(plot::listAttribute2table, path, "listAttribute2table"):
plot::writeMeshContours := loadproc(plot::writeMeshContours, path, "writeMeshContours"):
plot::writeGridContours := loadproc(plot::writeGridContours, path, "writeGridContours"):

plot::hull := loadproc(plot::hull, path, "hull"):
plot::delaunay := loadproc(plot::delaunay, path, "hull"):

plot::unapply := loadproc(plot::unapply, path, "unapply"):

unalias( path ):

plot::Name := hold(plot):

plot::evaluate:= FAIL:
plot::posteval:= FAIL:
plot::allEntries := FAIL:
plot::allAutoEntries := FAIL:
plot::whichEntry := FAIL:
plot::new_extelement := FAIL:
plot::create_dom_elem := FAIL:
plot::Content:=FAIL:
plot::eval := FAIL:
plot::domtype:= FAIL:
plot::type:= FAIL:
plot::isNeg:= FAIL:
plot::isInverted:= FAIL:
plot::slot:= FAIL:
plot::expr2text:= FAIL:
plot::indets:= FAIL:
plot::freeIndets:= FAIL:
plot::bool :=FAIL:
plot::length := FAIL:
plot::rectform:= FAIL:
plot::contains:= FAIL:
plot::expose := FAIL:
plot::hastype := FAIL:
plot::maprec := FAIL:

plot::op := FAIL:
plot::nops := FAIL:
plot::subsop := FAIL:
plot::undefinedEntries := FAIL:
plot::evaluateIndex := FAIL:
plot::sortSums := FAIL:
plot::CF := FAIL: // see GENERATE/CF.mu

/*--
make_slot -- create default method

make_slot always returns an error.

NOTE: This MUST be the last method defined for plot.
--*/

plot::make_slot:= proc(DOM, Slot)
    begin
      if domtype(Slot) = DOM_STRING then
         context(hold(error)("plot::".Slot." does not exist, see ?plot")):
      else
         context(hold(error)("plot::".expr2text(Slot)." does not exist, see ?plot")):
      end_if;
end_proc:

prog::setcheckglobals(plot, {javaview}):
prog::setcheckglobals(display, {javaview}):

null():