// linalg::SSSGaussJordan -- solve a system of linear equations 
//     with coefficient matrix A und right-hand-side b by 
//     (standard) Gauss-Jordan elimination of the matrix

linalg::SSSGaussJordan:= proc(A, b) 
        local m, n, Mat, T, R, Rone, Rnormal, charind, 
              rank, kernelCols, detA, i, ind, j, k, 
              kernelColVectors, l, nb, B;
      begin 
        if args(0) < 2 then 
          error("expecting two arguments") 
        end_if;
        Mat:= A::dom;
        if Mat::hasProp(Cat::Matrix) <> TRUE then
          error("expecting a matrix of 'Cat::Matrix'");
        end_if;
        if Mat <> b::dom then 
          error("the 2nd argument must be of the same type as the 1st argument");
        end_if;
        if A::dom::matdim(A)[1] <> b::dom::matdim(b)[1] then 
          error("illegal operands");
        end_if;
        R:= A::dom::coeffRing;
        Rone:= R::one;
        Rnormal:= if R::normal <> FAIL then
                     normal; // normal should be overloaded by R::normal
                  else
                     id;
                  end_if;
        if Mat::constructor <> Dom::Matrix then 
          //==================================================
          // Convert the matrix into a matrix over Dom::Matrix
          //==================================================
          A:= Dom::Matrix(R)(A);
          b:= Dom::Matrix(R)(b);
        end_if;
        [m, n]:= A::dom::matdim(A):
        nb:= b::dom::matdim(b)[2];
        [T, rank, detA, charind]:= linalg::gaussJordan(A::dom::concatMatrix(A,b), All);
        //==============================================
        // Is there any solution at all? 
        // Check whether rank(A,B) = rank(A). 
        //==============================================
        if rank > 0 then
          for j from rank to n do
            if linalg::zeroTest( T[rank,j] ) = FALSE then 
              break;
            end_if
          end_for;
          if j > n then 
            return([]);
          end_if
        end_if;
        //==============================================
        // Positioning of the characteristic columns 
        // "along the main diagonal". Insert 1 on the 
        // main diagonal in zero rows. 
        //==============================================
        if m < n then 
          T:= T::dom::stackMatrix(T, Dom::Matrix(R)(n-m, n + nb));
        end_if;
        kernelCols:= {};
        detA:= 0;
        for i from 1 to n do
          if linalg::zeroTest(T[i,i]) <> FALSE then 
            for l from i+1 to n do 
              if linalg::zeroTest(T[i,l]) = FALSE then 
                T:= T::dom::swapRow(T, i, l);
              end_if;
            end_for;
            T[i,i]:= Rone;
            kernelCols:= kernelCols union {i};
          else
            if detA = 0 then
               detA:= T[i,i]:
            else
               assert(linalg::zeroTest(T[i,i] - detA) <> FALSE);
            end_if;
          end_if;
        end_for;
        //==============================================
        // Remove zero rows at the end of the matrix
        //==============================================
        for i from min(m,n) + 1 to m do  
          T:= T::dom::delRow(T, i);
        end_for;

        //================================================
        // Compute a special solution of the system.
        //================================================
        [T, B]:= [T[1..n, 1..n], T[1..n, n+1..n+nb]];

        for i from 1 to n do
          for j from 1 to nb do 
            if T[i, i] <> Rone then 
               B[i, j]:= Rnormal(B[i, j]/T[i,i]);
            end_if:
          end_for:
        end_for: // compute a solution for rhs = i-th column ob b

        //===============================================
        // Compute the kernel of the matrix A: kernelCols 
        // contains the indices of the  columns, where we 
        // inserted 'Rone' on the main diagonal. 
        //================================================
        kernelColVectors:= [];
        for ind in sort([op(kernelCols)]) do 
          k:= Dom::Matrix(R)(n,1, [-T[i, ind]/T[i,i] $ i = 1..n]);
          k[ind, 1]:= -k[ind, 1];
          kernelColVectors:= append(kernelColVectors, k);
        end_for;

        return([B, kernelColVectors, kernelCols]);
end_proc:
