/*
 linalg::charmat  --  return the characteristic matrix

 linalg::charmat(A,x)

 A: square matrix over a commutative ring
 x: valid polynomial indeterminate
 
 This function constructs the characteristic matrix, defined
 by x*E(n) - A, where E(n) is the identity matrix of 
 dimension n=matdim(A).
 The characteristic matrix will be defined over the domain 
 Dom::DistributedPolynomial([x],R), where R is the coefficient 
 domain of A. R must be a commutative ring.

 Note: Evaluations of the characteristic matrix C for a matrix
       A of matrix domain M for special values u of x can be 
       done by

     evalp( C,x=u )
*/

linalg::charmat:= proc(A,x)
    local n, S, Mat;
begin
    if args(0) <> 2 then error("expecting 2 arguments") end_if;
    Mat:= A::dom;

    if testargs() then
      if Mat::hasProp( Cat::Matrix ) <> TRUE then
        error("first argument is not of 'Cat::Matrix'")
      end_if;
      if not Mat::coeffRing::hasProp( Cat::CommutativeRing ) then
        error("expecting matrix over 'Cat::CommutativeRing'")
      end_if; 
      n:= Mat::matdim(A);
      if n[1] <> n[2] then
        error("not a square matrix")
      end_if:
      if poly(x) = FAIL then
        error("illegal indeterminate")
      end_if
    end_if;

    S:= Dom::DistributedPolynomial( [x],Mat::coeffRing );

    if Mat = Dom::Matrix(Mat::coeffRing) then 
      Mat:= Dom::Matrix(S);
    else 
      Mat:= Dom::DenseMatrix(S);
    end_if;
    
    A:= Mat::coerce( A );

    if A = FAIL then
       if S::Name = FAIL then S::key else S::Name end_if;
       error("unable to convert matrix to a matrix over ".expr2text(%))
    else
       n:= Mat::matdim(A)[1];
       return(Mat::_plus( 
           Mat::create( n,n,[S::new(x) $ n],hold(Diagonal) ), Mat::_negate(A) 
       ))
    end_if
end_proc:

