/*++
    linalg::charpoly -- returns the characteristic polynomial

    linalg::charpoly(A,x)

    A: square matrix over a commutative ring
    x: indeterminate
     
    Computes the characteristic polynomial of the matrix A
    in x, i.e. det( x*E(n)-A ), where E(n) is the 
    identity matrix with n=matdim(A).

    The coefficient domain of A has to be a commutative ring.

    Hessenberg's algorithm is used for matrices over Dom::Float and
    Dom::IntegerMod. 

    The algorithm works in any field, and requires only O(n^3) coefficient 
    operations instead of O(n^4) for other methods like det(x*Id-A).
    But the entries of the matrix could swell extremely, so that
    the usage of Hessenberg's algorithm is restricted to the stated 
    coefficient domains.

    Reference: A Course in Computational Algebraic Number Theory,
               by Henri Cohen, GTM 138, Springer Verlag, 
               algorithm 2.2.9 page 55 
++*/

linalg::charpoly:= proc(A,x)
    local R, n;
begin
    if testargs() then
        if args(0) <> 2 then error("expecting 2 arguments") end_if;
        if A::dom::hasProp( Cat::Matrix ) <> TRUE then
            error("first argument is not of 'Cat::Matrix'")
        end_if
    end_if;

    R:= A::dom::coeffRing;
    n:= A::dom::matdim(A);

    if testargs() then
        if not R::hasProp(Cat::CommutativeRing) then
            error("expecting a square matrix over 'Cat::CommutativeRing'")
        elif n[1] <> n[2] then
            error("not a square matrix")
        elif poly(x) = FAIL then
            error("illegal indeterminate")
        end_if
    end_if;

    case n[1]
    of 1 do
        userinfo(2,"use direct formula for 1x1 matrices");
        return(Dom::DistributedPolynomial( [x],R )( 
            poly([[_negate(A[1,1]),0],[R::one,1]],[x],R)
        ))
    of 2 do
        userinfo(2,"use direct formula for 2x2 matrices");
        return(Dom::DistributedPolynomial( [x],R )(
            poly([[R::one,2],[-A[2,2] - A[1,1],1],[A[1,1]*A[2,2] - A[1,2]*A[2,1],0]], [x], R)
        ))
    otherwise
        if R = Dom::Float or
           R::constructor = Dom::IntegerMod and R::hasProp(Cat::Field)
        then
            userinfo(2,"use Hessenberg's algorithm");
            return(
                Dom::DistributedPolynomial([x],R)( linalg::charpolyHessenberg( A,x,R,n[1] ) )
            )
        else
            userinfo(2,"use Berkowitz's algorithm");
            return(
                Dom::DistributedPolynomial([x],R)( linalg::charpolyBerkowitz( A,x,R,n[1] ) )
            )
        end_if
    end_case
end_proc:

linalg::charpolyHessenberg:= proc(H,x,R,n)
    local m,i,t,j,u,p,Rzero, Riszero, Rmult, Rdivide, Rplus;
begin
    // step 1: initialize:
    Rzero:= R::zero;
    Riszero:= R::iszero;
    Rmult:= R::_mult;
    Rdivide:= R::_divide;
    Rplus:= R::_plus;

    // step 2: search for non-zero:
    for m from 2 to n-1 do
        i:= m+1; while i<=n and Riszero((t:= H[i,m-1])) do i:= i + 1 end_while;

        if i <= n then
            if i > m then
                H:= assignElements(H,([[i,j],H[m,j]], [[m,j],H[i,j]]) $ j = m-1..n);
                H:= assignElements(H,([[j, i], H[j, m]], [[j, m], H[j, i]]) $ j = 1..n);
            end_if;

            // step 3: eliminate:
            for i from m+1 to n do
                if not Riszero( (u:= Rdivide(H[i,m-1],t)) ) then
                    H:=assignElements(H,[[i,j],Rplus(H[i,j],-Rmult(u,H[m,j]))]
                          $ j=m..n, [[i,m-1],Rzero]);
                    H:=assignElements(H,[[j,m],Rplus(H[j,m],Rmult(u,H[j,i]))]
                          $ j=1..n);
                end_if
            end_for
        end_if;
   end_for;

   // step 5:
   p[0]:= poly(1,[x],R);
   // step 6:
   for m from 1 to n do
       p[m]:= poly(hold(_plus)(x,-H[m,m]),[x],R) * p[m-1]; 
       t:= R::one;
       for i from 1 to m-1 do
           t:= Rmult( t, H[m-i+1,m-i] );
           p[m]:= p[m] - multcoeffs(p[m-i-1],Rmult(t,H[m-i,m]))
       end_for
   end_for;
   p[n]
end_proc:

/*--
K. F. Gehrs, 2000/09/16

    linalg::charpolyBerkowitz -- returns the characteristic polynomial

    linalg::charpolyBerkowitz(A,x,R,n)

    A: square matrix
    x: indeterminate
    R: coefficient domain of A (a commutative ring)
    n: number of rows of A
 
    Computes the characteristic polynomial of the matrix A
    in x, i.e. det( x*E(n)-A ), where E(n) is the 
    identity matrix with n = matdim(A).

    The coefficient domain of A has to be a commutative ring.

    Reference: The Berkowitz Algorithm, Maple and Computing the 
               Characteristic Polynomial in an Arbitrary Commutative Ring
               by Jounaidi Abdeljaoued
               May 1996
--*/
linalg::charpolyBerkowitz:=
proc(A,x,R,n)
  local Rzero, Rmult, Rplus, Rnegate, Rone, Vect, r, C, S, Q, polcar, i, j, k;
begin
    Rone:= R::one;
    Rzero:= R::zero;
    Rmult:= R::_mult;
    Rplus:= R::_plus;
    Rnegate:= R::_negate;

    Vect:= (S:= (Q:= array( 1..n+1 )));
    Vect[1]:= Rnegate(Rone); 
    Vect[2]:= A[1,1];
    C[1]:= Vect[1];

    for r from 2 to n do
        S:= assignElements(S, [i] = A[i,r] $ i=1..r-1);
        C[2]:= A[r,r];
        for i from 1 to r-2 do
            C[i+2]:= Rzero;
            ( C[i+2]:= Rplus( C[i+2], Rmult( A[r,j], S[j])) ) $ j=1..r-1;
            for j from 1 to r-1 do
                Q[j]:= Rzero;
                ( Q[j]:= Rplus( Q[j], Rmult( A[j,k], S[k])) ) $ k=1..r-1
            end_for;
            S:= assignElements(S, [j] = Q[j] $ j=1..r-1);
        end_for;
        C[r+1]:= Rzero;
        ( C[r+1]:= Rplus( C[r+1], Rmult( A[r,j], S[j])) ) $ j=1..r-1;
        for i from 1 to r+1 do
            Q[i]:= Rzero;
            ( Q[i]:= Rplus( Q[i], Rmult( C[i+1-j], Vect[j])) ) $ j=1..min(r,i)
        end_for;
        Vect:= assignElements(Vect, [i] = Q[i] $ i=1..r+1);
    end_for;

    polcar:= poly( [[Vect[i],n-i+1] $ i=1..n+1],[x],R );
    if lcoeff(polcar) = Rone then 
        return( polcar )
    else 
        return( multcoeffs(polcar,Rnegate(Rone)) ) 
    end_if
end_proc:

