/*
        linalg::eigenvectors(A)  --  computes the eigenvectors of matrix A

        A: square matrix

        eigenvectors(A) computes the eigenvalues and eigenvectors 
        of the matrix A.

        Method:
        1. eigenvalues of the matrix are computed using the function
           linalg::eigenvalues. 

        2. For each eigenvalue e the matrix equation (I*e - A)*X = 0, 
           where I denotes the identity matrix, is solved using the 
           Gauss-Jordan algorithm (linalg::gaussJordan).

        Result:
        A list of sublists, where each sublist consists of an eigenvalue
        l of A, its algebraic multiplicity and a basis for the eigenspace of l.

        Comments:
        An error message is given if A is not of category Cat::Field.

        If the coefficient ring of A is Dom::Float then numeric::eigenvalues
        is used. Note that in this case the list e of eigenvalues is sorted 
        which means that e[i]<=e[j] for i<j. The eigenvectors are normalized 
        (see numeric::eigenvectors).
*/

linalg::eigenvectors := proc(A)
    local n, s, e, i, k, r, R, Aplus, done, ev, evList, En, Mat, Afloated, j, v, Res;
begin
    if testargs() then
      if args(0) <> 1 and args(0) <> 2 and args(2) = "CalledFromODELinsys" then
        error("expecting a matrix of 'Cat::Matrix'")	
      elif A::dom::hasProp( Cat::Matrix ) <> TRUE then
        error("expecting a matrix of 'Cat::Matrix'")
      elif not (A::dom::coeffRing)::hasProp( Cat::Field ) then
        error("expecting a matrix over a 'Cat::Field'")
      end_if
    end_if;

    Mat:= A::dom;
    R:= Mat::coeffRing;
    n:= Mat::matdim(A);
    if n[1] <> n[2] then 
      error("not a square matrix")
    end_if;
    n:= n[1];

    if linalg::checkForFloats(A) = FALSE then 
      Res:= linalg::symbolicEigenvectors(A);
      if Res <> FAIL then 
        return(Res);
      end_if;
    end_if;
    
    if args(0) <> 2 and R::constructor = Dom::ExpressionField then
      if map({op(A)}, domtype) minus {DOM_RAT, DOM_INT} = {} then
        return(linalg::eigenvectorsRational(A, 2));
      end_if;
    end_if;    
    
/*
    OLD VERSION FOR THE FLOAT CASE: BUGGY!!! ...but for safety still here
    
    if R = Dom::Float then
        userinfo(1,"call 'numeric::eigenvectors'");
        s:= numeric::eigenvectors(A, hold(NoResidues) );

        // collect multiple eigenvalues:
        ev:= []; k:= 1;
        Mat:= A::dom::constructor(R);
        s[2]:= Mat::coerce( s[2] );
        if s[2] = FAIL then 
          return(FAIL);
        end_if;
        while s[1] <> [] do
            n:= split( s[1],_equal,s[1][1] );
            r:= nops(n[1]);

            if (e:= R::coerce(n[1][1])) <> FAIL then
                ev:= append( ev,[ e,r,[Mat::col(s[2],i) $ i=k..k+r-1] ] )
            end_if;
            k:= k + r;
            s[1]:= n[2]
        end_while;
        return( ev )
    // New branch in the if-construction: 
    // check whether the matrix contains floats and can be converted 
    // to 'float(Matrix)'. If this is the case, call the corresponding 
    // 'numeric'-routine as in the case above. 
*/

    if (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
        n:= Mat::matdim(A);
        if n[1] <> n[2] then 
            error("not a square matrix")
        end_if;
        n:= n[1];
        userinfo(1,"call 'numeric::eigenvectors'");
        s:= numeric::eigenvectors(Afloated, hold(NoErrors));
        Mat:= A::dom::constructor(R);

        //---------------------------------------------
        // Walter, 17.1.07: changed the search strategy
        // for (nearly) equal eigenvalues
        //---------------------------------------------
        ev:= []: // list of all spectral data
        done:= {}; // set of eigenval indices which are
                   // alread listed in ev
        s[1]:= map(s[1], numeric::complexRound);
        s[2]:= map(s[2], numeric::complexRound);
        for k from 1 to n do
           if contains(done, k) then 
              next; 
           end_if;
           e:= R::coerce(s[1][k]);
           if e = FAIL then 
              done:= done union {k};
              next;
           end_if;
           evList:= []; // list of all eigenvectors for this eigenvalue
           for j from 1 to n do
              if contains(done, j) then
                 next;
              end_if;
              if abs(s[1][j] - s[1][k])<= abs(s[1][k])*10^(-DIGITS) then
                 done:= done union {j};
                 v:= [s[2][i,j] $ i = 1..n];
                 if Mat::coerce(v) <> FAIL then
                   evList:= append(evList, Mat(v));
                 end_if;
              end_if;
           end_for:
           ev:= append(ev, [e, nops(evList), evList]);
        end_for:
                
        /* ================================================
        // this is the old code for collecting the
        // same (similar) eigenvalues
        ev:= []; k:= 1;
        while s[1] <> [] do
            m:= split(s[1],_equal, s[1][1] ); 
            r:= nops(m[1]);
            evList:= [];
            if (e:= R::coerce(m[1][1])) <> FAIL then
              for j from k to k+r-1 do
                v:= [s[2][i,j] $ i = 1..n];
                v:= map(v, numeric::complexRound);
                if Mat::coerce(v) <> FAIL then
                  evList:= append(evList, Mat(v));
                end_if;
              end_for;
              ev:= append(ev,[e,r,evList])
            end_if;
            k:= k + r;
            s[1]:= m[2]
        end_while;
        ================================================ */
        return( ev )
    else
        userinfo(1,"compute the eigenvalues of the given matrix");
        s:= linalg::eigenvalues(A,hold(Multiple));	

        if hastype(s,RootOf) then 
          error("cannot compute the explicit representation of the eigenvalues; use 'numeric::eigenvectors'");
        end_if;
        
        if type(s) = piecewise then 
          s:= piecewise::disregardPoints(s);
          if type(s) = piecewise then 
            warning("cannot find a simple representation of eigenvalues");
            warning("some cases are ignored");
            s:= s[1]; // sloppy workaround: if nothing helps to get rid of 
                      // piecewise, just choose the first branch
          end_if;
        end_if;
        
        A:= Mat::_negate(A);
        Aplus:= Mat::_plus;
        En:= Mat::identity( n );

        ev:= [];
        for k from 1 to nops(s) do
            e:= En;
            (e[i,i]:= op(s,[k,1])) $ i=1..n;
            userinfo(2,"compute kernel of the characteristic manifold");
            userinfo(3,"for the eigenvalue ",op(s,[k,1]));
            if linalg::floatable(A) = TRUE then 
              // print("In 'linalg::eigenvectors'");
              r:= linalg::nullspace(Aplus(e,A), `#FloatCheck`); 
              // The Matrix A is floatable. Hence, call
              // 'linalg::nullspace' with second argument `#FloatCheck`. Then we 
              // proceed as follows: 'linalg::nullspace' calls 'linalg::gaussJordan'
              // with additional argument `#FloatCheck`. Then 'linalg::gaussJordan' 
              // calls 'A::dom::gaussElim' with additional argument `#FloatCheck`. 
              // Then, 'A::dom::gaussElim' uses a special heuristic for zeroTest
              // and we will be better in finding 'eigenvectors' than in the general 
              // case where A perhaps contains symbolic entries. 
            else 
              r:= linalg::nullspace( Aplus(e,A) ); // Former way of proceeding: do 
                                                   // what we always did before, i.e., 
                                                   // step into symbolic computation
            end_if;
            if r = [] then
                warning("basis of eigenspace for eigenvalue ".expr2text(op(s,[k,1])).
                           " not found (zero test failed?)");
                return( FAIL )
            else
                ev:= append(ev,[op(s,[k,1]),op(s,[k,2]),r])
            end_if
        end_for;

        return( ev )
    end_if
end_proc:

