/* --------------------------------------------------------------------------
Compute the set of eigenvectors of a matrix with rational entries
   
Calls:

            linalg::symbolicEigenvectors(A, <"JordanFrom">, <"All">)

Parameters:            A -- a matrix 
            
            "JordanForm" -- an internal flag indicating that the 
                            Jordan 'J' form schould be computed (in 
                            case that 'A' can be diagonalized). 
         
                   "All" -- an internal flag indicating that the 
                            Jordan form 'J' together with the transition 
                            matrix 'P' should be returned (in case that 
                            'A' can be diagonalized) such that 
                            'A = P J P^(-1)'. 

Return value: - If no internal flag is given, the output is generated
                in the same form as 'linalg::eigenvectors' does. 
     
              - If "JordanForm" is used, a diagional matrix made up of 
                the eigenvalues of 'A' is returned. 
     
              - If "JordanForm" and "All" is used, a list of two matrices 
                is returned, i.e. '[J,P]' where 'J' is the Jordan form of 
                'A' (diagional matrix made up of the eigenvalues of 'A' is 
                returned, and 'P' is the transition matrix build up of the 
                eigenvectors of 'A' such that 'A = P J P^(-1)'.

              - FAIL if anything goes wrong 

---------------------------------------------------------------------------  */

linalg::symbolicEigenvectors:= proc(A)
  local AlgExtMatrix, ind, p, x, pFactors, i, j, q, FracField, M, n, 
        eigenvalues, eigenvectors, parametricEigenvectors, ev, Mat, 
        indetList, evList, numericIndets, t, sqrMatrix, B;
begin 
  // Do only the necessary argument checking. Most of these things are 
  // done by 'linalg::eigenvectors' calling this functions. 
  if A::dom::matdim(A)[1] <> A::dom::matdim(A)[2] then 
    return(FAIL);
  end_if;  
  n:= A::dom::matdim(A)[1];
  x:= genident();
  // The strategy is as follows: 
  //
  // (*) If 'A' does not contain any identifiers we try to convert it to a 
  //     matrix over 'Dom::Rational'. 
  // (*) If 'A' contains (indexed) identifiers 'ind', we try to convert 'A' 
  //     to a matrix over the fraction field of the polynomial ring in the 
  //     variables 'ind' over the rational numbers. 
  //
  // NOTE. This strategy may be extended in the future considering more 
  //       general algebraic extensions allowing roots or even 
  //       transcendental elements in the coefficients of the matrix 'A'. 
  ind:= freeIndets(A);
  if nops(ind) > 0 then 
    if A::dom::constructor = Dom::Matrix or A::dom::constructor = Dom::DenseMatrix then 
      B:= [A::dom::nonZeroOperands(A)];
    else 
      B:= [op(A)];
    end_if;
    B:= misc::maprec(B,{"_index"}=(elem -> genident()));
    if has(map(B,
               elem -> if rationalize(expr(elem))[2]<>{} then 
                         FAIL 
                       else 
                         elem 
                       end_if),
           FAIL) then 
      return(FAIL)
    end_if; 
  end_if;  
  // Beware of indexed indentifiers! That's why we use 'numeric::indets' 
  // and substitute them to be able to create polynomials in these 
  // identifiers. 
  numericIndets:= numeric::indets(map({A::dom::nonZeroOperands(A)},expr));
  if ind = numericIndets then 
    ind:= [op(ind)];
    indetList:= [];
  else 
    ind:= [op(ind)];
    indetList:= map([op(numericIndets)], elem -> elem = genident());
    ind:= map(indetList, elem -> rhs(elem));
    A:= subs(A,indetList);
  end_if;  
  // Make sure that there are no floats in the matrix. 
  A:= map(A,numeric::rationalize);  
  p:= expr(linalg::charpoly(A,x));
  Mat:= A::dom;
  // Special treatment for 'Dom::SquareMatrix' needed, since the eigenvectors 
  // cannot be created as elements of this domain. 
  sqrMatrix:= FALSE;
  if Mat::constructor = Dom::SquareMatrix then 
    sqrMatrix:= A::dom;
    Mat:= Dom::Matrix(A::dom::coeffRing);
  end_if;
  if Mat::constructor <> Dom::Matrix then 
    A:= A::dom::convert_to(A,Dom::Matrix(Mat::coeffRing));
  end_if;
  if nops(ind) = 0 then  
    // The matrix will be considered over 'Dom::Rational'. 
    FracField:= Dom::Rational;
    if has(map([A::dom::nonZeroOperands(A)],Dom::Rational::convert@expr),FAIL) then 
      return(FAIL);
    end_if;
  else 
    // The matrix will be considered over the fraction field of the polynomial ring 
    // in the variables 'ind' over the rational numbers. 
    FracField:= Dom::Fraction(Dom::DistributedPolynomial(ind,Dom::Rational));
    if traperror(Dom::Matrix(Dom::AlgebraicExtension(FracField,poly(p,[x])))) <> 0 or  
      A::dom::convert_to(A,Dom::Matrix(Dom::AlgebraicExtension(FracField,poly(p,[x])))) = FAIL then 
      return(FAIL);
    end_if;      
  end_if;
  // Factor the characteristic polynomial (over the rationals) and proceed 
  // to compute the eigenvectors modulo the irreducible factors of 'p'. 
  p:= coerce(factor(poly(p,[x])),DOM_LIST);
  pFactors:= {};
  eigenvectors:= {};
  for i from 2 to nops(p) step 2 do  
    pFactors:= pFactors union {[p[i],p[i+1]]};
  end_for; 
  for q in pFactors do 
    // Compute the eigenvalues. The result must be of type 'Dom::Mulitset'.  
    eigenvalues:= solve(q[1]=0,x,MaxDegree=4,IgnoreSpecialCases);
    if type(eigenvalues) <> DOM_SET then 
      return(FAIL)
    end_if;  
    // Prepare the domain for computing modulo the irreducible factor 'q[1]'. 
    AlgExtMatrix:= Dom::Matrix(Dom::AlgebraicExtension(FracField, poly(q[1],[x])));  
    M:= AlgExtMatrix(A)-x*AlgExtMatrix::identity(n);
    // Solve the linear sytem giving the parametrized eigenvectors modulo 'q[1]'. 
    // Convert the parametrized eigenvectors back to the appropriate matrix domain 
    // 'Mat' of the input matrix 'A'. 
    parametricEigenvectors:= map(linalg::nullspace(M),
                                 elem -> elem::dom::convert_to(elem,Mat::constructor()));
    if has(parametricEigenvectors,FAIL) or parametricEigenvectors = [] then 
      return(FAIL); 
    end_if;
    // Insert the eigenvalues given by 'q[1]' into the parametrized eigenvectors. 
    for ev in eigenvalues do 
      if traperror((t:= map(evalAt(parametricEigenvectors,x=ev),
                            elem -> elem::dom::convert_to(elem,Mat)))) = 0 and 
         not has(t,FAIL) then        
        // [eigenvalue,multiplicity,List of basis vectors spanning the eigenspace] 
        evList:= [ev,q[2],t]; 
        if args(0) > 1 and nops(evList[3]) < evList[2] then 
          // This means that the geometric multiplicity of the eigenvalue 'evList[1]'
          // is less than the algebraic multiplicity. The Jordan form of 'A' cannot 
          // be found by considering only eigenvectors. 
          return(FAIL);
        end_if;
        eigenvectors:= eigenvectors union {evList};
      end_if;  
    end_for;
  end_for;
  // Take care to make a backsubstitution in case of indexed identifiers and 
  // convert 'eigenvectors' to a list of lists. 
  indetList:= map(indetList, elem -> rhs(elem) = lhs(elem));
  if nops(indetList) > 0 then 
    eigenvectors:= [op(map(eigenvectors,subs,indetList))];
  else 
    eigenvectors:= [op(eigenvectors)];
  end_if; 
  // Generate the return values according to the given input parameters.  
  if args(0) = 1 then 
    // The format also returned by 'linalg::eigenvectors'. 
    return(eigenvectors);
  elif args(0) = 2 and args(2) = "JordanForm" then  
    // Return the Jordan canonical form. 
    if sqrMatrix = FALSE then 
      return(Mat(n,n,[(eigenvectors[i][1] $ eigenvectors[i][2]) 
                       $ i = 1..nops(eigenvectors)],Diagonal));
    else                   
      return(sqrMatrix([(eigenvectors[i][1] $ eigenvectors[i][2]) 
                       $ i = 1..nops(eigenvectors)],Diagonal));
    end_if;                   
  elif args(0) = 3 and args(2) = "JordanForm" and args(3) = "All" then  
    // Return the Jordan canonical form and the corresponding transition 
    // matrix. 
    if sqrMatrix = FALSE then 
      return([Mat(n,n,[(eigenvectors[i][1] $ eigenvectors[i][2]) 
                         $ i = 1..nops(eigenvectors)],Diagonal),
              Mat::concatMatrix(eigenvectors[i][3][j] $ j = 1..nops(eigenvectors[i][3]) 
                                   $ i = 1..nops(eigenvectors))]);  
    else                                
      return([sqrMatrix([(eigenvectors[i][1] $ eigenvectors[i][2]) 
                       $ i = 1..nops(eigenvectors)],Diagonal),
              sqrMatrix(Mat::concatMatrix(eigenvectors[i][3][j] $ j = 1..nops(eigenvectors[i][3]) 
                                        $ i = 1..nops(eigenvectors)))]);  
    end_if;                                    
  end_if;  
  
  return(FAIL);
end_proc:

