/*
   linalg::expr2Matrix  --  builds up a matrix from a set or list of 
                            equations

   expr2Matrix(eqns [,vars] [,coeffRing] [,Include])

   eqns, vars - sets or lists
   Include    - ident

   Generates the extended coefficient matrix from the linear system of 
   equations eqns in the unkown vars. 

   If vars is not given, 'indets(eqns,PolyExpr)' is used to get the 
   unknown vars of the equations. 

   With option 'Include', the negative of the right side of the equations 
   is included as the last column of the matrix.

   If coeffRing is missing then the default coefficient domain for 
   matrices will be used (i.e. 'ExpressionField()').
   If coeffRing is given then it must be of category CommutativeRing.

   This function can be used for building up the coefficient matrix of
   expressions instead of equations in the same way. In this case
   expressions are treated as equations, where the right side is set
   to zero.
*/

linalg::expr2Matrix := proc(eqs:Type::Union(DOM_SET,DOM_LIST))
    local i, j, b, A, R, tmp, vars, include, 
    extravars, n, neqs, neweq, newvars, newvarsSet, showassumptions, 
    subseqs, uservars, var, varsSet;
begin
    // Extracting the correct parameters:
    vars:= NIL;
    include:= NIL;
    uservars:= FALSE;
    showassumptions:= FALSE;
    case args(0)
    of 0 do 
      error("no arguments given")
    of 1 do 
      R:= Dom::ExpressionField();
      break
    of 2 do 
      tmp:= args(2);
      if contains( {DOM_LIST,DOM_SET},domtype(tmp) ) then
        R:= Dom::ExpressionField();
        vars:= tmp;
        uservars:= TRUE;
      elif tmp = Include then
        R:= Dom::ExpressionField();
        include:= tmp
      else
        R:= tmp
      end_if;
      break
    of 3 do 
      if contains( {DOM_LIST,DOM_SET},domtype(args(2)) ) then
        vars:= args(2);
        uservars:= TRUE;
        if args(3) = Include then 
          R:= Dom::ExpressionField();
          include:= args(3)
        else
          R:= args(3)
        end_if
      elif args(2) = Include then
        R:= Dom::ExpressionField();
        include:= args(2)
      else
        R:= args(2)
      end_if;
      break
    of 4 do 
      vars:= args(2);
      uservars:= TRUE;
      if args(3) = Include then 
        R:= args(4);
        include:= args(3)
      else 
        R:= args(3);
        include:= args(4)
      end_if;
      break
    otherwise
      error("expecting not more than four arguments")
    end_case;

    // Check the parameters:
    if testargs() then
      if R::dom = DOM_DOMAIN then
        if R::hasProp(Cat::CommutativeRing) <> TRUE then
          error("expecting component ring of category 'Cat::CommutativeRing'")
        end_if
      else
        error("expecting option 'Include'")
      end_if;
      if include <> NIL and include <> Include then
        error("expecting option 'Include'")
      end_if
    end_if;
    if vars <> NIL then
      if nops(vars) = 0 then
        error("missing indeterminates")
      elif has( map(vars,poly),FAIL ) or has(vars,Type::ConstantIdents) then
        error("invalid indeterminate found")
      end_if
    end_if;

    //-----------------------------------------
    // pre-processing
    //-----------------------------------------
  if not contains({DOM_SET, DOM_LIST}, domtype(eqs))
     then error("expecting a set or a list of equations or expressions")
  end_if;
  if domtype(eqs)=DOM_SET then eqs:=[op(eqs)] end_if; // convert to list 
  
  // convert polynomials 
  (if testtype(eqs[i],DOM_POLY) 
      then eqs[i]:= expr(eqs[i]);
   end_if;) $ i=1.. nops(eqs);

  // convert equations a=b to expressions a-b 
  (if testtype(eqs[i],"_equal") then 
      eqs[i]:= op(eqs[i],1)-op(eqs[i],2);
  end_if;) $ i=1.. nops(eqs);
   // force full evaluation of the equations: 
  // (eqs[i]:=eval(eqs[i]);) $ i=1..nops(eqs);

  if nops(eqs)=0 then
     if showassumptions then return([[],[],[]])
                        else return([])
     end_if
  end_if;

  if args(0) = 1 or not uservars
  then // get rid of Type::ConstantIdents (i.e., PI etc.).
       varsSet:= indets(eqs, PolyExpr) minus Type::ConstantIdents;
       // Note that indets(..,PolyExpr) enters things such as
       // 1/PI into varsSet. Eliminate this trash:
       extravars:= varsSet minus map(varsSet, float):
       // pick out those extravars that do not contain
       // symbolic variables
       extravars:= select(extravars, 
                          var ->  case domtype(float(var))
                                  of DOM_FLOAT do
                                  of DOM_COMPLEX do
                                     return(TRUE)
                                  otherwise
                                     return(FALSE)
                                  end_case):
       varsSet:= varsSet minus extravars;
       vars:=DOM_SET::sort(varsSet);
  else // linsolve is called with vars 
       if not contains({DOM_SET, DOM_LIST}, domtype(vars))
         then error("expecting a set or a list of unknowns")
       end_if;
       // force evaluation of vars ?
       // convert vars to List 
       if domtype(vars)<>DOM_LIST
          then //Variables are also needed as set 
               varsSet:= vars;
               vars:=DOM_SET::sort(vars);
          else varsSet:= {op(vars)};
       end_if; 
  end_if;

  n:=nops(vars);

  if n=0 then
     // eliminate trivial equations
     eqs:= map(eqs, proc(eq) begin 
                      if iszero(eq)
                         then null()
                      else eq end_if
                    end_proc);
     if eqs =[] and showassumptions=FALSE  then return([]) end_if;
     if eqs =[] and showassumptions=TRUE   then return([[], [], []]) end_if;
     if eqs<>[] and showassumptions=FALSE  then return(FAIL) end_if;
     if eqs<>[] and showassumptions=TRUE   then
                if indets(eqs,PolyExpr)={} then return([FAIL,[], []])
                                           else return([[], eqs, []])
                end_if;
     end_if;
  end_if;

  neqs:=nops(eqs);

     //--------------------------------------
     // compute coefficient matrix
     //--------------------------------------
     // eqs:= map(eqs, float):

     // vars = a list of indeterminates provided by the user
     // These can be expressions. This is a problem because
     // of diff(x[i], x[j]) <> 0! Replace every indeterminate
     // by a freshly generated identifier:
     newvars:= map(vars, var -> if domtype(var) <> DOM_IDENT then
                                     return(genident("x")) 
                                else return(var);
                                end_if);
     newvarsSet:= {op(newvars)};

     // generate the equations to be passed to subs
     // subs(eqs, [x = x, f(y) = x1, z = z]).
     // Better, reduce this to
     // subs(eqs, [f(y) = x1]):
     subseqs:= 
     zip(vars, newvars, (old, new) -> if old = new then 
                                           return(null())
                                      else return(old = new)
                                      end_if):
     A:= table();
     for i from 1 to neqs do
       neweq:= subs(eqs[i], subseqs);
       for var in (indets(neweq, PolyExpr) intersect newvarsSet) do
         tmp:= diff(neweq, var);
         j:= contains(newvars, var);
         A[i, j]:= tmp;
       end_for;
     end_for;

     //--------------------------------------
     // extract right hand side
     //--------------------------------------
     tmp:= subs(eqs, [vars[i] = 0 $ i=1..n]);
     b:= array(1..neqs, 1..1, [[-tmp[i]] $ i=1..neqs]):
     if include = Include then 
       for i from 1 to neqs do
         A[i,n+1]:= tmp[i];
       end_for;
     else 
       for i from 1 to neqs do
         A[i,n+1]:= -tmp[i];
       end_for;
     end_if;

     A:= Dom::Matrix(R)(neqs, n+1, A);
     //b:= Dom::Matrix(R)(neqs, 1, [[-tmp[i]] $ i=1..neqs]);
     
    if A = FAIL then
      error("coefficients can not be converted to elements of the component ring: ".expr2text(R))
    else
      return( A )
    end_if
end_proc:


/* OLD CODE - STILL HERE FOR SAFETY!!!!!!!!!!!!!!!!!!!!!!!!!!
   (did not respect sparsity)

linalg::expr2Matrix := proc(eqns:Type::Union(DOM_SET,DOM_LIST))
    local k, v, i, j, r, c, b, A, R, tmp, vars, include;
begin
    // Extracting the correct parameters:
    vars:= NIL;
    include:= NIL;
    case args(0)
    of 0 do 
      error("no arguments given")
    of 1 do 
      R:= Dom::ExpressionField();
      break
    of 2 do 
      tmp:= args(2);
      if contains( {DOM_LIST,DOM_SET},domtype(tmp) ) then
        R:= Dom::ExpressionField();
        vars:= tmp
      elif tmp = Include then
        R:= Dom::ExpressionField();
        include:= tmp
      else
        R:= tmp
      end_if;
      break
    of 3 do 
      if contains( {DOM_LIST,DOM_SET},domtype(args(2)) ) then
        vars:= args(2);
        if args(3) = Include then 
          R:= Dom::ExpressionField();
          include:= args(3)
        else
          R:= args(3)
        end_if
      elif args(2) = Include then
        R:= Dom::ExpressionField();
        include:= args(2)
      else
        R:= args(2)
      end_if;
      break
    of 4 do 
      vars:= args(2);
      if args(3) = Include then 
        R:= args(4);
        include:= args(3)
      else 
        R:= args(3);
        include:= args(4)
      end_if;
      break
    otherwise
      error("expecting not more than four arguments")
    end_case;

    // Check the parameters:
    if testargs() then
      if R::dom = DOM_DOMAIN then
        if R::hasProp(Cat::CommutativeRing) <> TRUE then
          error("expecting component ring of category 'Cat::CommutativeRing'")
        end_if
      else
        error("expecting option 'Include'")
      end_if;
      if include <> NIL and include <> Include then
        error("expecting option 'Include'")
      end_if
    end_if;
    if vars <> NIL then
      if nops(vars) = 0 then
        error("missing indeterminates")
      elif has( map(vars,poly),FAIL ) or has(vars,Type::ConstantIdents) then
        error("invalid indeterminate found")
      end_if
    end_if;

    include:= bool( include=Include );

    eqns:= [op(eqns)];
    r:= nops(eqns);

    // build the right side vector:
    b:= array(1..r,1..1);
    for i from 1 to r do
      if type( eqns[i] ) <> "_equal" then 
        b[i,1]:= 0; // convert later ...
      else
        // save the right side
        b[i,1] := op( eqns[i],2 );
        eqns:= subsop( eqns, i=op( eqns[i],1 ) )
      end_if
    end_for;

    if vars = NIL then
      vars := _union( indets(eqns[j],hold(PolyExpr)) $ j=1..r );
      vars := map( vars,x->(if has(x,Type::ConstantIdents) then null() else x end) );
    end_if;
 
    c:= nops(vars);
    A:= array(1..r,1..c+1,[ [R::zero $ c+1] $ r ]);
 
    for i from 1 to r do 
      // get constant coefficient j of i-th equation:
      j:= poly( eqns[i],[op(vars)] );
      (j:= coeff(j,0)) $ k=1..nops(vars);

      eqns := subsop( eqns,i=eqns[i]-j );
      j:= b[i,1] - j;

      // get constant coefficient v of j:
      v:= poly( j,[op(vars)] );
      (v:= coeff(v,0)) $ k=1..nops(vars);
      b[i,1]:= v;

      eqns:= subsop( eqns,i=eqns[i]-j+v ); 
      for j from 1 to c do
        v:= poly( eqns[i],[op(vars)] );
        case degree(v,op(vars,j))
        of 0 do 
          A[i,j]:= 0;
          break
        of 1 do 
          v:= coeff( v,op(vars,j),1 );
          if domtype(v) = DOM_POLY then
            if degree(v,[op(vars)]) <> 0 then
               error("expecting linear equations")
            end_if;
            v:= expr(v)
          end_if;
          A[i,j]:= v;
          break
        otherwise
          error("expecting linear equations")
        end_case
      end_for;
      if include then
        A[i,c+1]:= -b[i,1]
      else
        A[i,c+1]:= b[i,1]
      end_if
    end_for;

    A:= (Dom::Matrix(R))::coerce( A );
    if A = FAIL then
      error("coefficients can not be converted to elements of the component ring: ".expr2text(R))
    else
      return( A )
    end_if
end_proc:

*/

