/*
 factorLU( A ) -- computes LU decomposition of a matrix A

 A: matrix of category 'Cat::Matrix'

 linalg::factorLU(A) returns the list [L, U, pivindex, pivelem]: 
 (1) L as the lower triangular factor.
 (2) U as the upper triangular factor.
 (3) pivindex as a list [ p[1],p[2], .. ] representing the row exchanges
     in pivoting steps, i.e., B = PA = LU, where B[i,j] = A[p[i],j].
 (4) pivelem as a list of the used pivots.

 Note that LU factorization of a given matrix can be nonunique
 and may or may not exist (e.g. A=[[0,1],[1,0]]). The latter case
 is not recognized by this function and may lead to invalid factors
 L and U!

 The algorithm also works for singular A. In this case, pivelem
 contains zero pivot elements and either L or U are singular.

 L and U are chosen nonsingular iff A is nonsingular.
*/

linalg::factorLU:= proc(A)
    local m, n, k, i, j, d, R, L, p, pivindex, Mat,
          Riszero, Rinvert, Rmult, Rsubtract, Afloated;
begin
    if args(0) <> 1 then error("expecting one argument") end_if;
    Mat:= A::dom;

    if testargs() then
        if Mat::hasProp(Cat::Matrix) <> TRUE then
            error("expecting a matrix of 'Cat::Matrix'");
        elif not Mat::coeffRing::hasProp(Cat::Field) then
            error("expecting matrix over a 'Cat::Field'")
        end_if
    end_if;

    R:= Mat::coeffRing;

    case R
    of Dom::Float do
        userinfo(1,"call 'numeric::factorLU'");
        p:= numeric::factorLU( A );
        return( [Mat::create(p[1]), Mat::create(p[2]), p[3]] )

    of Dom::Real do
    of Dom::Complex do
        userinfo(1,"call 'numeric::factorLU' with option 'Symbolic'");
        p:= numeric::factorLU(  Mat::expr(A),Symbolic );
        return( [Mat::coerce(p[1]), Mat::coerce(p[2]), p[3]] )

    otherwise
        if R::hasProp(Cat::Field) and R::hasProp( Ax::systemRep ) then
            userinfo(1,"call 'numeric::factorLU' with option 'Symbolic'");
            p:= numeric::factorLU(  A, Symbolic );
            return( [Mat::create(p[1]), Mat::create(p[2]), p[3]] )
        elif (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
            // new branch for matrices e.g. over Dom::ExpressionFiel(normal)
            // with single floeting point components. This is the last hope 
            // for a numeric computation. Otherwise there is no other chance
            // than to skip to the symbolic code below. 
            p:= numeric::factorLU(Afloated, Symbolic);
            return([Mat::create(p[1]), Mat::create(p[2]), p[3]]);
        end_if
        
    end_case;

    // for any component ring which was not handled previously
    // use the following implementation of the LU decomposition:

    Riszero:= R::iszero;
    Rinvert:= R::_invert;
    Rmult:= R::_mult;
    Rsubtract:= R::_subtract;

    d:= Mat::matdim(A);
    m:= d[1]; n:= d[2];


    p:= [ i $ i=1..m ];	 // list of row permutations
    for j from 1 to min(m-1,n) do
        for pivindex from j to m do
           if not Riszero(A[pivindex,j]) then break end_if
        end_for;
        if pivindex>m then next end_if; // no nontrivial pivot element
 
        if pivindex<>j then
             A:= Mat::swapRow( A,j,pivindex );
             d:=p[j]; p[j]:= p[pivindex]; p[pivindex]:= d;
        end_if;

        if not Riszero( A[j,j] ) then
            d:= Rinvert( A[j,j] );
            for i from j+1 to m do
                A[i,j]:= Rmult( A[i,j],d ):
                if not Riszero( A[i,j] ) then
                    (A[i,k]:= Rsubtract( A[i,k], Rmult(A[i,j],A[j,k]) )) $ k=j+1..n
                end_if
            end_for
        end_if
    end_for;

   // LU data are stored in lower and upper part of A:
   // Build L:
   L:= Mat::identity( m );
   ((L[i,j]:= A[i,j]) $ i=j+1..m ) $ j=1..min(m-1,n);

   // Build U:
   d:= R::zero; ((A[i,j]:= d) $ i=j+1..m ) $ j=1..n;

   return([L,A,p])
end_proc:

