// --------------------------------------------------------------------------
// 29.11.04 (Kai and Walter):
//           the original version of this algorithm as published by
//           Ozello is replaced by the modified version below. The
//           original version still exists as linalg::frobeniusFormOzello.
//   
//           Note that linalg::jordanForm should call the old algorithm
//           linalg::frobeniusFormOzello rather than this modified
//           linalg::frobeniusForm, because the form tends to produce
//           smaller blocks than the latter (wrong Frobenius form, but
//           much better for further processing in jordanForm).
// --------------------------------------------------------------------------
/*
    linalg::frobeniusForm  --  returns the the Frobenius form of a matrix,
                               also called the Rational Canonical form of a matrix

    linalg::frobeniusForm(A <,All>)

    A  : a square matrix over Cat::Field
    All: identifier (option)

    Computes the Rational Canonical Form of the matrix A,

                                +-------+
                                |B1    0|
                                |  .    |
    i.e., the unique matrix B = |   .   | with companion matrices B1,...,Br
                                |    .  |
                                | 0   Br|
                                +-------+

                               +---------------+
                               |0       -a(0)  |
                     with Bi = |1 .      .     | for all i from 1 to r.
                               |  . .    .     |
                               |    1 0 -a(n-1)|
                               +---------------+

    The polynomial m(i):=X^n+a(n-1)*X^(n-1)+...+a(1)*X+a(0) thereby is the
    minimal polynomial of the matrix Bi.
    For the polynomials m(1),...,m(r) we receive: m(i) divides m(i+1)
    for i from 1 to r-1.

    By giving the command 'linalg::frobeniusForm(A,All)', the tranformation
    matrix P is given, so that we get: (P^-1) * A * P = B


    Reference: Calcul exact des formes de Jordan et de Frobenius d'une
               matrice, by Patrick Ozello, 
               "Docteur de I'Universite
               (Phd-Thesis, Technologique et Medicale de Grenoble, 1984
               algorithms on pages 30 to 43.)

   Modified, because the algorithm in Ozello's thesis was 
   buggy (e.g., for diagonal matrices, the block structure
   is messed up)
*/

linalg::frobeniusForm:= proc(A)
local frobenius, _frobenius, 
      polymin, chercheco, chercheli, unbloc, zeroadroite,
      multCol, addCol, swapCol, multRow, addRow, swapRow, 
      n, B, Rnormal;
begin
//-----------------------------------------------
frobenius:= proc(A, n, m)
local L, d, k, i;
begin
    d := 1;
    [L, A] := _frobenius(A, n, m, d);
    k := nops(L);
    while 0 < k do
        for i from 1 to k do
            A:= addCol(A, d, n, L[i], d, 1);
            A:= addRow(A, d, m, L[i], d, -1)
        end_for;;
        [L, A] := _frobenius(A, n, m, d);
        k := nops(L);
        if 1 < k then 
           d := L[1]; 
           L := L[2 .. k]
        end_if;
        k := k - 1
    end_while;
    if args(0) = 3 then
         return(A)
    else return([A[1..m, 1..m], A[m+1..2*m, 1..m]]);
    end_if;
end_proc:
   //-----------------------------------------------
   _frobenius:= proc(A, n, m, d)
   local j, L;
   begin
       j := d - 1;
       L := [];
       while j < m do
           [j, A] := unbloc(A, n, m, j + 1);
           if j < m then 
              L := [op(L), j + 1] 
           end_if
       end_while;
       [L, A];
   end_proc:
   //-----------------------------------------------
   unbloc:= proc(A, n, m, h)
   local j, k;
   begin
       k:= h;
       while k <= m do
           if k <> h then
               A:= swapCol(A, h, n, h, k);
               A:= swapRow(A, h, m, h, k)
           end_if;
           [j, A] := polymin(A, n, m, h);
           A:= zeroadroite(A, n, m, h, j);
           k:= chercheco(A, m, h, j + 1)
       end_while;
       [j, A]
   end_proc:
   /*---------------------------------------------------
        Procedure "polymin"
        At first we determine an index k, so that A[j,k]<>0.
        The first step of this procedure will turn the matrix A to a
        matrix of the form:
            +---------------+
            |   C1  |   B1  |
            |       |       |
            |-------|-------|
            | 0...0 |       |
            | 0...0 |   B2  |
            +---------------+
   --------------------------------------------------*/
   polymin:= proc(A, n, m, h)
   local j, k, g, pivot;
   begin
       j := h + 1;
       while j - 1 <= m do
           // Procedure "chercheli":
           // Returns an index k so that B[j,k] <> 0
           k := chercheli(A, m, j - 1);
           if k = m + 1 then return([j - 1, A]) end_if;
           if k <> j then
               A:= swapRow(A, j - 1, m, k, j);
               A:= swapCol(A, 1, n, k, j)
           end_if;
           pivot := Rnormal(A[j, j - 1]);
           if pivot <> 1 then
               A:= multRow(A, j - 1, m, j, _invert(pivot));
               A:= multCol(A, h, n, j, pivot)
           end_if;
           for g from h to m do
               if g <> j and A[g, j - 1] <> 0 then
                   pivot := Rnormal(A[g, j - 1]);
                   if pivot <> 0 then
                       A:= addRow(A, j - 1, m, g, j, -pivot);
                       A:= addCol(A, h, n, g, j, pivot)
                   end_if
               end_if
           end_for;;
           j := j + 1
       end_while;
       [j, A];
   end_proc:
   /*---------------------------------------------------
       Procedure "zeroadroite"
       Transforms the matrix to a matrix of the form

           +---------------+
           |   C1  |*******|
           |       |0.....0|
           |-------|-------|
           | 0...0 |       |
           | 0...0 |   B3  |
           +---------------+
   ---------------------------------------------------*/
   zeroadroite:= proc(A, n, m, g, j)
   local h, k, pivot;
   begin
       for h from j downto g + 1 do 
           for k from j + 1 to m do
               pivot := Rnormal(A[h, k]);
               if pivot <> 0 then
                   A:= addCol(A, g, j, h - 1, k, -pivot);
                   if n <> m then 
                      A:= addCol(A, m + 1, n, h - 1, k, -pivot)
                   end_if;
                   A:= addRow(A, j + 1, m, h - 1, k, pivot)
               end_if;
           end_for;
       end_for;
       A;
   end_proc:
   /*-------------------------------------------------
     chercheli returns an index k such that A[j,k] <> 0
   ---------------------------------------------------*/
   chercheli:= proc(A, n, k)
   local j;
   begin
       j := k + 1;
       while j <= n and iszero(Rnormal(A[j, k])) do 
           j := j + 1 
       end_while ;
       return(j);
   end_proc:
   /*-------------------------------------------------
       chercheco: analog to cherceli 
   ---------------------------------------------------*/
   chercheco:= proc(A, n, k, j)
   begin
       while j <= n and iszero(Rnormal(A[k, j])) do 
          j := j + 1 
       end_while;
       return(j);
   end_proc:
   //-----------------------------------------------
   swapRow:= proc(A, c1, c2, r1, r2)
   local j;
   begin
       for j from c1 to c2 do
           [A[r1, j], A[r2, j]]:= [A[r2, j], A[r1, j]];
       end_for;
       A;
   end_proc:
   //-----------------------------------------------
   swapCol:= proc(A, r1, r2, c1, c2)
   local k;
   begin
       for k from r1 to r2 do
           [A[k, c1], A[k, c2]]:= [A[k, c2], A[k, c1]];
       end_for;
       A;
   end_proc:
   //-----------------------------------------------
   addCol:= proc(A, r1, r2, c1, c2, c)
   local k;
   begin
       for k from r1 to r2 do 
           A[k, c2] := c*A[k, c1] + A[k, c2]
       end_for;
       A;
   end_proc:
   //-----------------------------------------------
   addRow:= proc(A, c1, c2, r1, r2, c)
   local j;
   begin
       for j from c1 to c2 do 
           A[r1, j] := c*A[r2, j] + A[r1, j]
       end_for;
       A;
   end_proc:
   //-----------------------------------------------
   multCol:= proc(A, r1, r2, j, c)
   local k;
   begin
       for k from r1 to r2 do 
          A[k, j] := c*A[k, j] ;
       end_for;
       A;
   end_proc:
   //----------------------------------------------
   multRow:= proc(A, c1, c2, k, c)
   local j;
   begin
       for j from c1 to c2 do 
           A[k, j] := c*A[k, j];
       end_for;
       A;
   end_proc:
   //----------------------------------------------
   
   //------------------------------------------------
   // main of linalg::frobeniusForm
   //------------------------------------------------
   if args(0) < 1 or 2 < args(0) then
       error("wrong number of arguments");
   end_if;
   if testargs() then
       if A::dom::hasProp( Cat::Matrix ) <> TRUE then
           error("argument is not of 'Cat::Matrix'")
       end_if;
       if not A::dom::coeffRing::hasProp( Cat::Field ) then
           error("expecting matrix over 'Cat::Field'")
       end_if;
       n:= A::dom::matdim(A);
       if n[1] <> n[2] then
           error("not a square matrix")
       end_if;
       if args(0) = 2 and args(2) <> hold(All) then
           error("2nd argument must be the option 'All'")
       end_if
   end_if;
   n := A::dom::matdim(A)[1]:
   Rnormal:= A::dom::coeffRing::normal;
   if Rnormal = FAIL then 
     Rnormal:= () -> args(1);
   end_if;
   if args(0) = 2 then
        if args(2) <> All then
           error("expecting option 'All' as second argument");
        end_if;
        B:= linalg::stackMatrix(A, A::dom::identity(n));
        B:= map(frobenius(B, 2*n, n, All), map, Rnormal);
        // If A is Dom::SquareMatrix(n,R) or Dom::MatrixGroup(n,n,R) etc,
        // linalg::stackMatrix changes the type to Dom::Matrix(R).
        // Reconvert to the original type
        if B::dom <> A::dom then
           A:= [A::dom::convert(B[1]), A::dom::convert(B[2])];
        end_if; 
   else A:= map(frobenius(A, n, n), Rnormal);
   end_if;
   return(A);
end_proc:
