/*
 hessenberg( A<, All> ) -- computes a Hessenberg matrix of A

   A:   a square matrix of category 'Cat::Matrix'
 All: (optional) identifier

 linalg::hessenberg(A) returns a Hessenberg matrix H of the n x n 
 matrix A. This is a matrix with the following properties:

        For all i,j in {1,...,n} with i>j+1: H[i,j] = 0.

 For each matrix over a field exists a Hessenberg matrix which 
 is similar to A.

 linalg::hessenberg(A,All) returns the list [H,T] where
 H is a Hessenberg matrix H of A and T the corresponding
 transformation matrix, i.e., a matrix such that H = T*A*T^(-1).

 The component domain must be of category Cat::Field.

 Reference: K.-H. Kiyek / F. Schwarz: "Lineare Algebra"
            Teubner Studienbuecher Mathematik, B.G.Teubner
            Stuttgart, Leipzig, 1999.
*/

linalg::hessenberg:= proc(A)
    local n, i, k, a, R, T, Mat, Riszero, Rdivide, return_T;
begin
    if args(0) < 1 then
      error("expecting a least one argument")
    end_if;
    Mat:= A::dom;

    if testargs() then
      if args(0) > 2 then 
        error("too many arguments") 
      elif Mat::hasProp(Cat::Matrix) <> TRUE then
        error("expecting a matrix of 'Cat::Matrix'");
      elif not Mat::coeffRing::hasProp(Cat::Field) then
        error("expecting matrix over a 'Cat::Field'")
      elif args(0) = 2 and args(2) <> hold(All) then
        error("expecting option 'All'")
      end_if
    end_if;

    return_T:= bool(args(0)=2);

    R:= Mat::coeffRing;
    Riszero:= R::iszero;
    Rdivide:= R::_divide;

    n:= Mat::matdim(A);
    if n[1] <> n[2] then
      error("not a square matrix")
    end_if;
    n:= n[1];

    if return_T then T:= Mat::identity( n ) end_if;
    for k from 1 to n-2 do
      if Riszero( A[k+1,k] ) then
        for i from k+2 to n do
          if not Riszero( A[i,k] ) then
            A:= Mat::swapRow( A,k+1,i );
            A:= Mat::swapCol( A,k+1,i );
            if return_T then 
              T:= Mat::swapRow( T,k+1,i )
            end_if;
            break
          end_if
        end_for
      end_if;
      if not Riszero( A[k+1,k] ) then
        for i from k+2 to n do
          a:= Rdivide( A[i,k], A[k+1,k] );
          A:= linalg::addRow( A,k+1,i,R::_negate(a) );
          A:= linalg::addCol( A,i,k+1,a );
          if return_T then 
            T:= linalg::addRow( T,k+1,i,R::_negate(a) )
          end_if
        end_for
      end_if
    end_for;
    if return_T then 
      return( [A,T] ) 
    else 
      return( A ) 
    end_if
end_proc:

