/*
        inverseLU( A )     -- compute the inverse of A by LU decomposition
        inverseLU( L,U,p ) -- compute the inverse of A=LU.

        A,L,U: nonsingular square matrix of category 'Cat::Matrix'
        p    : list of positive integers

        linalg::inverseLU( A ) determines the inverse of the nxn matrix
        A by solving the linear system Ax=E using LU decomposition
        (E denotes the nxn identity matrix).

        If A is singular then an error occurs.

        linalg::inverseLU(L, U, p) computes the inverse of the matrix 
        A = L*U with L and R nonsingular matrices turned out
        by LU-decomposition of a (nonsingular) matrix A. p is a 
        list [ r[1],r[2], ... ] representing the row exchanges of 
        A in pivoting steps.

        The matrices must be defined over a 'Cat::Field'.
*/

linalg::inverseLU:= proc(L,U,p)
    local i, j, n, LU, Mat;
begin
    if args(0) <> 1 and args(0) <> 3 then  
        error("invalid number of arguments" ) 
    end_if;

    Mat:= L::dom;

    if testargs() then
        if Mat::hasProp(Cat::Matrix) <> TRUE then
            error("expecting a matrix of category 'Cat::Matrix'")
        elif not Mat::coeffRing::hasProp(Cat::Field) then
            error("expecting matrix over a 'Cat::Field'")
        elif args(0) = 2 and Mat <> U::dom then
            error("types of matrices differ")
        end_if
    end_if;
    n:= Mat::matdim(L);
    if n[1] <> n[2] then 
        error("not a square matrix")
    elif testargs() and args(0) = 3 then  
        if Mat::matdim(U) <> n then
            error("incompatible operands")
        elif not testtype( p,Type::ListOf(Type::PosInt,op(n)) ) then
            error("invalid list for row interchanges")
        end_if
    end_if;

    n:= n[1];
    if args(0) = 1 then
        LU:= linalg::factorLU( L );
        p:= LU[3]
    else
        LU:= [L,U]
    end_if;

    L:= linalg::matlinsolveLU(LU[1], LU[2], Mat::identity( n ));
    if p <> [ i $ i=1..n ] then
        U:= L;
        ( (U[i,p[j]]:= L[i,j];) $ j=1..n;) $ i=1..n;
        return( U )
    else
        return( L )
    end_if
end_proc:

