/*
 linalg::inverse( A )

 A: square matrix
 
 Computes the matrix inverse over a field.
*/

linalg::inverse :=
proc(A)
 local m, n, R, Mat, Rnormal, det, hasFloats, symbolicity, 
       subst, B, kernel, kernelCols, ops, floatmode;
begin
  if args(0) < 1 then
     error("expecting an argument of category 'Cat::Matrix'");
  end_if;
/*
  if args(0) > 2 then
      error("expecting at most two arguments: a matrix of category 'Cat::Matrix' ".
            "and, optionally, one of the flags '...', '...', or '...'");
  end_if;
*/
  Mat:= A::dom;
  if Mat::hasProp(Cat::Matrix) <> TRUE then
      error("expecting a matrix of 'Cat::Matrix'")
  elif not (Mat::coeffRing)::hasProp(Cat::Field) then
      error("expecting a matrix over 'Cat::Field'")
  end_if;
  [m, n]:= Mat::matdim(A);
  if m <> n then
     error("expecting a square matrix")
  end_if;
  R:= Mat::coeffRing;

  // =======================
  // Special case Dom::Float
  // =======================

  if R = Dom::Float then
     userinfo(1,"call 'numeric::inverse'");
     A:= numeric::inverse(A, ReturnType = Dom::Matrix()):
     if A = FAIL then
        return(FAIL);
     else
        return(Mat(A));
     end_if:
  end_if:

  // ================================
  //  Special cases dimension = 1, 2
  // ================================

  case n
  of 1 do 
      if linalg::zeroTest(A[1,1]) <> FALSE then
        return(FAIL);
      else
        return(Mat(1, 1, [[1/A[1,1]]]));
      end_if:
  of 2 do
     userinfo(1,"using explicit formula for the inverse");
     if (Rnormal:= R::normal) = FAIL then
       if R::hasProp(Ax::systemRep) then 
         Rnormal:= normal 
       else 
         Rnormal:= () -> args(1); 
       end_if
     end_if;
     det:= Rnormal(A[1,1]*A[2,2] - A[1,2]*A[2,1]);
     if linalg::zeroTest(det) <> FALSE then
        return(FAIL)
     end_if;
     return(map(Mat(2,2, [[ A[2,2], -A[1,2]],
                          [-A[2,1],  A[1,1]]]), n->Rnormal(n/det)));
  end_case;

  // =============
  // Float inverse
  // =============
  if R = Dom::ExpressionField() then
     //---------------------------------------------------
     // is there a float element in the matrix?
     //---------------------------------------------------
     ops:= {A::dom::nonZeroOperands(A)};
     if hastype(subs(ops, I = #I), DOM_FLOAT) then
        floatmode:= TRUE;
     else
        floatmode:= FALSE;
     end_if;
     //---------------------------------------------------
     // There is a float element in the matrix. Try to convert
     // all elements to floats. We need a copy B of A, because
     // we still need the original A if no float conversion is
     // possible.
     //---------------------------------------------------
     if floatmode then
        A:= A::dom::mapNonZeroes(A, float);
        ops:= {A::dom::nonZeroOperands(A)};
        if map(ops, domtype) minus  {DOM_FLOAT, DOM_COMPLEX} <> {} then
           floatmode := FALSE;
        end_if;
     end_if;
     //---------------------------------------------------
     // if floatmode is still TRUE, all elements of A were 
     // successfully converted to floats. Call 
     // numeric::inverse, which tries HardwareFloats!
     //---------------------------------------------------
     if floatmode then
        B:= numeric::inverse(A, ReturnType = Dom::Matrix());
        if not has(B, FAIL) and
           B::dom <> Mat then
           return(Mat(B));
        else
           return(B);
        end_if;
     end_if;
  end_if;

  // ==========
  // Cat::Field
  // ==========

  if R::hasProp(Cat::Field) then

     [A, subst]:= [linalg::rationalizeMatrix(A, ReturnType = matrix)];
     symbolicity:= linalg::symbolicity(A, "MatrixIsRationalized");

     // ToDo: use misc::maprec

     hasFloats:= has(map({op(A)}, domtype@op, 1), DOM_FLOAT);  
     if hasFloats then
        A:= map(A, numeric::rationalize);
     end_if:
     if symbolicity = 0 then // A contains only numerical entries
        if hasFloats then
           A:= numeric::inverse(A, NoWarning);
        else
           A:= numeric::inverse(A, Symbolic);
        end_if;
        if A = FAIL then 
           return(FAIL);
        end_if;
        if subst <> {} then 
           A:= A::dom::mapNonZeroes(A, eval@subs, subst);
        end_if;
        if A::dom <> Mat then 
           A:= Mat(A);
        end_if;
        return(A);
     end_if:

     if symbolicity = 1/2 then // A contains very few symbolic elements
        A:= linalg::inverseOverField(A);
        if A = FAIL then
           return(FAIL);
        end_if;
        if subst <> {} then 
           A:= A::dom::mapNonZeroes(A, eval@subs, subst);
        end_if;
        if A::dom <> Mat then 
           A:= Mat(A);
        end_if;
        if hasFloats then
           return(A::dom::mapNonZeroes(A, float))
        else
           return(A);
        end_if:
     end_if:

     if symbolicity = 1 then // A contains a few symbolic elements
        userinfo(3, "calling linalg::polynomializeMatrix");
        A:= [linalg::polynomializeMatrix(A.(A::dom::identity(n)), 
                                         ReturnType = matrix, 
                                         "MatrixIsRationalized")][1];
        if A = FAIL then
           return(FAIL);
        end_if;
        [A, B]:= [A[1..n, 1..n], A[1..n, n+1 .. 2*n]];
        userinfo(3, "calling linalg::SSS with the option 'UsePoly'");
        A:= linalg::SSS(A, B, "UsePoly");
        if A = [] then 
           return(FAIL);
        end_if;
        [A, kernel, kernelCols]:= A;
        if nops(kernel) > 0 or A = FAIL then
           return(FAIL);
        end_if;
        if subst <> {} then
           A:= A::dom::mapNonZeroes(A, eval@subs, subst):
        end_if;
        if A::dom <> Mat then 
           A:= Mat(A);
        end_if;
        if hasFloats then
           return(A::dom::mapNonZeroes(A, float))
        else
           return(A);
        end_if:
     end_if:
        
     if linalg::symbolicity(A) = 2 then 
        userinfo(3, "calling linalg::symbolicInverse");
        A:= linalg::symbolicInverse(A, subst);
        if A = FAIL then 
           return(A);
        end_if;
        if subst <> {} then
           A:= A::dom::mapNonZeroes(A, eval@subs, subst):
        end_if;
        if A::dom <> Mat then 
           A:= Mat(A);
        end_if;
        if hasFloats then
           return(A::dom::mapNonZeroes(A, float));
        else
           return(A);
        end_if;            
     end_if:
  end_if;

  // ==============
  // not Cat::Field
  // ==============
  // ToDo
  return(FAIL);
end_proc:

linalg::inverse := funcenv(linalg::inverse):
linalg::inverse::Content := (Out, data) -> Out::Capply(Out::Cinverse, 
                                                   Out(op(data))):
