// inverse of a matrix over a coeff ring of category Cat::Field

linalg::inverseOverField:= 
proc(A)
  local Mat, m, n, R, Rone, Rzero, Rnormal, polyR, types, B, L, list, i, j, 
        nt, minnt, usenormal, nonormalTypes, nonzeroes,
        jthcol, pos, piv, pivpos, pivindex, pivfound,
        minlength, thislength, term, terms, l;
begin
     userinfo(1, "using linalg::inverseOverField (via Gaussian elimination)"):
     Mat:= A::dom:
     [m, n]:= Mat::matdim(A);
     if m <> n then error("not a square matrix"); end_if:

     R:= Mat::coeffRing;
     polyR:= R:
     Rone:= R::one;
     Rzero:= R::zero;
     if R::characteristic <> FAIL and
        R::characteristic > 0 and
        R = Dom::IntegerMod(R::characteristic) then
          polyR:= IntMod(R::characteristic);
          Rzero:= 0;
          Rone := 1;
     end_if;
     if R::hasProp(Ax::systemRep) then
          polyR:= Expr;
          Rzero:= R::zero:
          Rone := R::one;
     end_if;

     //------------------------------------------------------
     // generic code for inverting a (sparse) matrix over any
     // ring of Cat::Field:
     //------------------------------------------------------

     //-----------------------------------------------
     // convert A from polys = columns to polys = rows.
     // Use B as container
     //-----------------------------------------------
     B:= [0 $ m]; // initialize container for rows
     // build up row for row to avoid doubling of memory
     // due to doubly nested sequences
     if polyR = IntMod(R::characteristic) then
       for i from 1 to m do
         B[i] := poly([[expr(A[i,j]), n-j] $j=1..n],[hold(`#_X_`)],polyR);
       end_for;
     else
       for i from 1 to m do
         B[i] := poly([[A[i,j], n-j] $j=1..n],[hold(`#_X_`)],polyR);
       end_for;
     end_if;
     // Now B = A but with polys = rows instead of polys = columns
     A:= B;

     if has(A, FAIL) then error("should not happen (1)") end_if;
     if R::one = FAIL then error("should not happen (2)") end_if;

     // There is no need to use normal if only numerical
     // types are involved:
     usenormal:= FALSE;
     nonormalTypes:= {DOM_FLOAT, DOM_COMPLEX,
                      DOM_INT, DOM_RAT, DOM_INTERVAL};
     if R::normal <> FAIL then
        Rnormal:= R::normal;
        for i from 1 to m do
          types:= map({coeff(A[i])}, domtype):
          if types minus nonormalTypes <> {} then
             usenormal:= TRUE;
             break;
          end_if;
        end_for;
     else 
        if R::hasProp(Ax::systemRep) then
          Rnormal:= normal
        else
          Rnormal:= () -> args(1);
        end_if;
     end_if;

     // initialize B as identity matrix, interpretation: polys=rows
     B:= [poly([[Rone,j]], [hold(`#_X_`)], polyR) $ j=1..n];
     if has(B, FAIL) then error("should not happen (3)") end_if;

     piv:= [ 0 $ n]; // initialize container for pivot elements
     for j from 1 to n do
         if usenormal = TRUE then
            for i from j to n do
              A[i]:= mapcoeffs(A[i], Rnormal);
              B[i]:= mapcoeffs(B[i], Rnormal);
            end_for;
         end_if;

         nt:=n-j;
         // j.th col of A from diagonal downwards. Only store the
         // non-trivial terms in jthcol. First generate a dense list.
         // Eliminate trivial entries by converting to a polynomial,
         // then reconvert to list by poly2list. This is faster
         // than sparseList:= select(denseList , x -> not iszero(x[1])):

         userinfo(2, "searching for pivot element in column ".expr2text(j));

         jthcol:= poly([[coeff(A[i],nt), i] $ i=j..n ], [hold(`#_X_`)], polyR);
         jthcol:= poly2list(jthcol);
         if jthcol=[] then
         userinfo(1,"matrix not invertible");
         return(FAIL);
         end_if;
         // start pivot search
         pivpos:= 1;         // position of pivot element in list jthcol
         pivfound:= FALSE;   // pivot element not found yet.
         minnt:= nt+1;       // minimal number of terms in row. The row
                             // with the smallest number of terms is used
                             // as pivot row in order to preserve sparsity
            //-------------------------------
            // symbolic pivot strategy
            //-------------------------------
            minlength:= 0;      // complexity of potential pivot elements in jthrow.
                                // Should be minimal (this is the secondary
                                // criterion for the pivot search).
            for pos from 1 to nops(jthcol) do // just search the non-trivial terms
               i:= jthcol[pos][2]: //row index i corresponding to
                                   //pos-th term in jthcol
               // cut diagonal element from i-th row
               A[i]:= lmonomial(A[i],Rem)[2];
               // A[i] is now remainder of row i without j.th element
               // j.th elements are stored in jthcol
               if not pivfound then
                  nt:= nterms(A[i]); //number of non-trivial terms in
                                     // remainder of row i
                  if nt=0 then pivfound:= TRUE; end_if; // ideal pivot element!!
                  //pivot criterion nt<minnt is good strategy for non-float
                  //computations.
                 if nt<minnt then // primary pivot criterion
                    pivpos:= pos; minnt:= nt;
                 else // secondary pivot criterion
                      thislength:= length(jthcol[pos][1]);
                      // avoid initialization of minlength by infinity,
                      // this would be too expensive.
                     if minlength=0 then minlength:= thislength; end_if;
                     if nt=minnt and thislength<minlength then
                         pivpos:= pos; minlength:= thislength;
                     end_if;
                 end_if;
               end_if;
            end_for;
         // jth pivot element found, store in table piv
         piv[j]:=   jthcol[pivpos][1];
         pivindex:= jthcol[pivpos][2];
         userinfo(3, "choosing pivot element ".expr2text(R::coerce(piv[j])).
                     " (row ".expr2text(pivindex).")"):
         // do the elimination
         for term in jthcol do
           i:= term[2];
           if i<>pivindex then
             l:= term[1]/piv[j];
             if l=FAIL then
                error("should not happen (".expr2text(3,j).")");
             end_if;
             case l
             of R::one do
                       A[i]:=A[i]-A[pivindex];
                       B[i]:=B[i]-B[pivindex];
                       break;
             of R::negate(R::one) do
                       A[i]:=A[i]+A[pivindex];
                       B[i]:=B[i]+B[pivindex];
                       break;
             otherwise /* -----------------------------------------------
                       // fast but not correct for non-commutative rings:
                       A[i]:= A[i]-mapcoeffs(A[pivindex],_mult,l);
                       B[i]:= B[i]-mapcoeffs(B[pivindex],_mult,l);
                       ------------------------------------------------*/
                       A[i]:= A[i]-mapcoeffs(A[pivindex], (x, y) -> y*x, l);
                       B[i]:= B[i]-mapcoeffs(B[pivindex], (x, y) -> y*x, l);
             end_case;
           end_if;
         end_for; // elimination done

         if pivindex<>j then // exchange rows
            [A[j],A[pivindex]]:= [A[pivindex],A[j]];
            [B[j],B[pivindex]]:= [B[pivindex],B[j]];
         end_if;
     end_for;
     // ----------- elimination finished ----------------------
     // Now A is in upper triangular form with diagonal elements
     // piv[1],..,piv[n] and strictly upper triangular rows given
     // by the polynomials A[1],..A[n-1], A[n] = poly(0,x) :
     // All piv[i]<>0, otherwise FAIL would have been returned above.
     // Start backsubstitution:
     // ----------------------------------------------------------
     // fast, but not correct for non-commutative rings:
     // B[n]:= mapcoeffs(B[n],_divide,piv[n]);
      //----------------------------------------------------------
     B[n]:= mapcoeffs(B[n], (x, y) -> y*x, 1/piv[n]);
     if B[n]=FAIL then error("should not happen (4)") end_if;
     for i from n-1 downto 1 do
         terms:= poly2list(A[i]);
      //---------------------------------------------------------------------
      // fast, but not correct for non-commutative rings:
      // B[i]:= B[i] -
      //       _plus(mapcoeffs(B[n-term[2]],_mult,term[1]) $ term in terms );
      // B[i]:= mapcoeffs(B[i], _divide, piv[i]);
      //---------------------------------------------------------------------
         B[i]:= B[i] -
             _plus(mapcoeffs(B[n-term[2]], (x, y) -> y*x, term[1]) $ term in terms );
         B[i]:= mapcoeffs(B[i], (x, y) -> y*x, 1/piv[i]);
         if B[i]=FAIL then error("should not happen (4)") end_if;
         if usenormal = TRUE then
              B[i]:= mapcoeffs(B[i], Rnormal);
           end_if;
     end_for;
     // Now B stores the inverse of A with polys = rows.

     //-----------------------------------------
     // Still need to reconvert to polys = cols.
     //-----------------------------------------

     nonzeroes:= _plus(nops(poly2list(B[i])) $ i=1..m);
     if nonzeroes < m*n/2 then //sparse case
        // A = container for the columns
        A:= [[[Rzero,0] $ m] $ n];
        for i from 1 to m do
           L:= poly2list(B[i]);
           for list in L do
              // fill the columns
              A[list[2]][i]:= [list[1], i];
           end_for;
        end_for;
        // So far, the row A[j] is just a list.
        // Convert the columns to polynomials.
        for j from 1 to n do
           A[j]:= poly(A[j], [hold(`#_X_`)], polyR);
        end_for;
     else //dense case
        A:= [0 $ n]; // initialize container for columns
        // build up column for column to avoid doubling of memory
        // due to doubly nested sequences
        for j from 1 to n do
           A[j] := poly([[coeff(B[i], j), i] $i=1..m],[hold(`#_X_`)],polyR);
           if has(A[j],FAIL) then
              error("should not happen (5)")
           end_if;
       end_for;
     end_if;
     A:= matrix::create(m, n, A);
     return(Mat(A));
end_proc:
