/*
        linalg::jacobian -- computes the Jacobian matrix of a vector function

        jacobian(f, v)

        f : list or vector of expressions (the vector function)
        v : list of variables

 jacobian(f,v) computes the Jacobian matrix of f with respect to v.
 The (i,j)-th entry of matrix is defined to be diff(f[i],v[j]).

 If f is given as a list then the Jacobian matrix is defined over
 ExpressionField().

 If f is a vector over a coefficient domain R then the Jacobian
 matrix is defined over R.
*/

linalg::jacobian := proc(f,x)
    local i, j, m, n;
begin
    if testargs() then
      if args(0) <> 2 then error("expecting 2 arguments") end_if;
      if domtype(f) <> DOM_LIST and not testtype( f,linalg::vectorOf(Type::AnyType) ) then
        error("expecting vector function in form of a list or vector")
      elif domtype(x) <> DOM_LIST then
        error("expecting a list of (indexed) identifiers")
      end_if
    end_if;

    n:= nops(x); m:= nops(f);
    if domtype(f) = DOM_LIST then
      return(Dom::Matrix()( m, n, [[diff(f[i],x[j]) $ j=1..n] $ i=1..nops(f)] ))
    else
      n:= nops(x);
      m:= nops(f);
      f::dom::coerce( 
          array( 1..m, 1..n, [[diff(f[i],x[j]) $ j=1..n] $ i=1..m] )
                    )
    end_if
end_proc:

