/*
        linalg::laplacian -- computes the laplacian of a scalar function

  Call:   linalg::laplacian(f, x, <ogSystem, <c>>)

  Parameters
          f       : an arithmetical expression
          x       : a list of (indexed) identifiers
          ogSystem: (optional) name or list of the scale factors of an 
                    orthogonally curvilinear coordinate system.
                    Also something like
                    linalg::orCoordTab[name, Scales](op(x), <parameters>)
        c       : the parameter for the coordinate systems
                  'Torus' and 'EllipticCylindrical'

 This function computes the laplacian of f in the variables given 
 in x. If only two arguments are given, linalg::laplacian operates on
 the cartesian coordinate system: the result is
         diff(f, x[1], x[1]) + diff(f, x[2], x[2]) + ... 

 Otherwise, linalg::laplacian operates on the orthogonally 
 curvilinear coordinate system specified by a list of scale 
 factors or by a name which scale factors are defined in table 
 linalg::ogCoordTab.

        Presently, the following orthogonal coordinate systems
        are available in linalg::ogCoordTab
                   Cartesian
                   Cylindrical
                   Spherical
                   Torus
                   EllipticCylindrical
                   RotationParabolic
                   ParabolicCylindrical


Examples:
>> linalg::laplacian(x[1] *exp(x[2]), [x[1], x[2]])

                        x[1] exp(x[2])

>> linalg::laplacian(x[1] *exp(x[2]), [x[1], x[2], Cartesian])

                        x[1] exp(x[2])

>> linalg::laplacian(r^2*cos(phi)*z, [r, phi, z], Cylindrical)

                         3 z cos(phi)

>> linalg::laplacian(r^2*cos(phi)*z, [r, phi, z], [1, r, 1])

                         3 z cos(phi)

>> linalg::laplacian(r^2*cos(phi)*z, [r, phi, z], 
                     linalg::ogCoordTab[Cylindrical, Scales](r, phi, z)
                    )

                         3 z cos(phi)

>> linalg::laplacian(u^2*v^3*w^4, [u, v, w], 
                     linalg::ogCoordTab[EllipticCylindrical, Scales](u, v, w, z)
                    )

         3  4      2    4       2  3  2  2       2
     (2 v  w  + 6 u  v w  - 12 u  v  w  z  cos(v)  +

            2  3  2  2        2      2        2    2       2
        12 u  v  w  z  cosh(u) ) / (z  cosh(u)  - z  cos(v) )
*/

linalg::laplacian := proc(f, x)
    local g, i, gradient, c;
begin

    // type checking:
    if testargs() then
        if args(0) < 2 then
            error("expecting at least 2 arguments")
        elif args(0) > 4 then
            error("expecting no more than 4 arguments")
        elif not testtype( f,Type::Arithmetical ) then
            error("1st argument must be of 'Type::Arithmetical'")
        elif not testtype( x,Type::ListOf(Type::Unknown) ) then 
            error("2nd argument must be a list of (indexed) identifiers")
        end_if;
        if args(0) = 3 then
            g:= args(3);
            if domtype(g) <> DOM_LIST 
                 and type(linalg::ogCoordTab[g,hold(Scales)]) = "_index" 
            then
                error("illegal or undefined coordinate system")
            end_if;
            // presently only 3-dim transformations are installed
            // in ogCoordTab:
            if domtype(g) <> DOM_LIST and nops(x) <> 3 then
               error("unknown orthogonal coordinate system")
            end_if;
        end_if
    end_if;

    // this is the body of linalg::grad:
    if args(0) = 3 then
        if args(0) = 4 then
           c:= args(4)
        else
           c:= null();
        end_if;
        g:= args(3);
        if domtype(g) <> DOM_LIST then
            g:= linalg::ogCoordTab[g,hold(Scales)](op(x), c)
        end_if;
        gradient:= [diff(f,x[i])/g[i] $ i = 1..nops(x)]:
    else
        gradient:= [diff(f,x[i]) $ i=1..nops(x)]:
    end_if:

    // laplacian = divergence(gradient(f)):

    linalg::divergence(gradient, args(i) $ i=2..args(0));
end_proc:

linalg::laplacian := funcenv(linalg::laplacian):
linalg::laplacian::Content := (Out, data) ->
                              if nops(data) <> 2 then
                                return(Out::stdFunc(data));
                              else
                                Out::Capply(Out::Claplacian, 
                                            Out(op(data, 1)),
                                            map(op(op(data,2)), Out)):
                              end_if:
