/*
 matlinsolveLU( L,U,b ) -- solve the linear system LU*x=b

 L, U: nonsingular square matrices of category 'Cat::Matrix'
 b   : matrix of category 'Cat::Matrix'

 linalg::matlinsolveLU( L,U,b ) returns the solution of the
 linear system A*x=b, whereas A has the LU factorization A=LU.
 The algorithm first computes the solution of Ly=b by forward
 substitution and ends by computing Ux=y by backward substitution.

 If b is a matrix with more than one column then the result is
 a matrix X which columns satisfy the equation A*col(X,i)=col(b,i) 
 (i=1..ncols(B)).

 The decomposition of A has been computed with linalg::factorLU.
 Each diagonal entry of L must be equal to 1.

 The algorithm does not work for singular A, or for matrices
 not defined over Cat::Field's.
*/

linalg::matlinsolveLU:= proc(L,U,B)
    local i, j, k, R, nL, nB, tmp,
          Rmult, Rplus, Rsubtract, Rinvert, Riszero;
begin
    if testargs() then
        if L::dom::hasProp(Cat::Matrix) <> TRUE then
            error("expecting a square matrix of 'Cat::Matrix'");
        end_if;

        R:= L::dom::coeffRing;
        if not R::hasProp(Cat::Field) then
            error("expecting a matrix over a 'Cat::Field'");
        elif R <> U::dom::coeffRing or R <> B::dom::coeffRing then 
            error("component rings of the matrices differ")
        end_if
    else
        R:= L::dom::coeffRing
    end_if;
 
    nL:= L::dom::matdim(L)[2];
    nB:= B::dom::matdim(B)[2];
    if L::dom::matdim(L)[1] <> nL then error("not a square matrix") end_if; 
    if B::dom::matdim(B)[1] <> nL then error("incompatible dimensions") end_if;

    Rmult:= R::_mult;
    Rplus:= R::_plus;
    Rsubtract:= R::_subtract;
    Rinvert:= R::_invert;
    Riszero:= R::iszero;

    for k from 1 to nB do
        // solve Ly = col(B,k) by forward substitution:
        for i from 2 to nL do
            B[i,k]:= Rsubtract( B[i,k], Rplus(Rmult(L[i,j],B[j,k]) $ j=1..i-1) )
        end_for;

        // solve Ux = y by backward substitution:
        for i from nL downto 1 do 
            if Riszero( U[i,i] ) then
                error("expecting a nonsingular matrix")
            else
                tmp:= Rinvert( U[i,i] );
                B[i,k]:= Rmult(
                   tmp, Rsubtract( B[i,k],Rplus(Rmult(U[i,j],B[j,k]) $ j=i+1..nL) )
               )
            end_if
        end_for
    end_for;

    return(B)
end_proc:

