

/* otimes -- a special multiplication for multivariate polynomials 
Author: Walter Oevel 
Date:   May 2009

Call:  linalg::otimes_mu(P, PP, i, k)

   P, PP are multivariate polynomial expressions depending 
   on the variables #Z[i], #Z[i+1], .. , #Z[k]. It is the
   product P*PP in which only the 0th and 1st order terms 
   of the variables are present. 

REFERENCE:

   Modified Gauss Algorithm for Matrices with Symbolic Entries,
   B. Fuchssteiner, ACM Communications in Computer Algebra, 42(3),
   3. September 2008

   Fuchssteiner's work is an enhanced version of:

   Efficient Gaussian Elimination Method for Symbolic
   Determinants and Linear Systems, T. Sasaki, H. Murao,
   ACM Transactions on Mathematical Software, Vol. 8,
   No. 3, September 1982, Pages 228-289.
*/

linalg::otimes_Fu:= proc(P, PP, i, k)
local P0, P1, PP0, PP1, Z;
begin
  if iszero(P) then
     return(0);
  elif iszero(PP) then
     return(0);
  end_if;
  if i <= k then 
    P0 := coeff(P, #Z[i], 0);
    PP0:= coeff(PP,#Z[i], 0);
    P1 := coeff(P, #Z[i], 1);
    PP1:= coeff(PP,#Z[i], 1);
    Z:= #Z[i];
    if iszero(P0) then
       return(linalg::otimes_Fu(P1, PP0, i+1, k)*Z);
    elif iszero(PP0) then
       return(linalg::otimes_Fu(P0, PP1, i+1, k)*Z);
    elif iszero(P1) then
       return(linalg::otimes_Fu(P0, PP0, i+1, k)+
              linalg::otimes_Fu(P0, PP1, i+1, k)*Z);
    elif iszero(PP1) then
       return(linalg::otimes_Fu(P0, PP0, i+1, k) 
             +linalg::otimes_Fu(P1, PP0, i+1, k)*Z);
    else // the full monty
       return(linalg::otimes_Fu(P0, PP0, i+1, k) 
             +linalg::otimes_Fu(P1, PP0, i+1, k)*Z
             +linalg::otimes_Fu(P0, PP1, i+1, k)*Z);
    end_if;
  else // i > k, P = P0, PP = PP0 do not depend on #Z[i]
    assert(domtype(P*PP) = DOM_POLY);
    return(op(P * PP, 1))
  end_if;
end_proc:
