//    
// stefanw, 6.3. 1998 #

/*
    linalg::pseudoInverse -- compute the Moore-Penrose inverse

    pseudoInverse(A)

    A - any matrix over the expressions or complex numbers


        Source: Boullion/ Odell, Generalized Inverse Matrices,
        Wiley-Interscience, 1971, page 74/75.
*/

linalg::pseudoInverse:=
proc(A)
    local a, AA, Astar, i, n, ak, bk, ck, dk, star, R, Rnormal;
begin
    if testargs() then
        if args(0) <> 1 then 
           error("expecting one argument, a matrix of 'Cat::Matrix'")
        elif A::dom::hasProp(Cat::Matrix) <> TRUE then
           error("expecting matrix of 'Cat::Matrix'")
        elif not A::dom::coeffRing::hasProp(Cat::Field) then
           error("expecting matrix over a 'Cat::Field'")
        end_if
    end_if;

    //-----------------------------------------------
    // Call the corresponding numeric::pseudoInverse?
    // Note the symbolic code below does not use
    // any pivoting or other measures for numerical
    // stabiliziation:
    //-----------------------------------------------
    if linalg::checkForFloats(A) <> FALSE then
       return(numeric::pseudoInverse(A));
    end_if:

    //--------------
    // symbolic mode
    //--------------
    star:= AAA -> AAA::dom::transpose(conjugate(AAA));

    // If the rows of the matrix A are linearly independent and A contains 
    // symbolic components, we use A^T (A A^T)^(-1) to compute the 
    // pseudoinverse. 
    // If the inversion fails, the standard former implementation by 
    // Stefan Wehmeier is used. 
    if indets(A) <> {} then 
      AA:= (A*star(A))^(-1);
      if AA <> FAIL then 
        return(star(A) * AA)
      end_if;
    end_if;

    // compute initial partitions
    n:= A::dom::matdim(A)[2];
    AA:= [0 $ n-1 , A ];
    for i from n-1 downto 1 do
        // delete column such that AA[i] is the 
        // matrix consisting of the first i columns of A
        AA[i]:= A::dom::delCol(AA[i+1], i+1)
    end_for;
    a:=A::dom::col(A,1);
    if iszero(a) then
        Astar:= a::dom::transpose(a)
    else
        Astar:= ((star(a)*a)^(-1))*star(a);                
    end_if;
    R:= A::dom::coeffRing;
    if (Rnormal:= R::normal) = FAIL then
       if R::hasProp(Ax::systemRep) then
          Rnormal:= normal
       else
          Rnormal:= () -> args(1);
       end_if;
    end_if;

    for i from 2 to n do
        // compute the pseudoinverse of AA[i]:
        userinfo(2, "Computing pseudo-inverse of submatrix ");
        userinfo(2, "consisting of the first ".expr2text(i)." columns");
        ak:=A::dom::col(A,i);
        dk:=Astar*ak;	
        ck:=ak - AA[i-1]*dk;
        if iszero(Rnormal(ck)) then
            bk:= (1/(1+linalg::scalarProduct(dk, dk)))*star(dk)*Astar
        else
            // compute pseudoinverse of ck:
            bk:=(star(ck) * ck)^(-1)*star(ck);
	    if bk = FAIL then 
	      return(FAIL);
	    end_if;
        end_if;
        Astar:= Astar::dom::stackMatrix( Astar - dk*bk, bk)
    end_for;

    return( Astar )
end_proc:
