/*
       linalg::sylvester -- computes the Sylvester matrix of two polynomials

       sylvester(f,g,[x])

       sylvester(f,g) computes the Sylvester matrix of f and g.
       The polynomials f and g must be univariate polynomials
       both of category Cat::Polynomial(R) or both of type DOM_POLY.

       sylvester(f,g,x) computes the Sylvester matrix of f and g
       in respect to the indeterminate x.
       f and g must be both of category Cat::Polynomial or both of 
       type DOM_EXPR or DOM_POLY.

       If the polynomials are univariate then the coefficient ring 
       of the Sylvester matrix is R where R is the coefficient ring 
       of the polynomials. Are the polynomials of type DOM_POLY 
       and their coefficient ring is the ring Expr, the coefficient 
       ring of the Sylvester matrix is Dom::ExpressionField(). 
       Are the polynomials multivariate then the coefficient domain 
       of the Sylvester matrix is the ring 
       Dom::DistributedPolynomial(ind,R), where ind is a list of all
       the unknowns of f and g without the unknown x.
*/

linalg::sylvester := proc(f,g)
    local cf, cg, m, n, ffcn, gfcn, R, X, i, j;
begin
    if testargs() then
        if args(0) < 2 or args(0) > 3 then error("expecting 2 or 3 arguments") end_if;

        case args(0)
        of 2 do
            if domtype(f) = DOM_POLY then
                if domtype(g) <> DOM_POLY then
                    error("types of the polynomials differ") 
                elif op(f,3) <> op(g,3) then
                    error("expecting polynomials over the same coefficient ring")
                end_if;

                X:= op(f,2);
                if X <> op(g,2) then
                    error("indeterminates of the polynomials differ")
                elif nops(X) <> 1 then
                    error("expecting two univariate polynomials, or specify an indeterminate")
                else
                    X:= X[1]
                end_if
            elif f::dom::hasProp( Cat::Polynomial ) = TRUE then 
                if g::dom::hasProp( Cat::Polynomial ) <> TRUE then
                    error("types of the polynomials differ")
                elif f::dom::coeffRing <> g::dom::coeffRing then
                    error("expecting polynomials over the same coefficient ring")
                end_if;
                X:= f::dom::mainvar(f);
                if X <> g::dom::mainvar(g) then
                    error("the main variable of the polynomials differ")
                end_if
            else
                error("expecting polynomials")
            end_if;
            break
        of 3 do
            X:= args(3);
            if poly(X) = FAIL then
                error("illegal indeterminate")
            end_if;
            if domtype(f) = DOM_POLY then
                if domtype(g) <> DOM_POLY then
                    error("types of the polynomials differ")
                elif op(f,3) <> op(g,3) then
                    error("expecting polynomials over the same coefficient ring")
                end_if
            elif f::dom::hasProp( Cat::Polynomial ) = TRUE then
                if g::dom::hasProp( Cat::Polynomial ) <> TRUE then
                    error("types of the polynomials differ")
                elif f::dom::coeffRing <> g::dom::coeffRing then
                    error("expecting polynomials over the same coefficient ring")
                end_if
            else
                 f:= poly(f,[X]); g:= poly(g,[X]);
                 if f = FAIL or g = FAIL then
                   error("unable to convert arguments to polynomials")
                 end_if
            end_if
        end_case
    elif args(0) = 2 then
        if domtype(f) = DOM_POLY then X:= op( op(f,2),1 ) 
        else X:= f::dom::mainvar(f)
        end_if
    else
        X:= args(3)
    end_if;

    if f::dom::hasProp( Cat::Polynomial ) = TRUE then
        m:= f::dom::degree(f,X); n:= g::dom::degree(g,X);
        if m+n = 0 then 
            error("both polynomials have degree 0")
        end_if;

    cf:= f::dom::indets(f); cg:= g::dom::indets(g);
    R:= f::dom::coeffRing;
    if nops(cf) > 1 or nops(cg) > 1 then
      if (j:= contains(cf,X)) > 0 then delete cf[j] end_if;
        for j in cg do
           if contains(cf,j) = 0 then cf:= cf.[j] end_if
        end_for;
        R:= Dom::DistributedPolynomial( cf,R )
      end_if;
      cf:= f::dom::coeff(f,X,m-i) $ i=0..m;
      cg:= g::dom::coeff(g,X,n-i) $ i=0..n
    else
    if domtype(f) <> DOM_POLY then f := poly(f,[X]) end_if;
        if domtype(g) <> DOM_POLY then g := poly(g,[X]) end_if;
             m:= degree(f,X); n:= degree(g,X);
        if m+n = 0 then              
            error("both polynomials have degree 0")
        end_if;

        R:= op(f,3);
        if R = Expr then
            R:= Dom::ExpressionField()
        elif type(R) = "function" and op(R,0) = hold(IntMod) then
            R:= Dom::IntegerMod( op(R) )
        elif R::hasProp( Cat::CommutativeRing ) <> TRUE then
            error("illegal coefficient ring")
        end_if;

        cf:= op(f,2); cg:= op(g,2);
        if nops(cf) > 1 or nops(cg) > 1 then 
            if (j := contains(cf,X)) > 0 then delete cf[j] end_if;
            for j in cg do
                if contains(cf,j) = 0 and j <> X then cf:= cf.[j] end_if 
            end_for;
            R:= Dom::DistributedPolynomial( cf,R )
        end_if;
        cf:= coeff(f,X,m-i) $ i=0..m;
        cg:= coeff(g,X,n-i) $ i=0..n
    end_if;

    ffcn:= i -> [ 0 $ i-1, cf, 0 $ n-i ];
    gfcn:= i -> [ 0 $ i-1, cg, 0 $ m-i ];

    (Dom::Matrix(R))::coerce(
      array( 1..n+m,1..n+m,[ ffcn(i) $ i=1..n, gfcn(i) $ i=1..m ] )
    )
end_proc:

