/*++
 linalg::symbolicRank(M) -- computes the rank of the matrix M

 Parameters: M -- a square matrix

 This routine is meant for matrices with symbolic parameters.
 It replaces different parameters by different random integers
 and computes the rank of the resulting matrix that contains
 only rationals.
++*/

linalg::symbolicRank := proc(M)
local symbols, allsymbols, boundsymbols,
      mypower, r, rr, randnumb, x, p, pI,
      f, constructor, dim;
save SEED;
begin
  if not contains({matrix, densematrix}, M::dom) then
     return(FAIL);
  end_if;
  SEED:= 1:
  r:= random(1..10^6); 
  rr:= proc() 
       option remember; 
       begin 
         r():
       end_proc:

  // Look for a good prime modulus p to do modular arithmetic with.
  // 
  // 1) We wish to have solutions x of 
  //      x^n = a mod p   
  //    for as many pairs (n, a) as possible. Taking logarithms,
  //    this boils down do have solutions y of
  //      n*y = b mod p 
  //    for as many pairs (n, b) as possible. There is a solution
  //    for any b iff gcd(n, p-1) = 1 (little Fermat). Thus, we
  //    should choose p such that p - 1 has as few prime factors 
  //    as possible. Since p-1 is even, the best we can do is
  //    to choose
  //       p = 2*q + 1      (#)
  //    with some prime number q.
  // 2) Further, we wish to replace I be sqrt(-1) mod p. The
  //    square root of -1 exists iff p mod 4 = 1, i.e. iff
  //       p = 4*q + 1      (##)
  //    with some prime number q.
  //    By satisfying (##), we satisfy (#), too (with the 
  //    prime q replaced by 2*prime). Searching for such a 
  //    prime p via
  /*
     p:= 10^11: 
     for i from 0 to 10^2 do 
       p:= nextprime(p+1): 
       if (p-1) mod 4 = 0 and isprime( (p-1) div 4) then 
        print("p" = p, "q" = (p-1) div 4); 
       end_if:
     end_for;
  */
  // we find p = 100000001909 = 4*25000000477 + 1.
  /* Here is a more sophisticated search for p = 24*prime + 1
     where p is such that sqrt(ithprime(j)) mod p exists for
     as many j = 1, 2, ... as possible:
  p:= 10^11:
  for i from 0 to 10^7 do
   p:= nextprime(p+1):
   if (p-1) mod 4 = 0 and
      (p-1) mod 24 = 0 and
      isprime( (p-1) div 24) then
        donext:= FALSE;
        for j from 1 to 30 do 
           if j > 15 then
             print(p);
           end_if;
           if numlib::mroots(poly(#x^2 - ithprime(j), [#x]), p) = [] then
              donext:= TRUE;
              break;
           end_if;
        end_for:
        if donext then 
           next;
        end_if;
        print(i, p, (p-1) div 24);
   end_if:
  end_for;

  We find: p = 102015168649 = 24*4250632027 + 1
  */

  p:= 102015168649; // the modulus for modular arithmetic
                    // We have p = 24*4250632027 + 1
                    // and sqrt(q) mod p exists for q = ithprime(1), .. , ithprime(18) 
  pI:=  5040444323: // = numlib::mroots(poly(#x^2 + 1, [#x]), p), 
                    // i.e., pI^2 mod p = - 1,
                    // i.e., pI = sqrt(-1) mod p.
 
  mypower:= proc(x, n) 
            local d, root;
            begin
             if (not contains({DOM_INT, DOM_RAT}, domtype(x))) or
                (not contains({DOM_INT, DOM_RAT}, domtype(n))) then
                return(#power(args()));
             end_if;
             // Now, both x and n are integers/rationals.
             if domtype(n) = DOM_INT then 
                return(powermod(x, n, p)): 
             else
                [n, d]:= [op(n)] ;
                // if x^(1/d) mod p exists, choose one of the possible
                // modular roots. Thus, x^(2/d), x^(3/d) etc. are 
                // identified as powers of x^(1/d) via modular arithmetic.
                // However, to identify sqrt(2)*sqrt(3) with sqrt(6), we
                // would need a consistent choice of the 2 possible square 
                // roots for each number.
                root:= numlib::mroots(poly(#x^d - x, [#x]), p);
                if root <> [] then
                   return(powermod(root[1], n, p));
                else // x^(n/d)
                   // We identify x^(n/d) as (x^(1/d))^n:
                   return(powermod(rr(x^(1/d)), n, p));
                end_if;
             end_if:
            end_proc:
  constructor:= M::dom::constructor;
  dim:= op(M::dom::matdim(M));
  M:= expr([op(M)]);

  symbols:= indets(M);
  boundsymbols:= symbols minus freeIndets(M);
  allsymbols:= indets(M, All) minus {hold(_plus), hold(_mult)};

  // First step: replace the function names in symbolic function 
  // calls such as sin(1) or fact(n) or binomial(123, x) or 
  // transform::fourier(x^n, x, y) by rational numbers. Later, 
  // we will turn the arguments into rationals and we do not 
  // want to make binomial or transform::fourier raise an error 
  // when all their arguments are replaced by rationals.
  for f in allsymbols minus symbols do 
//  if f = hold(exp) then 
//     // the following replacement exp(x) --> randomnumber^x 
//     // satisfies the functional law exp(x+y) = exp(x)*exp(y):
//     M:= subs(M, f = (x -> mypower(rr(exp), x)), Unsimplified);
//  elif f = hold(_power) then
    if f = hold(_power) then
       M:= subs(M, f = mypower, Unsimplified);
    else // for all other types:
       M:= subs(M, f = r() + (() -> (#sum(args()))), Unsimplified):
    end_if;
  end_for;
  M:= eval(M):

  // Now, all function names should have been replaced by
  // random numbers. The only exceptions are  "_power" (was 
  // replaced by #power via mypower) and #sum. 
  // We are ready to descend into the arguments and replace
  // all remaining symbols by random integers:

  if boundsymbols <> {} then 
     // replace all bound symbols by the **same** random number
     // in order to identify int(f(x),x=0..1) and int(f(y),y=0..1)
     randnumb:= r():
     M:= subs(M, [x = randnumb $ x in boundsymbols], EvalChanges);
  end_if;

  // replace all other identifiers by random numbers 
  M:= subs(M, [x = rr(x) $ x in symbols], EvalChanges);

//if has(M, I) then
//   M:= subs(M, I = pI, EvalChanges);
//end_if:

  // Finally, calls of #power and #sum with rational/integer arguments
  // should have survived. Rewrite them to rationals:
  if has(M, #power) or 
     has(M, #sum) then
    M:= subs(M, [#power = mypower,
                 #sum = proc() 
                    local i, s; 
                  begin
                    s:= 0:
                    for i from 1 to args(0) do  
                      if contains({DOM_INT, DOM_RAT}, domtype(args(i))) then
                         // Multiply by i to avoid f(x, y) being identified 
                         // with f(y, x):
                         s:= s + i*args(i)
                      else
                         s:= s + rr(args(i));
                      end_if;
                    end_for;
                    return(s);
                 end_proc], EvalChanges);
  end_if;

  // Do a global eval:
  M:= eval(M);

  // Symbolic elements (domain elements) may have survived.
  // Get rid of them via rationalize:
  M:= rationalize(M, 
                  ApproximateFloats = TRUE,
                  StopOn = {DOM_INT, DOM_IDENT, DOM_RAT, DOM_COMPLEX}
                  )[1];

  symbols:= indets(M):
  M:= subs(M, [x = rr(x) $ x in symbols], EvalChanges);

  if has(M, I) then
     M:= subs(M, I = pI);
  end_if:
  M:= constructor(Dom::IntegerMod(p))(dim, M);
  return(op(M::dom::gaussElim(M), 2));
end_proc:
