// W.Oevel, 18/06/02 

/*++
    toeplitz - Toeplitz matrices

    Call:

    linalg::toeplitz(  n,  [a[-k],..,a[0],..,a[k]] <, CoefficientDomain> );
    linalg::toeplitz(m, n, [a[-k],..,a[0],..,a[k]] <, CoefficientDomain> );
                                                                   
    Parameter:

    m, n  --  the matrix dimension, positve integers 
    a[i]  --  arithmetical expressions.
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The m x n -Toeplitz matrix T given by 

           | a[ 0]  a[ 1]  ...   a[k]                    0   |
           | a[-1]  a[ 0] a[ 1]  ...  a[k]                   |
           |  ...   a[-1] a[ 0] a[ 1] ...  a[k]              |
       T = | a[-k]  ...   a[-1] a[ 0] a[1] ...  a[k]         |
           |        a[-k]  ...  a[-1] a[0] a[1] ... a[k]     |
           |               ...   ...  ...  ...  ... ... ...  |

    is returned by toeplitz(m, n, [[a[-k], ..., a[k]]). 
    It is of domain type Dom::Matrix().

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    linalg::toeplitz(n, [a[-k],..,a[0],..,a[k]], ..) is
    equivalent to
    linalg::toeplitz(n, n, [a[-k],..,a[0],..,a[k]], ..)

    Examples:

3.0.0 > linalg::toeplitz(4, [1, 2, 3, 0, 4, 5, 6])

                      +-            -+
                      |  0, 4, 5, 6  |
                      |              |
                      |  3, 0, 4, 5  |
                      |              |
                      |  2, 3, 0, 4  |
                      |              |
                      |  1, 2, 3, 0  |
                      +-            -+

    See also:  linalg::invtoeplitz, Dom::Matrix
++*/

linalg::toeplitz:= proc()
   local m, n, a, R, k;
begin
    if args(0) < 2 then 
       error("expecting at least 2 arguments") 
    end_if;
    if args(0) = 2 then
       [m, n, a, R]:= [args(1), args(1), args(2), Dom::ExpressionField()]:
    end_if;
    if args(0) = 3 then
       if domtype(args(2)) = DOM_LIST then
            [m, n, a, R]:= [args(1), args(1), args(2), args(3)]:
       else [m, n, a, R]:= [args(1), args(2), args(3), Dom::ExpressionField()]:
       end_if;
    end_if;
    if args(0) = 4 then
       [m, n, a, R]:= [args(1), args(2), args(3), args(4)]:
    end_if;
    if args(0) > 4 then 
       error("too many arguments") 
    end_if;

    if not testtype(m, Type::PosInt ) then 
       error("1st argument: expecting a positive integer (the row dimension)") 
    end_if;
    if not testtype(m, Type::PosInt ) then 
       error("2nd argument: expecting a positive integer (the row dimension)") 
    end_if;
    if not testtype(a, DOM_LIST ) then 
       error("expecting a list of entries for the diagonals");
    end_if;
    if (k:= nops(a)) = 0 then
       error("expecting a non-empty list of entries for the diagonals");
    end_if;
    if not testtype(k, Type::Odd) then
       error("expecting an odd number of entries for the diagonals");
    end_if;
    k:= (k-1)/2:
    if k>m-1 then
       if m = n then
            error("the number of entries for the diagonals ".
                  "exceeds the actual number of diagonals");
       else error("the number of entries for the lower diagonals ".
                  "exceeds the actual number of lower diagonals");
       end_if;
    end_if;
    if k>n-1 then
       error("the number of entries for the upper diagonals ".
             "exceeds the actual number of upper diagonals");
    end_if;
    if domtype(R) <> DOM_DOMAIN then
       error("the component ring should be specified by a domain ".
             "of type 'DOM_DOMAIN'. Received: ".expr2text(R));
    end_if;
    if R::hasProp(Cat::Rng) <> TRUE then 
       error("the component domain must be of category 'Cat::Rng'")
    end_if;

/* ---------------------------------------------------------------------
   //-------------------------------------------------------------------
   // Remark (valid when Dom::Matrix still had a dense internal representation)
   // We could implement toeplitz via
   // linalg::toeplitz:= (m, n, a, R) -> Dom::Matrix(R)(m, n, a, Banded).
   // However, the following code is about 8 times faster
   // than Dom::Matrix(R)::new.
   //-------------------------------------------------------------------
   //----------------------------------------------------------
   A:= array(1..m,1..n, [[0 $ n] $ m]);
   for d from -k to k do
      Ad:= a[d + k + 1];
      if iszero(Ad) then next end_if;
      if d < 0 then
         for i from 1 to min(d + m, n) do
             A[i - d, i] := Ad;
         end_for;
      end_if;
      if d = 0 then
         for i from 1 to min(m, n) do
             A[i, i] := Ad;
         end_for;
      end_if;
      if d > 0 then
         for i from 1 to min(m, n - d) do
             A[i, d + i] := Ad;
         end_for;
      end_if;
   end_for:
   //----------------------------------------------------------
   case R
   of Dom::Complex do
   of Dom::Real do
   of Dom::Numerical do
   of Dom::Rational do
   of Dom::ExpressionField() do
      // conversion of the components is not necessary!
      return((Dom::Matrix(R))::create( A ))
   otherwise
      return(Dom::Matrix(R)( A ))
   end_case
----------------------------------------------------------------------*/
   Dom::Matrix(R)(m, n, a, Banded):
end_proc: 
