//   
// walter, 25/09/98

/*
  toeplitzSolve - solve a banded Toeplitz system

  linalg::toeplitzSolve([a[-k],..,a[0], a[k]], y )

  Parameters:

    a[i]       -- the toeplitz nodes: 
                  arithmetical expressions
                  or domain elements
    y          -- a vector or list [y[1],..,y[n]]
                  of arithmetical expressions 
                  or domain elements
  Synopsis:

  Let T be Toeplitz matrix generated by [a[-k],..,c[k]]:
            ( a[ 0]  a[ 1 ]  ..  a[k]               0   ) 
        T = ( a[-1]  a[ 0 ]  ..  ..   a[k]              )
            ( ..  ..   ..    ..  ..   ..                )
            ( a[-k]  a[-k+1] ..  a[0] ..            ..  )
            (        a[-k]  a[-k+1].. a[0]   ..     ..  )
            (               ..    ..   ..  ..  ..   a[1])
            (  0                 a[-k] ..  .. a[-1] a[0])

  toeplitzSolve solves T*x = y and returns x.

  Return Value: a vector or list [x[1],..,x[n]]. The return type
                coincides with the type of the input vector/list y


  The call 

     linalg::toeplitzSolve(n, [a[-k],..,a[k], [y[1], .., y[n]])

  returns the vector or list [x[1],..,x[n]] representing the solution of
  the linear system 
            sum(a[j - i]*x[j], j=1..n) = y[i], i=1..n.
  (where a[j-i] = 0 for |j-i| > k)

  The implemented Levinson algorithm needs O(n^2) operations.
  See Press et al, Numerical Recipes, Chapter 2.8

  Warning: the Levinson algorithms requires that all principal
  minors of the Toeplitz matrices are non-zero.
  Thus, toeplitzSolve might return FAIL even if the Toeplitz-Matrix
  is invertible!


  Example:
  >> A := [-2, -1, 5, 1, 2]:
  >> y := [1, 2, 3, 4, 5, 6, 7]:
  >> // We are solving
  >> B:= linalg::toeplitz(nops(y), nops(y), A):
  >> c:= matrix(y):
  >> B, "* x", "=", c
     +-                           -+              +-   -+
     |   5,  1,  2,  0,  0,  0, 0  |              |  1  |
     |                             |              |     |
     |  -1,  5,  1,  2,  0,  0, 0  |              |  2  |
     |                             |              |     |
     |  -2, -1,  5,  1,  2,  0, 0  |              |  3  |
     |                             |              |     |
     |   0, -2, -1,  5,  1,  2, 0  |, "* x", "=", |  4  |
     |                             |              |     |
     |   0,  0, -2, -1,  5,  1, 2  |              |  5  |
     |                             |              |     |
     |   0,  0,  0, -2, -1,  5, 1  |              |  6  |
     |                             |              |     |
     |   0,  0,  0,  0, -2, -1, 5  |              |  7  |
     +-                           -+              +-   -+
  >> linalg::toeplitzSolve(A, y)

        [239/35517, 5128/35517, 14597/35517, 15518/35517, 11827/35517,

         38752/35517, 20735/11839]

  >> linalg::toeplitzSolve(A, c)

                       +-             -+
                       |   239/35517   |
                       |               |
                       |   5128/35517  |
                       |               |
                       |  14597/35517  |
                       |               |
                       |  15518/35517  |
                       |               |
                       |  11827/35517  |
                       |               |
                       |  38752/35517  |
                       |               |
                       |  20735/11839  |
                       +-             -+
*/

linalg::toeplitzSolve:= proc(a,y)
local n, mn, R, k, m, j, listinput, dtype,
      x, g, h, sxn, sgn, shn, sgd, sd,
      m1, m2, J, gg, hh, gt1, gt2, ht1, ht2,
      Rzero ;
begin 
  if args(0)<2 or args(0)>3 then error("wrong number of arguments") end_if;
  if not testtype(a, linalg::vectorOf(Type::AnyType) ) and
     not testtype(a, DOM_LIST ) then
       error("the first argument should be a vector of category 'Cat::Matrix' ".
             "or a list");
  end_if;
  if not testtype(a, DOM_LIST) then
     R:= a::dom::coeffRing:
     if (R = FAIL and not testtype(y, DOM_LIST)) and
        (R <> FAIL and not testtype(y, linalg::vectorOf(R))) then
            error("the second argument should be a vector of same type as the first argument ".
                  "or a list");
       end_if;
  end_if;

 //----------------------------------------------------
 // convert vectors (matrices) a, y to lists for faster
 // indexed reading and writing.
 // Remember the dimension of y (can be 1 x n or n x 1)
 // and the coefficient ring.
 //----------------------------------------------------
  if domtype(a) <> DOM_LIST then
     a:= [op(a)]:
  end_if;
  if domtype(y) <> DOM_LIST then
     listinput:= FALSE:
     mn:= op(y::dom::matdim(y)): 
     R:= y::dom::coeffRing;
     dtype:= domtype(y): // Dom::Matrix(R) or Dom::DenseMatrix(R)
     y:= [op(y)]:
  else
     listinput:= TRUE:
     if nops(y) = 0 then 
        error("empty right hand side vector");
     end_if;
     R:= (y[1])::dom;
  end_if;

  //----------------------------------------------------
  // determine dimension n and number of bands 2*k + 1
  //----------------------------------------------------
  k:= nops(a);
  if k mod 2 <> 1 then
     error("expecting an odd number of Toeplitz entries");
  end_if;
  k:= (k - 1)/2:   // k = number of bands above/below diagonal
  n:= nops(y);
  if k > n then 
     error("too many bands in the Toeplitz matrix");
  end_if;

  //----------------------------------------------------
  // for convenience, convert list of Toeplitz entries
  // to a table indexed from -k to k:
  //----------------------------------------------------
  a:= table( (-k-1 + j = a[j]) $ j=1..(2*k +1) );
  // now, the Toeplitz entries are a[-k], .., a[0], .. , a[k]

  //-------------------------------------------------------
  // Levinson's boundary method for solving (non-symmetric) 
  // Toeplitz systems. The implementation follows
  // Numerical Recipes, Chapter 2.8
  //-------------------------------------------------------

  if iszero(a[0]) then // The Levinson algorithm cannot proceed.
     return(FAIL);     // This does not necessarily mean that
  end_if;              // the Toeplitz matrix is singular!
  
  if R::zero <> FAIL then
     Rzero:= R::zero
  else
     Rzero:= 0;
  end_if;
  g:= [Rzero $ n];
  h:= [Rzero $ n];
  x:= [Rzero $ n];
  x[1]:= y[1]/a[0];
  if n <> 1 then
     if k = 0 then
        g[1]:= Rzero;  // Need to convert 0 to the field
        h[1]:= Rzero:  // over which y is defined?!?
     else
        g[1]:= a[ 1]/a[0];
        h[1]:= a[-1]/a[0];
     end_if;
     for m from 1 to n do
         m1:= m+1;
         sxn:= -y[m1];
         sd:=  -a[0];
         for j from max(1, m1 - k) to m do
           J:= m1-j;
           sxn:= sxn + a[-J]*x[j];
           sd:=  sd  + a[-J]*g[J];
         end_for;
         if iszero(sd) then // The Levinson algorithm cannot proceed.
            return(FAIL);   // This does not necessarily mean that
         end_if;            // the Toeplitz matrix is singular!
         x[m1]:= sxn/sd;
         for j from 1 to m do
             x[j] := x[j] - x[m1]*g[m1 - j];
         end_for;
         if m1 = n then
            break;  // We are done. Go to 'return' below
         end_if;

         //--------------------------------------
         // prepare for the next step m -> m+1.
         // We need to update the list g:
         // Up to now: g = [g[1], .. , g[m], 0, .., 0]
         // Transform into
         // g' = [g'[1], .. , g'[m+1], 0, .., 0]
         //--------------------------------------
         if m1 > k then
           sgn:= Rzero;  // Need to convert 0 to the field
           shn:= Rzero;  // over which y is defined?!?
         else
           sgn:= -a[ m1];
           shn:= -a[-m1];
         end_if;
         sgd:= -a[0];
         for j from max(1, m1 - k) to m do
           J:= m1 - j;
           sgn:= sgn + a[ J]*g[j];
           sgd:= sgd + a[ J]*h[J];
           shn:= shn + a[-J]*h[j];
         end_for;
         if iszero(sgd) then // The Levinson algorithm cannot proceed.
            return(FAIL);    // This does not necessarily mean that
         end_if;             // the Toeplitz matrix is singular!
         g[m1]:= sgn/sgd;
         h[m1]:= shn/sd;
         gg:= g[m1];
         hh:= h[m1];
         if iszero(gg) and iszero(hh) then
            next;
         end_if;
         m2:= m1 div 2:
         for j from 1 to m2 do
           J:= m1 - j;
           gt1:= g[j];
           gt2:= g[J];
           ht1:= h[j];
           ht2:= h[J];
           g[j]:= gt1 - gg*ht2;
           g[J]:= gt2 - gg*ht1;
           h[j]:= ht1 - hh*gt2;
           h[J]:= ht2 - hh*gt1;
         end_for; // for j from 1 to m2 do
     end_for; // for m from 1 to n do
  end_if; // of n <> 1

  //-----------------------------------------------------
  // convert the solution vector x to the type of the
  // input vector y (a list, a matrix or a sparse matrix)
  //-----------------------------------------------------
  if listinput then
       return(x)
  else return(dtype(mn, x));
  end_if:
end_proc:

