// W.Oevel, 18/06/02 

/*++
    vandermonde - Vandermonde matrices

    Call:

    linalg::vandermonde(v <, CoefficientDomain> );
                                                                   
    Parameter:

    v  --  a vector or list v = [v[1], .. , v[n]]
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n Vandermonde matrix V is given by 
    V[i,j]= v[i]^(j-1). Its inverse is computed 
    by invvandermonde(v) with a runtime of O(n^2).

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    Examples:

3.0.0 > linalg::vandermonde([v1, v2]), 
        normal(linalg::invvandermonde([v1, v2]))

                         +-                      -+
                         |      v2          v1    |
            +-       -+  |   -------,  - -------  |
            |  1, v1  |  |   v2 - v1     v2 - v1  |
            |         |, |                        |
            |  1, v2  |  |       1         1      |
            +-       -+  |  - -------,  -------   |
                         |    v2 - v1   v2 - v1   |
                         +-                      -+

    See also:  linalg::invvandermonde, Dom::Matrix

++*/

linalg::vandermonde:= proc(v, R=Dom::ExpressionField())
   local RR, n, a, i, j;
begin
   if testargs() then
       if args(0) = 0 then 
          error("expecting at least one argument (a list of nodes)");
       end_if;
       if args(0) > 2 then 
          error("too many arguments") 
       end_if;
       if not testtype(v,linalg::vectorOf(Type::AnyType) ) and
          not testtype(v, DOM_LIST ) then
            error("the first argument should be a vector of category 'Cat::Matrix' ".
                  "or a list");
       end_if;
       if R::hasProp(Cat::Rng) <> TRUE then 
          error("the component domain must be of category 'Cat::Rng'")
       end_if;
   end_if;

   if domtype(v) <> DOM_LIST then
      RR:= v::dom::coeffRing; // coeff ring of the input data
      v:= [op(v)]:
      if args(0) = 1 then // no ring R specified by the user
          R:= RR;  // output ring = input ring
      end_if;
   else
      RR:= Dom::ExpressionField();
   end_if;

   n:= nops(v);

   /* ----------------------------------------
      vandermonde[i, j] = a[i,j] = v[i]^(j-1)
   -----------------------------------------*/
   a:= array(1..n, 1..n):
   for i from 1 to n do
    a[i, 1]:= RR::one:
    for j from 2 to n do
      a[i, j]:= v[i]*a[i, j-1];
    end_for:
   end_for:
   //-------------------------------------------------------
   // Conversion of the array  to a suitable matrix
   //-------------------------------------------------------
   if R = RR then // output type = input type
        // conversion of the components is not necessary!
        return((Dom::Matrix(R))::create( a ))
   else return(Dom::Matrix(R)( a ))
   end_if;
end_proc:
