
/*

linopt::standardForm - Convert a system of linear inequalities into standard form
	
A system of linear inequalities is in standard form when all the inequalities are 
of the form <= . The left hand side contain no constant, the right hand side 
contain a constant only.

Calling sequence:

linopt::standardForm( LinCon <, opt1> )

Parameter:

LinCon - a set or list of linear constraints. A linear constraint 
         can be an equation or a non-strict inequality.
opt1   - (optional) name NonNegative or a set of variables.
         NonNegative: All variables are assumed to be non-negative.
	 The variables of the set are assumed to be non-negative.

Return Value:

A set or list (depends on the type of the input variable LinCon) 
of linear inequalities in standard form. If no linear conditions
are given (LinCon is the empty list or empty set), the result is
just an empty list or empty set.

Note:

No simplifications are made. For example the two constrains x<=0 and
x <= -5 could be simplified to one constrain, namely x <= -5. 
This functions does not check if the system is feasible. The 
simplication could detect simple cases, for example x<=0 and x>=3. 

Example:

Constraints:

>> c := {3*x+4*y-23 >= 3*z, 5*x-4*y-3*z >= 10, x=5, y<=2, -y+2>=0}:

>> linopt::standardForm(c);

   {y <= 2, 3*z - 4*y - 3*x <= -23, 4*y - 5*x + 3*z <= -10, x <= 5, -x <= -5}

>> c := [3*x<=4*y, 2 >= x+y]: linopt::standardForm(c, NonNegative)

   [3*x - 4*y <= 0, x + y <= 2, -y <= 0, -x <= 0]

*/

linopt::standardForm := proc(LinCon, cond)
   local c, con, f, v, get_constant, new_set, non_neg, set;
begin

   // Definition of the local procedure get_constant.

   // Return the constant term of the linear function f
 
   get_constant := proc(f)
      local c;
   begin 
      case type(f)
        of "_mult" do
           return(0);
        of "_plus" do
           // if the sum contains a constant, then it is the last op
           c := op(f, nops(f));
           return((if testtype(c, Type::Numeric) then c else 0 end_if))
        otherwise
           if testtype(f, Type::Numeric) then f else 0 end_if
      end_case
   end_proc:

   // main program

   case args(0)
    of 1 do
       non_neg := {};
       break;
    of 2 do
      if cond = hold(NonNegative) then
         non_neg := indets(LinCon)
      elif domtype(cond) = DOM_SET then
         non_neg := cond
      else
         error("Illegal option")
      end_if;
      break;
    otherwise
      error("wrong number of arguments")
   end_case;

   if testtype(LinCon, DOM_LIST) then
      if LinCon = [] then return( [] ) end_if;
      set := {op(LinCon)}
   else // LinCon is a set
      if LinCon = {} then return( {} ) end_if;
      set := LinCon
   end_if;

   if not testtype(set, Type::SetOf(Type::Relation)) then
      error("List or set of equations or non-strict inequalities expected.")
   end_if;

   if not testtype(non_neg, Type::SetOf(DOM_IDENT)) then
      error("2nd argument: NonNegative or set of identifiers expected.")
   end_if;

   if non_neg <> {} then
      new_set := map(non_neg, x -> (-x <= 0))
   else
      new_set := {}
   end_if;

   for con in set do
       f := op(con, 1) - op(con, 2);
       v := indets(f);
       if Type::Linear(f, [ op(v) ]) = FALSE then
          error("constraints must be linear.")  
       end_if; 

       case type(con) 
         of "_less" do
            error("Can't handle strict inequalities.")
         of "_unequal" do
            error("equations or non-strict inequalities expected.")
         of "_equal" do 
	    c := get_constant(f);
            new_set := new_set union { f - c <= -c, -f + c <= c };
            break;
         of "_leequal" do
            c := get_constant(f);
            new_set := new_set union { f - c <= -c };
            break;
       end_case
   end_for;
   
   if testtype(LinCon, DOM_LIST) then
      // convert the set into a list
      [ op(new_set) ]
   else
      new_set
   end_if

end_proc:

