// Lookup tables following the paper
// Searching Techniques for Integral Tables
// by T. H. Einwohner and Richard J. Fateman
// ISSAC '95

matchlib::einwohner :=
proc(ex, mainvar, tr, patterns, oldmainvar, ta, conds, results, allowedNewVars={})
  local cands, cand, res, pattern, cond, retval, s, substs_ret, inds;
begin
  substs_ret := null();
  if mainvar <> oldmainvar and has(ex, oldmainvar) then
    s := solvelib::getIdent(Any, indets(ex));
    ex := subs(ex, oldmainvar=s);
    substs_ret := substs_ret, s=oldmainvar;
  end_if;
  cands := matchlib::einwohner::candidates(ex, mainvar, tr, ta, oldmainvar);
  res := table();
  for cand in cands do
    pattern := patterns[cand[1]];
    cond := conds[cand[1]];
    retval := results[cand[1]];
    
    res := matchlib::einwohner::single_cand(pattern, cond, retval,
      cand[2], cand[3], ex, oldmainvar, mainvar, res);
  end_for;
  res := map([op(res)], op, 1);
  res := subs(res, substs_ret);
  res := prog::sort(res, l -> intlib::Simplify::defaultValuation(l[2]));
  // calling solve from lib code is hard and we do sometimes get bogus results.
  // filter those out where new identifiers appeared:
  inds := freeIndets(ex);
  res := select(res, r -> freeIndets(r) minus inds
    minus {mainvar, `#matchlib::block`, hold(infinity), hold(undefined)}
    minus allowedNewVars = {});
//  if res = [] then
//    FAIL
//  else
//    piecewise(op(res));
//  end_if;
  select(res, r -> is(r[1], Goal=FALSE)<>FALSE);
end_proc:

matchlib::einwohner := funcenv(matchlib::einwohner):

matchlib::einwohner::single_cand :=
proc(pattern, cond, retval, eqns, pat_conds, ex, oldmainvar, mainvar, res)
  local ids, exidents, conflicts, var, sol, sol_ex,
    s, s_ex, substs, substs_ex;
begin
  ids := freeIndets(map(eqns, op, 1));
  // same idents in replacements? if yes, replace by sth else
  exidents := freeIndets(map(eqns, op, 2));
  conflicts := (exidents intersect ids) union select(ids, property::hasprop);
  if conflicts <> {} then
    substs := table();
    for var in conflicts do
      substs[var] := solvelib::getIdent(Any,
        freeIndets(pattern) union {mainvar, oldmainvar} union
        freeIndets([op(res)]) union
        ids union exidents union map({op(substs)}, op, 2));
    end_for;
    // substitute in the pattern -- this has the advantage that
    // we don't need any substitutions in the result afterwards.
    // Since we substitute anyway, it's reasonable to undo our 
    // special `-a` identifiers right here. That saves some
    // ambiguities for the solver.
    for var in cond do
      if op(cond, 0) = hold(_equal) then
        if domtype(op(cond, 1)) = DOM_IDENT then
          substs[op(cond, 1)] := subs(op(cond, 2), substs);
        elif domtype(op(cond, 2)) = DOM_IDENT then
          substs[op(cond, 2)] := subs(op(cond, 1), substs);
        end_if;
      end_if;
    end_for;
    pattern := subs(pattern, substs, Unsimplified);
    cond := subs(cond, substs, Unsimplified);
    retval := subs(retval, substs, Unsimplified);
    eqns := map(eqns, eq -> subs(op(eq, 1), substs, Unsimplified)=op(eq, 2));
    pat_conds := subs(pat_conds, substs, Unsimplified);
    res := matchlib::einwohner::single_cand(pattern, cond, retval,
      eqns, pat_conds, ex, oldmainvar, mainvar, res);
    return(res);
  end_if;
  if ids = {} then
    sol := {[]};
  else
    sol := matchlib::einwohner::solve
      (select(eqns, eq -> freeIndets(op(eq, 1))<>{}) union
       select({op(cond)}, eq -> not matchlib::isBlock(eq) and testtype(eq, "_equal") and freeIndets(eq) minus ids = {}), ids);
  end_if;
  case domtype(sol)
  of DOM_SET do
    sol := [[TRUE, sol]];
    break;
  of piecewise do
    sol := [op(sol)];
    break;
  otherwise
    sol := [];
    break;
  end_case;
  // to also find a match for the pattern f(1,x) on f(n,x):
  exidents := freeIndets(map(select(eqns, eq -> freeIndets(op(eq, 1))={}), op, 2));
  if exidents = {} then
    sol_ex := {[]};
  else
    sol_ex := matchlib::einwohner::solve
      (select(eqns, eq -> freeIndets(op(eq, 1))={}), exidents);
  end_if;
  case domtype(sol_ex)
  of DOM_SET do
    sol_ex := [[TRUE, sol_ex]];
    break;
  of piecewise do
    sol_ex := [op(sol_ex)];
    break;
  otherwise
    sol_ex := [];
    break;
  end_case;
  for s in sol do
    for s_ex in sol_ex do
      for substs in s[2] do
        for substs_ex in s_ex[2] do
          // check whether the match actually is one - 
          // the algorithm may find bogus ones, but this check should be fast
//          if subs(pattern, substs, substs_ex, oldmainvar=mainvar) =
//             subs(ex, substs_ex) then
          if matchlib::einwohner::testeq(subs(pattern, substs, substs_ex, oldmainvar=mainvar), // can't use bool(a=b), since solve may have returned
                                         subs(ex, substs_ex)) then                             // parameters in different form (e.g., expanded)
            traperror((res[[_and(op(subs(pat_conds, substs, oldmainvar=mainvar)),
              s[1], s_ex[1], op(substs_ex),
              op(map(subs(cond, substs, oldmainvar=mainvar),
              eval@matchlib::unblock))), subs(retval, substs, substs_ex, oldmainvar=mainvar)]] := 1));
          end_if;
        end_for;
      end_for;
    end_for;
  end_for;
  res;
end:


// this list has been generated by:
/*
ids := table(0):
misc::maprec(intlib::patterns[2],
  {DOM_EXPR} = (ex -> (ids[op(ex, 0)] := ids[op(ex, 0)] + 1; ex))):
prog::sort(map([op(ids)], op, 1), i -> ids[i]);
*/
// Changing this list requires recompiling all pattern lists.
// TODO: The list should actually be stored with the compiled
// patterns and passed in to analyze and everything else.
matchlib::einwohner::rarity_list := hold([coth, Ei, gamma, tanh, abs, psi,
  besselJ, arcsinh, arccosh, arccot, hypergeom, arccos, arcsin, cosh, sinh,
  cot, arctan, ln, tan, exp, sin, cos, "sqrt", "invert", _plus, _mult,
  _power, "expression", "var", "constant"]):

// exprlist -> float, for prog::sort
// lower number means rarer
matchlib::einwohner::rarity :=
proc(ex)
  local i, rarity, n, rec;
  option remember;
begin
  n := nops(matchlib::einwohner::rarity_list)+1.0;
  i := 0;
  rarity := 0.0;
  rec := 
  proc(l)
  begin
    rarity := rarity + 
      contains(matchlib::einwohner::rarity_list, l[1])*n^(-i);
    i := i+1;
    if not l[1] in {"var", "constant", "expression"} and not
      (op(l[1],0) = #plural and op(l[1], 1) in {"var", "constant", "expression"}) then
      map(l[2..-1], rec);
    end_if;
  end_proc;
  rec(ex);
  rarity;
end_proc:

matchlib::einwohner::ACoperators := hold([_mult, _plus, piecewise]):

// almost prog::exprlist,
// but leave constants (i.e., not depending on mainvar) as is
// and sort operands of AC operators by rarity
matchlib::einwohner::exprlist:=
proc(ex, mainvar, genericExpr={})
  local rtree;
begin
  rtree:=
    proc(ex)
      local i, tree;
    begin
      if domtype(ex) = piecewise then
        ex := hold(piecewise)(op(ex));
      end_if;
      if domtype(ex) = DOM_EXPR and 
        (has(ex, mainvar) or has(ex, genericExpr)) then
        tree:= [];
        tree:= append(tree, op(ex, 0));
        for i from 1 to nops(ex) do
          tree:= append(tree, rtree(op(ex, i)))
        end_for;
        return(tree);
      elif domtype(ex) = DOM_LIST then
        tree := [#DOM_LIST, op(map(ex, rtree))];
        return(tree);
      else
        return(ex)
      end_if
    end_proc:

  rtree(ex)
end_proc:

matchlib::einwohner::findplurals :=
proc(l, force=FALSE)
  local head, i, j, ll;
begin
  head := l[1];
  if contains(matchlib::einwohner::ACoperators, head)=0 and not force then
//    // special case: The lists in the first two operands of hypergeom
//    // are commutative, too.
//    if head = hold(hypergeom) then
//      if testtype(l[2], DOM_LIST) and l[2][1] = #DOM_LIST then
//        l[2] := matchlib::einwohner::findplurals(l[2], TRUE);
//      end_if;
//      if testtype(l[3], DOM_LIST) and l[3][1] = #DOM_LIST then
//        l[3] := matchlib::einwohner::findplurals(l[3], TRUE);
//      end_if;
//    end_if;
    return(l);
  end_if;
  l[2..-1] := prog::sort(l[2..-1], matchlib::einwohner::rarity);
  i := 2;
  while i < nops(l) do
    if l[i][1] in {"var", "expression"} then
      i := i+1;
      next;
    end_if;
    // l[i][1] ... l[j-1][1] identical:
    j := i+1;
    while j <= nops(l) and l[i][1] = l[j][1] do
      j := j+1;
    end_while;
    if j-i > 1 then
      // multiple factors of the same type
      if l[i][1]="constant" then
        if contains(matchlib::einwohner::ACoperators, head)>0 then
          l[i..j-1] := [["constant", head(op(map(l[i..j-1], op, 2)))]];
          l[i] := l[i].[matchlib::einwohner::getprop(l[i][2])];
        else
          ll := prog::sort(l[i..j-1], matchlib::einwohner::rarity);
          l[i..j-1] := [[#plural(l[i][1], j-i), ll]];
        end_if;
      else
        ll := prog::sort(l[i..j-1], matchlib::einwohner::rarity);
        l[i..j-1] := [[#plural(l[i][1], j-i), op(map(ll, e -> e[2..-1]))]];
      end_if;
    end_if;
    i := i+1;
  end_while;
  l;
end:

// take an expression and create a list of half-done
// prefix lists from that. If reorder=TRUE, will
// create multiple copies of "plurals" with the same 
// head following.
matchlib::einwohner::analyze :=
proc(x, mainvar=x, reorder=FALSE, genericExpr={})
  local i, l;
begin
  if x = mainvar then [["var", x]]
  elif x in genericExpr then [["expression", x]]
  elif not has(x, mainvar) and not has(x, genericExpr) and not has(x, #DOM_LIST) then
    [["constant", x, matchlib::einwohner::getprop(x)]]
  elif domtype(x)=DOM_LIST then
    l := matchlib::einwohner::cartesianProduct(op(map(x[2..-1],
      matchlib::einwohner::analyze, mainvar, reorder, genericExpr)));
    x := map(l, t -> [x[1]].t);
    x := map(x, matchlib::einwohner::findplurals);
    for i from 1 to nops(x) do
      case x[i][1]
      of hold(_power) do
        case x[i][3]
        of ["constant", -1, {-1}] do
          x := x.[["invert", x[i][2]]]; break;
        of ["constant", 1/2, {1/2}] do
          x := x.[["sqrt", x[i][2]]]; break;
        of ["constant", -1/2, {-1/2}] do
          x := x.[["invert", ["sqrt", x[i][2]]]]; break;
        end_case;
        break;
      end_case;
    end_for;
    // Look for multi-level plurals
    if reorder then
      x := map(x,
        t -> op(map(matchlib::einwohner::cartesianProduct(op(map(t[2..-1],
          matchlib::einwohner::reorderings))), e -> [t[1]].e)));
    end_if;
    x;
  else error("now that's odd"); end_if;
end:

// undo analyze, probably on a subexpression
matchlib::einwohner::reconstruct_expression :=
proc(l)
begin
  if l[1] in {"var", "expression", "constant"} then
    return(l[2]);
  end;
  l[2..-1] := map(l[2..-1], matchlib::einwohner::reconstruct_expression);
  case l[1]
  of "sqrt" do return(sqrt(l[2])); break;
  of "invert" do return(1/l[2]); break;
  end_case;
  l[1](op(l[2..-1]));
end_proc:

// For multi-level plurals which do not have a definite ordering,
// (i.e., which have the same prefix lists in multiple children,
// i.e., differ only in their constants), add all possible reorderings:
matchlib::einwohner::reorderings :=
proc(x)
  local pl, l, i, j, perm, p;
begin
  if domtype(x) = DOM_LIST and domtype(x[1])=DOM_EXPR and op(x[1], 0)=#plural then
    // TODO: Only ask for prefixLists of those operands
    // which have direct neighbours with the same list of heads
    pl := map(x[2..-1], map, matchlib::einwohner::prefixLists);
    l := [x];
    i := 1;
    repeat
      j := i+1;
      while j <= nops(pl) and pl[j] = pl[i] do
        j := j+1;
      end_while;
      if j > i+1 then
        // found multiple children with same head
        perm := combinat::permute(x[i+1..j]);
        l := map(l, e -> ((e[i+1..j] := p; e) $ p in perm));
      end_if;
      i := j;
    until i > nops(pl) end_repeat;
    l;
  else
    [x];
  end_if;
end_proc:

// take a list generated by analyze and create the prefix lists of that
matchlib::einwohner::prefixLists :=
proc(l)
  local pref, prefs, i, tails, tail;
begin
  pref := [l[1]];
  if pref in {["var"], ["expression"], ["constant"]} then
    return([pref]);
  end_if;
  //prefs[pref] := 1; // collect in table, return list
  prefs := table();
  // unary function? no recursion needed
  while nops(l) = 2 and
    not (domtype(l[1])=DOM_EXPR and op(l[1],0)=#plural) and 
    domtype(l[2])=DOM_LIST do
    l := l[2];
    pref := pref.[l[1]];
  end_while;
  // terminal or multiple args
  if domtype(l[1])=DOM_EXPR and op(l[1],0)=#plural then
    // multiple args
    for i from 2 to nops(l) do
      if op(l[1], 1) = "constant" then
        tails := [[["constant"]]];
      else
        tails := map(l[i], matchlib::einwohner::prefixLists);
      end_if;
      for tail in tails do
        prefs[pref.[i-1].op(tail)] := 1;
      end_for;
    end_for;    
  elif nops(l) > 2 and not l[1] in {"var", "expression", "constant"} then
    // multiple args
    for i from 2 to nops(l) do
      tails := matchlib::einwohner::prefixLists(l[i]);
      for tail in tails do
        prefs[pref.[i-1].tail] := 1;
      end_for;
    end_for;
  else
    prefs[pref] := 1;
  end_if;
  prefs := map([op(prefs)], op, 1);
// prefs can contain lists, as in [[D([1], f), ...]]
//  assert(not hastype({op(map(prefs, op))}, DOM_LIST));
  prefs;
end_proc:

// calling the compiler results in a list of four values,
// which must be passed to matchlib::einwohner in exactly
// the same order.
matchlib::einwohner::compile :=
proc(patterns, mainvar, genericExpr={})
  local i, j, pat, tr, prefs, Tprefs, Tanalyzed, Tconds, conds, pref, inds, ret;
begin
  map(anames(Properties), _save);
  map(select(indets(patterns) minus Type::ConstantIdents, x->protected(x)=None), _save);

  patterns := [op(map({op(patterns)}, matchlib::einwohner::specialcases, mainvar))];

  Tprefs := table();
  Tanalyzed := [0$nops(patterns)];
  Tconds := [0$nops(patterns)];
  for i from 1 to nops(patterns) do
    pat := patterns[i];
    
    tr := matchlib::einwohner::exprlist(pat[1], mainvar, genericExpr);
    conds := if nops(pat)>2 then pat[3] else [] end_if;
    Tconds[i] := conds;
    proc() // for local assumptions
    begin
      map(conds, c -> context(hold(traperror)(hold(assumeAlso)(c))));
    
      tr := matchlib::einwohner::analyze(tr, mainvar, TRUE, genericExpr);
      Tanalyzed[i] := tr;
      prefs := map(tr, matchlib::einwohner::prefixLists);
    end_proc();

    for j from 1 to nops(prefs) do
      for pref in prefs[j] do
        Tprefs[pref][[i,j]] := 1;
      end_for;
    end_for;
  end;
//  map(anames(Properties), _delete);

  ret := [subsop(map(Tprefs, t -> map({op(t)}, op, 1)), 0 = {}),
          map(patterns, op, 1), mainvar,
          Tanalyzed, Tconds, map(patterns, op, 2)];
  // to avoid name clashes, replace all free idents with protected versions
//  inds := {prog::find(ret[2], "constant")};
//  inds := map(inds, p->op(ret[2], p[1..-2].[2]));
//  inds := freeIndets(inds);
  inds := freeIndets(ret[2]) minus {mainvar} minus genericExpr;
  subs(ret, map(inds, t -> t=`#${}{}`.t), Unsimplified);
end:


// from the table of prefix lists, look for candidate matches
matchlib::einwohner::candidates :=
proc(ex, mainvar, tprefs, tanalyzed, oldmainvar)
  local analyzed, prefixes, cands, i;
begin
  analyzed := matchlib::einwohner::analyze
    (matchlib::einwohner::exprlist(ex, mainvar), mainvar);
  prefixes := map(analyzed, matchlib::einwohner::prefixLists);
//  cands := map(prefixes, c->_intersect(op(map(c, p -> tprefs[p]))));
  cands := map(prefixes,
    proc(c)
      name matchlib::einwohner::candidates::check_single_cand;
      local matches, i;
    begin
      // for each prefixList, get the list of possible matches
      matches := map(c, p -> map([p, p[1..i].["expression"] $ i=1..nops(p)-1],
          pp -> if contains(tprefs, pp) then tprefs[pp] else null() end_if));
      // then, check all combinations of these
      matches := matchlib::einwohner::cartesianProduct(op(matches));
      // for each combination, find the intersection, i.e., the list of patterns matching all of them
      matches := map(matches, _intersect@op);
      // discard empty intersections, ignore order
      {op(matches)} minus {{}};
    end);
  {op(map(cands[i], op@map, c -> (matchlib::einwohner::couldmatch(analyzed[i],
                      subs(tanalyzed[c[1]][c[2]], oldmainvar=mainvar), c[1]))))
                    $ i=1..nops(cands)};
end:

// check more carefully
matchlib::einwohner::couldmatch :=
proc(ex1, ex2, c)
  local i, j, res, substs, conds;
begin
  if ex1=ex2 then
    return([c, {}, {}]);
  end_if;
  if ex2[1] = "expression" then
    // the order of left and right hand side is important!
    return([c, {ex2[2]=matchlib::einwohner::reconstruct_expression(ex1)}, {}]);
  end_if;
  if ex1[1] <> ex2[1] or nops(ex1)<>nops(ex2) then
    return(null());
  end_if;
  if ex1[1]="constant" then
    case is(ex1[2] in ex2[3])
    of TRUE do return([c, {ex2[2]=ex1[2]}, {}]);
    of UNKNOWN do return([c, {ex2[2]=ex1[2]}, {ex1[2] in ex2[3]}]);
    of FALSE do return(null());
    end_case;
    assert(FALSE); // can't get here
  elif domtype(ex1[1])=DOM_EXPR and op(ex1[1], 0)=#plural then
    substs := {};
    conds := {};
    for i from 2 to nops(ex1) do
      if nops(ex1[i]) <> nops(ex2[i]) then
        return(null());
      end_if;
      for j from 1 to nops(ex1[i]) do
        res := matchlib::einwohner::couldmatch(ex1[i][j], ex2[i][j], c);
        if res = null() then
          return(null());
        end_if;
        // check for incompatible substitutions!
        substs := substs union res[2];
        conds := conds union res[3];
      end_for;
    end_for;
    return([c, substs, conds]);    
  else
    substs := {};
    conds := {};
    for i from 2 to nops(ex1) do
      res := matchlib::einwohner::couldmatch(ex1[i], ex2[i], c);
      if res = null() then
        return(null());
      end_if;
      // check for incompatible substitutions!
      substs := substs union res[2];
      conds := conds union res[3];
    end_for;
    return([c, substs, conds]);
  end_if;
end:

// utility: try a bit harder to get a DOM_SET from solve,
// avoiding new parameters.
matchlib::einwohner::solve :=
proc(eqs, inds, opts=null())
  local sol1, sol2, polys;
begin
  eqs := map(eqs,
    eq -> if domtype(lhs(eq))=DOM_LIST or domtype(rhs(eq))=DOM_LIST then
      if domtype(lhs(eq))=DOM_LIST and domtype(rhs(eq))=DOM_LIST and
        nops(lhs(eq)) = nops(rhs(eq)) then
        op(zip(lhs(eq), rhs(eq), _equal));
      else
        1 // solve(1) returns the empty set
      end_if
    else eq end_if);
  eqs := map(eqs, simplify::simplifyCondition);
    // solve cannot handle infinities in equations like x=infinity
  eqs:= subs(eqs, infinity = #infinity);  
  sol1 := solve(args());
  sol1:= subs(sol1, #infinity = infinity);
  if domtype(sol1) = DOM_SET then
    if indets(sol1) minus indets([eqs, inds])={} then
      return(sol1);
    else
      // solve introduced new indet, as in solve({k+l=1}, {k,l})
      // try looking for a better set to solve for.
      polys := map(eqs, eq -> poly(op(eq, 1)-op(eq, 2), sort(inds)));
      if has(polys, [FAIL, infinity]) then
        // TODO: Try rationalizing
      else
        traperror((
          polys := groebner::gbasis(polys);
          sol1 := solve(polys, groebner::stronglyIndependentSets(polys)[2]);
        ));
      end_if;
    end_if;
  end_if;
  
  sol2 := solve(args(), IgnoreProperties);
  if domtype(sol2) = DOM_SET and indets(sol2) minus indets([eqs, inds])={} then
    return(select(sol2, s -> is(_and(op(s)), Goal=FALSE) <> FALSE));
  end_if;
  sol2 := solve(args(), IgnoreProperties, IgnoreSpecialCases);
  if domtype(sol2) = DOM_SET and indets(sol2) minus indets([eqs, inds])={} then
    return(select(sol2, s -> is(_and(op(s)), Goal=FALSE) <> FALSE));
  end_if;
  sol1;
end_proc:

// special cases: a*x should match x,
// same for + and ^:

matchlib::einwohner::specialcases :=
proc(pattern, mainvar)
  local evalsubsts, substsaftereval, additional, add_pat, target, vars,
    v, v2, p2, i, permissibleNewIdents;
begin
   // for unblocking, these should be deactivated:
   evalsubsts := [hold(sum::sum_fn)=`#sum_fn`,
                  hold(sum::evalAtPoint)=`#evalAtPoint`,
                  hold(sum)=`#sum`,
                  hold(product::product_fn)=`#prod_fn`,
                  hold(product)=`#product`,
                  hold(fact)=`#fact`,
                  hold(int::addpattern::rewriteToDiff)=`#rewriteToDiff`,
                  hold(int::addpattern::rewriteToD)=`#rewriteToD`,
                  hold(int)=`#int`,
                  hold(binomial)=`#binom`,
                  hold(intlib::tryChangeVar)=`#tryCV`,
                  hold(intlib::intdef1)=`#intdef1`,
                  hold(hold)=`#hold`,
                  hold(evalAt)=`#evalAt`,
                  hold(testtype)=`#testtype`,
                  hold(iszero)=`#iszero`,
                  hold(has)=`#has`,
                  hold(bool)=`#bool`,
                  hold(diff)=`#diff`,
                  hold(Simplify)=`#Simplify`,
                  hold(intlib::Simplify)=`#iSimplify`,
                  hold(simplify)=`#simplify`,
                  hold(piecewise)=`#piecewise`,
                  infinity=`#infinity`,
                  -infinity=`#-infinity`,
                  hold(infinity)=`#infinity`,
                  hold(kroneckerDelta) = `#kroneckerDelta`,
                  hold(transform::ztrans::evalAtPoint)=`#ztrans::evalAtPoint`,
                  hold(transform::ztrans::mult2diff)=`#mult2diff`,
                  hold(transform::ztrans)=`#ztrans`,
                  hold(transform::invztrans::lookup_f1)=`#ztrans::invztrans::lookup_f1`,
                  hold(transform::invztrans)=`#transform::invztrans`,
                  hold(transform::laplace::evalAtPoint)=`#laplace::evalAtPoint`,
                  hold(hypergeom)=#hypergeom,
                  hold(specfunc::Wurzelbehandlung)=`#specfunc::Wurzelbehandlung`
                 ]:
   evalsubsts := map(evalsubsts, 
                         x -> x[1] =
                           if has(pattern, x[2]) then
                             genident("".x[2])
                           else x[2]
                           end);
   // undo the change:
   substsaftereval := map(evalsubsts, x -> op(x,2)=op(x,1)):

   additional := {};
   add_pat :=
   proc(pat, vars)
     local pos, v, p, set, e, t, value, do_unbl;
   begin
     if nops(vars) = 0 then return(); end;
     v := op(vars, 1);
     vars := vars minus {v};
     set[0] := FALSE;
     set[1] := FALSE;
     set[exp(1)] := FALSE;
     pos := [prog::find(pat[1], v)];
     for p in pos do
       if nops(p) = 0 then next; end;
       case op(pat[1], p[1..-2].[0])
       of hold(_mult) do 
         set[1] := TRUE;
         break;
       of hold(_plus) do
         set[0] := TRUE;
         break;
       of hold(_power) do
         set[1] := TRUE;
         set[0] := TRUE;
         if p[-1]=1 then
           set[exp(1)] := TRUE;
         end_if;
       end_case;
     end_for;
     do_unbl := e -> misc::maprec(e, 
        	   {matchlib::block} = (ex -> subs(matchlib::block(1), 
                                    1=eval(do_unbl(matchlib::unblock(ex))))));
     for value in map([op(set)], op, 1) do
       if set[value]=TRUE then
       traperror((
                   e := subs(pat, v=value, EvalChanges);
                    // simplification
                    e := do_unbl(e).[[], []];
                    if not contains(e[3].e[4], FALSE)>0 then
                      e[3] := select(e[3], _unequal, TRUE);
                      e[4] := select(e[4], _unequal, TRUE);
                      t := additional union {e[1..4]};
                      additional := t;
                      add_pat(e, vars);
                    end_if;
                   ));
       end_if;
     end_for;
     add_pat(pat, vars);
     null();
   end_proc:
   vars := freeIndets(pattern) minus {mainvar, `#matchlib::block`};

   // make sure x^(-a) matches x^(-5):   
   for v in vars do
     i := 0;
     v2 := hold(`-`).v;
     while has(pattern, {v2}) do
       i := i+1;
       v2 := hold(`-`).v.i;
     end_while;
     p2 := subs(pattern[1], -v=v2, Unsimplified);
     if p2 <> pattern[1] then
       vars := vars union {v2};
       if has(p2, v) then
         pattern[1] := p2;
         if nops(pattern) > 2 then
           pattern[3] := [-v=v2].pattern[3]
         else
           pattern := pattern.[[-v=v2]];
         end_if;
       else
         pattern[1] := p2;
         pattern := subs(pattern, v=-v2, Unsimplified);
         vars := vars minus {v};
       end_if;
     end_if;
   end_for;

   add_pat(subs(pattern, evalsubsts), vars);
   
   // add some rewritten forms -- see bug report #1588
   for target in [sincos] do
     additional := additional union
                 map(additional,
                     x -> [rewrite(x[1], target)].x[2..-1]);
   end_for:
   
   // add reordered forms of hypergeometric terms.
   // Note: While this may introduce a combinatorial number of
   // new patterns, in practice, we only have patterns for 2F1, 3F2 etc.,
   // where the number of additional patterns is rather low.
   add_pat :=
   proc(pat, hypergeoms = FAIL)
     local cand, scand, spat, l1, pl1, l2, pl2;
   begin
     if args(0) < 2 then
       hypergeoms := table();
       misc::maprec(pat,
         (ex -> bool(op(ex, 0)=#hypergeom)) = (ex -> (hypergeoms[ex] := ex)),
         Unsimplified);
       hypergeoms := map({op(hypergeoms)}, op, 1);
     end_if;
     if hypergeoms <> {} then
       cand := op(hypergeoms, 1);
       hypergeoms := subsop(hypergeoms, 1=null());
       l1 := op(cand, 1);
       l2 := op(cand, 2);
       for pl1 in combinat::permute(l1) do
         for pl2 in combinat::permute(l2) do
           scand := subsop(cand, 1 = pl1, 2 = pl2, Unsimplified);
           spat := subsop(pat, 1 = subs(pat[1], cand = scand,
             Unsimplified), Unsimplified);
           spat := (spat.[[], []])[1..4];
           additional := additional union {spat};
           if hypergeoms <> {} then
             add_pat(spat, hypergeoms);
           end_if;
         end_for;
       end_for;
     end_if;
     null();
   end:
   map(additional, add_pat);
   add_pat(subs(pattern, evalsubsts));
   
   // remove special cases where the conditions simply can't be fulfilled
   // note that has et al. are still blocked!
   additional := select(additional,
      proc(pat)
        local res;
      begin
        _lazy_or(nops(pat) = 2,
                 has(pat[3], hold(`#testtype`)),
                 traperror((res := is(_and(op(eval(matchlib::unblock(pat[3]))))))) <> 0,  
                 res <> FALSE,
                 if nops(pat) > 3 then
                   _lazy_or(traperror((res := is(_and(op(eval(matchlib::unblock(pat[4]))))))) <> 0,
                            res <> FALSE)
                 else
                  FALSE
                 end);
      end_proc):
   
   // evaluate conditions not depending on free variables
   additional := map(additional,
      proc(pat)
      begin
        if nops(pat) < 3 then pat else
           pat[3] := map(pat[3],
              cond -> if indets(cond) minus {`#matchlib::block`} = {} then 
                        is(subs(matchlib::unblock(cond), substsaftereval, EvalChanges))
                      else cond
                      end_if);
           pat[3] := select(pat[3],
                            x -> x <> TRUE and x <> matchlib::block(TRUE));
           if contains(pat[3], FALSE)>0 or
              contains(pat[3], matchlib::block(FALSE)) > 0
           then null() else pat end_if
         end_if;
      end);  
      
   additional := subs(additional, substsaftereval);

   // remove those where vars got lost

   // these idents may appear in the rhs without appearing in the lhs:
   permissibleNewIdents := indets(pattern[2]) minus indets(pattern[1])
     union {hold(infinity), hold(Z_), hold(Q_), hold(R_), hold(C_),
       hold(`#matchlib::block`)};
  
   // First remove conditions of has(var, _X) and not has(var, _X)
   // from the conditions of the pattern because these conditions
   // does not have a meaning for the validity of the pattern
   // conditons gioven in con have a form like:
   // array(1..2, (1) = `#matchlib::block`, (2) = not #has(#b, _X))
   additional := select(additional,
     pat -> (indets([op(pat[2..3]),
             select(pat[4],
     con ->  (case op(con,[2,0])
               of hold(`#has`) do
                 if op(con,2) = hold(_X) then
                   return(FALSE) ;
                 end_if:
                 break;
               of hold(_not) do
                 if op(con,[2,1,0]) = hold(`#has`) then
                   if op(con,[2,1,2]) = hold(_X) then
                   return(FALSE);
                   end_if:
                 end_if:
                 break;
               end_case:
              TRUE;) ) ])
                 intersect map(vars, op, 1)) minus indets(pat[1]) minus permissibleNewIdents = {});

   pattern, op(additional);
end:

// getprop with traperror
matchlib::einwohner::getprop :=
proc(ex)
  local res;
begin
  if traperror((res := getprop(ex, "Constant"=TRUE, "DisAllowed" = {Dom::ImageSet, piecewise}))) <> 0 then
    C_;
  elif indets(res) minus Type::ConstantIdents <> {} then
    C_;
  else
    res;
  end_if;
end_proc:

// limited Cartesian product
matchlib::einwohner::cartesianProduct :=
proc()
  local l;
begin
  l := _mult(op(map([args()], nops)));
  if iszero(l) or l > 300 then
    return([]);
  end_if;
  combinat::cartesianProduct(args());
end:

// testeq "properly" handling #defint(...) etc.
matchlib::einwohner::testeq :=
proc(ex1, ex2)
  local i;
begin
  if ex1 = ex2 then return(TRUE); end_if;
  if ex1::dom = DOM_EXPR and ex2::dom = DOM_EXPR and
    op(ex1, 0) = op(ex2, 0) and domtype(op(ex1, 0)) = DOM_IDENT and
    strmatch(expr2text(op(ex1, 0)), "^`?#") then
    if nops(ex1) <> nops(ex2) then
      return(FALSE);
    end_if;
    for i from 1 to nops(ex1) do
      if matchlib::einwohner::testeq(op(ex1, i), op(ex2, i)) = FALSE then
        return(FALSE);
      end_if;
    end_for;
    return(TRUE);
  end_if;
  bool(testeq(ex1, ex2)=TRUE);
end_proc:
