
//*******************************************************************************************
//                       The PSLQ Integer Relation Algorithm                                   
//                                                                                             
// Aut.: Helaman R.P. Ferguson and David Bailey "A Polynomial Time, Numerically Stable         
//       Integer Relation Algorithm" (RNR Technical Report RNR-92-032)    helaman@super.org    
// The implementation follows David Bailey and Simon Plouffe "Recognizing Numerical Constants" dbailey@nas.nasa.gov 
// Cod.: Raymond Manzoni  raymman@club-internet.fr                                             
//*******************************************************************************************
// this code is free...              

/*++ Given a list of constants x find coefficients sol[i] such that
      sum(sol[i]*x[i], i=1..n) = 0    (where n=nops(x))              
                                                                  
    x is the list of real expressions           
          float(x[i]) must evaluate to numeric type!                     
    precision is the number of digits needed for completion;         
          must be greater or equal to log10(max(sol[i]))*n               
    returns the list of solutions with initial precision
          and the confidence (the lower the better)

    Example:

    >> pslq([2*PI+exp(1),PI,exp(1)],20);
           
            Found with Confidence , 1.4780013e-28
                        +-         -+
                        | 1, -2, -1 |
                        +-         -+
++*/

misc::pslq:=
proc(x:DOM_LIST, precision:Type::PosInt)
  local gam, A, tB, H, n, i, j, k, s, y, absy, t, m, maxi, gami,
	 t0, t1, t2, t3, t4, mini, Confidence, finish;
  save DIGITS;
begin
  if traperror((x:= map(x, float))) <> 0 then
    error("List elements could not be converted to floats")
  end_if;

  if testargs() then
    if not testtype(x, Type::ListOf(Type::Real)) = TRUE then
      error("List must consist of real numbers")
    end_if;
  end_if;


  n:=nops(x);

  if (k:= contains(map(x, iszero), TRUE)) > 0 then
    userinfo(2, "Input contains zero entry");
    return(matrix([[0 $k-1, 1, 0 $n-k]]))
  end_if;

  DIGITS:=precision+6;
  Confidence:=10^(-precision);
  gam:=float(sqrt(4/3));
  
  // Initialization steps
  
  // step 1
  A:= tB:= [[0$i-1, 1, 0$n-i] $i=1..n]; // unit matrix

  // instead of using B, we work with the transpose of B
  // i.e., A is a list of lists representing rows
  // while tB is a list of lists representing of columns


  // step 2
  
  s:= [sqrt(_plus(x[j]^2 $j=k..n)) $k=1..n];
  y:= map(x, _divide, s[1]); 
  s:= map(s, _divide, s[1]);

  // step 3
  
  H:= [[-(y[i]*y[j])/(s[j]*s[j + 1]) $j=1..i-1, s[i+1]/s[i], 0 $n-1-i]
       $i=1..n-1,
       [-(y[n]*y[j])/(s[j]*s[j + 1]) $j=1..n-1]]; 

  // step 4
  
  for i from 2 to n do
    for j from i-1 downto 1 do
      if iszero(H[j][j]) then
        t:= RD_INF
      else
        t:=round(H[i][j]/H[j][j]);
      end_if;
      y[j]:=y[j] + t*y[i];
      for k from 1 to j do
        H[i][k]:=H[i][k]-t*H[j][k]
      end_for;
      A[i]:= zip(A[i], map(A[j], _mult, t), _subtract);
      tB[j]:= zip(tB[j], map(tB[i], _mult, t), _plus);
    end_for
  end_for;

  // Loop 

  
  while TRUE do

    // step 1

    gami:= [gam^i *specfunc::abs(H[i][i]) $i=1..n-1];
    maxi:= max(op(gami));
    m:= contains(gami, maxi);
    
    // step 2
    
    [y[m+1], y[m]]:= [y[m], y[m+1]];
    [A[m+1], A[m]]:= [A[m], A[m+1]];
    [tB[m+1], tB[m]]:= [tB[m], tB[m+1]];
    [H[m+1], H[m]] := [H[m], H[m+1]];
        
    // step 3
    
    if m < n-1 then
      t0:=sqrt(H[m][m]^2 + H[m][m + 1]^2);
      t1:=H[m][m]/t0;
      t2:=H[m][m + 1]/t0;
      for i from m to n do
        t3:=H[i][m];
        t4:=H[i][m + 1];
        H[i][m]:=t1*t3 + t2*t4;
        H[i][m + 1]:=-t2*t3 + t1*t4
      end_for
    end_if;

    // step 4
    
    finish:= FALSE;
    for i from m+1 to n do
      for j from min(i-1, m + 1) downto 1 do
        if iszero(H[j][j]) then
           // t = RD_INF;
           y[j]:= RD_INF;
           for k from 1 to j-1 do
             H[i][k]:= RD_INF
           end_for;
           A[i]:= [RD_INF $nops(A[i])];
           tB[j]:= [RD_INF $nops(tB[j])];
           H[i][j]:= 0.0;
           finish:= TRUE;
        else
           t:=round(H[i][j]/H[j][j]);
           y[j]:=y[j] + t*y[i];
           for k from 1 to j do
             H[i][k]:=H[i][k]-t*H[j][k]
           end_for;
           A[i]:= zip(A[i], map(A[j], _mult, t), _subtract);
           tB[j]:= zip(tB[j], map(tB[i], _mult, t), _plus);
         end_if;
      end_for
    end_for;

    // step 5
    
    DIGITS:=8; //low precision

    if finish then
       userinfo(5, "Exact linear combination detected")
    else
       userinfo(5, "Norm bound:".
             expr2text(1/sqrt(max(_plus(op(map(H[j], _power, 2)))
                $j=1..n))));
    end_if;

    // step 6

    absy:= map(y, specfunc::abs);
    [mini, maxi]:= [min, max](absy);
    
    userinfo(5," Min=".expr2text(mini).
		" Confidence=".expr2text(mini/maxi)); 
    if mini/maxi < Confidence then
//preferred to : if mini < 10^(- precision) then
      userinfo(5,"Found with Confidence ".expr2text(mini/maxi));
      // result is:
      matrix(1, n, tB[contains(absy, mini)]);
      break
    end_if;
    maxi:= max(op(map(map(A, op), specfunc::abs)));
    if maxi > 10^(precision+4) then
      return(FAIL)
    end_if;

    // reset to full precision, i.e., undo the DIGITS:= 8 above
    DIGITS:=precision+6;
  end_while;
  
end_proc:

// end of file 

