//----------------------------------------------------------------
// ncdata(n) -- computes numerical values of the weights          
// and abscissae of Newton-Cotes quadrature with n nodes.         
//                                                                
// return value: list of lists [ b , c ], where                   
//               b=[ b[1],..,b[n] ] , c=[c[1],..,c[n]] are the    
//               weights and abscissae of Newton-Cotes quadrature 
//               int(f(x),x=0..1)= sum( b[i]*f(c[i]), i=1..n )    
//               with n equidistant nodes c[i]=(i-1)/(n-1),i=1..n.
// Note:   n=2    trapezoidal rule                                
//         n=3    Simpson rule                                    
//         n=5    Milne rule                                      
//         n=7    Weddle rule                                     
//                                                                
// Example:                                                       
// >>  ([b,c]):=numeric::ncdata(8);                               
//----------------------------------------------------------------

numeric::ncdata:=proc(n) local j,k,b,nm1,m; option remember;
begin
   if args(0)<>1 then error("expecting one argument"); end_if;
   if domtype(n)<>DOM_INT then error("expecting integer number of nodes") end_if;
   if n<1 then error("expecting positive number of nodes") end_if;
   if   n=1 then return([[1],[1/2]])
   elif n=2 then b:=[1/2,1/2]
   elif n=3 then b:=[1/6,2/3,1/6]
   elif n=4 then b:=[1/8,3/8,3/8,1/8]
   elif n=5 then b:=[7/90,16/45,2/15,16/45,7/90]
   elif n=6 then b:=[19/288,25/96,25/144,25/144,25/96,19/288]
   elif n=7 then b:=[41/840,9/35,9/280,34/105,9/280,9/35,41/840]
   elif n=8 then b:=[751/17280,3577/17280,49/640,2989/17280,
                     2989/17280,49/640,3577/17280,751/17280]
   elif n=9 then b:=[989/28350,2944/14175,-464/14175,5248/14175,
                     -454/2835,5248/14175,-464/14175,2944/14175,
                     989/28350]
   elif n=10 then b:=[2857/89600,15741/89600,27/2240,1209/5600,       
                      2889/44800,2889/44800,1209/5600,27/2240,
                      15741/89600,2857/89600]
   elif n=11 then b:=[16067/598752,26575/149688,-16175/199584,
                      5675/12474,-4825/11088,17807/24948,
                      -4825/11088,5675/12474,-16175/199584,
                      26575/149688,16067/598752]
   else 
        nm1:=n-1; m:=trunc(n/2)+1;
        (b[j]:=nm1^(j-1)/j) $ j=1..n;
        (for j from n downto k+2 do b[j]:=b[j]-k*b[j-1] end_for) $ k=1..n-2;
      //# original code #
      //for k from nm1 downto 1 do
      //  (b[j]:=b[j]/k) $ j=k+1..n;
      //  (b[j]:=b[j]-b[j+1]) $ j=k..nm1
      //end_for:
      //# better: only need to compute b[m],..,b[n], since b[j]=b[n+1-j]#
        for k from nm1 downto 2 do
          (b[j]:=b[j]/k) $ j=max(m,k+1)..n;
          (b[j]:=b[j]-b[j+1]) $ j=max(m,k)..nm1
        end_for:
        (b[j]:=b[j]-b[j+1]) $ j=m..nm1;
        b:= [(b[n+1-j]$ j=1..m-1),(b[j] $ j=m..n)] //convert table -> list
   end_if;
   [b, [j/(n-1) $ j=0..(n-1)]];
end_proc:
