// 1.te Version  9.9.97 
// 2.te Version  1.7.98 
// 3.te Version 24.8.98 
// 4.te Version  1.9.99 
// 2.4.01: neue Option AbsoluteError = tol eingefuehrt 
//------------------------------------------------------------------------
// The following quadrature is the control routine for numerical          
// quadrature. It should be called in the form                            
//                                                                        
//      quadrature(f(x),x=x0..x1, <MaxCalls = m> )                        
// or                                                                     
//      quadrature(f(x),x=x0..x1,method=n)                                
// or                                                                     
//      quadrature(f(x),x=x0..x1,method=n,Adaptive=TRUE/FALSE)            
//                                                                        
// In general:
//    numeric::quadrature(f(x), x = x0..x1
//                        <, MaxCalls = m>
//                        <, method = n>
//                        <, Adaptive = TRUE/FALSE>
//                        <, AbsoluteError = tol>
//                       )
//
// (AbsoluteError = tol wird nicht dokumentiert, da etwas problematisch)
//
// Parameters:  f(x) -- univariate expression in indeterminate x          
//              m    -- positive integer or infinity                      
//              n    -- positive integer                                  
//              tol  -- a positive real numerical value. 
//                      quadrature attempts to keep the
//                      absolute error of the result below tol.
//                                                                        
// <method> is either GaussLegendre or NewtonCotes or GaussTschebyscheff  
// Following alternative names for these methods are accepted:            
//                                                                        
//    GaussLegendre,Gauss,GL,
//    GaussTschebyscheff,GT,GaussChebyshev,GC,                        
//    NewtonCotes,NC
//                                                                        
// The adaptive mechanism is based on a fixed quadrature rule specified   
// by one of these names with n nodes, where <n> may take any finite      
// value from {1,2,3,...}.                                                
// If no method is specified, then, depending on DIGITS, an adaptive      
// Gauss-Legendre quadrature with n=20,40,80 or 160 nodes (with           
// respective orders 40,80,160 or 320) is used as default.                
//                                                                        
// MaxCalls = m (with a positive integer m) is a stopping criterion:      
// after m evaluations of the integrand quadrature gives up and returns   
// an inaccurate result with a warning. m may be infinity. In this case   
// quadrature continues until it finds the result or runs into internal   
// problems (typically: evaluation of the integrand at a singularity)     
// The default value is m = MAXDEPTH*n .                                  
//
// It may also happen, that quadrature returns inaccurate results,        
// when the internal recursive depth exceeds MAXDEPTH. In this case       
// a warning is given.                                                    
//                                                                        
// Warnings occur only when quadrature is called interactively.           
// Otherwise FAIL is returned instead of warnings!                        
//
// *************************************************************
// Warning: the adaptive error control does not work well with
// AbsoluteError = tol. We recommend, not to use this option!
// *************************************************************
//                                                                        
// Examples:                                                              
// >>  use(numeric):                                                   
// >>  quadrature( exp(-x^2), x=0..1 );                                   
// >>  quadrature( exp(-x^2), x=0..infinity);                             
// >>  quadrature( exp(-x^2), x=0..1, MaxCalls=infinity );                
// >>  quadrature( exp(-x^2), x=0..1, GaussLegendre=40 );                 
// >>  quadrature( exp(-x^2), x=0..1, GL=160, Adaptive=FALSE );           
// >>  quadrature( exp(-x^2), x=0..1, NewtonCotes=10, Adaptive=TRUE );    
//                                                                        
// Technical remark: the data of Gaussian quadrature GL=20, GL=40, GL=80  
//      and GL=160 are pre-computed and stored with 200 DIGITS, whence    
//      there is no delay when using these methods with at most 200       
//      DIGITS. For different values of <n> or higher DIGITS these data   
//      will be computed before the actual quadrature. This may take some 
//      time. Therefore it is recommended to use these values of <n> and  
//      to restrict ones ambitions to DIGITS<=200.                        
//      The algorithm will converge eventually for arbitrarily high       
//      DIGITS and arbitrary integrands (provided the integral exists).   
//      However, convergence will be slow if very high precision is       
//      required. Also, convergence will be slow if the integrand is not  
//      smooth.                                                           
//------------------------------------------------------------------------
// The symbolic <abs> uses properties and is much too expensive.
// Substitute it by the cheap lazyAbs:
numeric::lazyAbs:= proc(x)
               name numeric::lazyAbs; 
               option noDebug;
               begin
                   x:= float(x);
                   case domtype(x)
                   of DOM_FLOAT do
                   of DOM_COMPLEX do
                             return(specfunc::abs(x));
                   otherwise return(procname(args()))
                   end_case;
               end_proc:

numeric::quadrature:=proc()
local alg_expr,eq,adaptive,method,opt,n,b,c,freeStyle, Quadsum, 
      adaptiveQuad, Is, Is0, x, x0, x1, test_x, macheps, fcalls, 
      maxcalls, interactive, problem, useAbsErr, tol, dummy, 
      leftsingularity, rightsingularity;
begin
  if args(0)<2 then error("expecting at least 2 arguments"); end_if:
  alg_expr:= args(1); 
  // Does the integrand contain 'infinity' or 'undefined'?
  if has(alg_expr, infinity) then
     // do not search for infinity in the ranges of
     // numeric::quadrature, numeric::int, int etc.
     misc::maprec(alg_expr, 
                  (x -> (testtype(x, "_equal") and
                         testtype(op(x, 2), "_range"))) = id,
                  {stdlib::Infinity} = (
                   x -> error("cannot integrate expressions involving 'infinity'")),
                  PostMap, NoOperators)
  end_if;
  if has(alg_expr, undefined) then
     error("cannot integrate expressions involving 'undefined'"):
  end_if;

  eq:= args(2);

  if (not testtype(eq,"_equal")) or
     (not testtype(op(eq,2), "_range"))
     then error("expecting range x=a..b");
  end_if;

  // ---------------------------------------------------

  x:=op(eq,1): 
  x0:=op(eq,[2,1]): 
  x1:=op(eq,[2,2]):
  if numeric::indets(x)={} then 
     error("no integration variable") 
  end_if:

  // ---------------------------------------------------
  // The following is to allow multiple integrations: return the call 
  // unevaluated, if there are symbols. Note that numeric::indets does
  // not count the integration variable of unevalated calls to 
  // numeric::quadrature inside alg_expr;

  if has(alg_expr, hold(int)) then 
     alg_expr:= subs(alg_expr, hold(int) = numeric::quadrature);
  end_if;

  if numeric::indets([alg_expr, x0, x1]) minus {x} <> {} then
     // this could mean that there are symbolic parameters
     // in alg_expr. In this case, numeric::quadrature should
     // return symbolically.
     // However, f:= numeric::odesolve2(...) produces a list,
     // so one would have to call
     // numeric::quadrature(f(x)[1], x=a..b).
     // In this case numeric::indets(f(x)[1]) --> {f(x)[1]}.
     // --> we end up here.
     // Heuristically, just subtitute a float for x and see,
     // whether alg_expr produces a real float:
     if has(x0, infinity) and has(x1, infinity) then
        test_x:= frandom();
     elif (not has(x0, infinity)) and has(x1, infinity) then
        test_x:= x0 + frandom();
     elif has(x0, infinity) and not has(x1, infinity) then
        test_x:= x1 - frandom()
     else
        test_x:= x0+0.12345*(x1-x0);
     end_if:

     case domtype(float(subs(alg_expr,x=test_x, EvalChanges)))
     of DOM_FLOAT do
     of DOM_COMPLEX do break;
     otherwise return(procname(args()))
     end_case:
  end_if;

  // the symbolic abs is much too expensive to carry around!
  if has(alg_expr, hold(abs)) then 
     alg_expr:= subs(alg_expr, hold(abs) = numeric::lazyAbs);
  end_if;

  // ---------------------------------------------------
  // transformation for infinite intervals. Do not (!!!) split integral
  // into 2 parts, otherwise the adaptive strategy based on a first rough
  // estimate of the (entire!) integral may be unrealistic!

  if has([x0, x1], infinity) then
    // if the borders were given by expr(-infinity) = hold(_mult)(-1, infinity),
    // we need an eval to turn this into the stdlib::Infinity object -infinity.
    [x0, x1]:= eval([x0, x1]);
  end_if:
  if x0 = RD_NINF then x0:= -infinity end_if;
  if x1 = RD_INF then x1:= infinity end_if;
  if has([x0,x1], infinity) then
     // We need x0<x1:
     if x0 =  infinity then x0:=x1; x1:= infinity; alg_expr:= -alg_expr; end_if;
     if x1 = -infinity then x1:=x0; x0:=-infinity; alg_expr:= -alg_expr; end_if;
     if x0 = -infinity and (not has(x1, infinity)) then
       // transformation y=1/(x1+1-x), i.e., x=x1+1-1/y: 
       if iszero(x1) and traperror(subs(alg_expr, x = float(0),EvalChanges)) <> 0 then
         // int(.., x = -infinity..0) = int(.., -infinity..-1) + int(..,x=-1..0)
         // Do *not* substitute int(.., x = -1..0), because the 
         // substitution x = - 1 + 1/y may be unstable for y = 1.
         alg_expr:=  subs(alg_expr, x =-1/x)/x^2  // int(..,x=-infinity..-1)
                   + subs(alg_expr, x = - x);     // int(..,x=-1..0)
         x0:= 0; x1:= 1:
       else
         alg_expr:= subs(alg_expr,x=x1+1-1/x)/x^2;
         x0:=0: x1:=1: 
       end_if;
     end_if;
     if (not has(x0, infinity)) and x1 = infinity then
       // transformation y=1/(x-x0+1), i.e., x=x0-1+1/y: 
       if iszero(x0) and traperror(subs(alg_expr, x = float(0),EvalChanges)) <> 0 then
         // int(.., x = 0..infinity) = int(.., x=0..1) + int(..,x=1..infinity)
         // Do *not* substitute int(.., x = 0..1) via x = - 1 + 1/y,
         // because this may be unstable for y = 1.
         alg_expr:= alg_expr                     // int(.., x=0..1)
                  + subs(alg_expr, x = 1/x)/x^2; // int(.., x=1..infinity)
         x0:= 0; x1:= 1:
       else
         alg_expr:= subs(alg_expr, x=x0-1+1/x)/x^2; 
         x0:=0: x1:=1: 
       end_if;
     end_if;
     if x0 = -infinity and x1 = infinity
     then // split into int(..,-infinity..0)+int(..,0..infinity) 
          // transform as above and combine:
          alg_expr:= (subs(alg_expr, x=(x-1)/x)+ 
                      subs(alg_expr, x=(1-x)/x))/x^2;
          x0:=0: x1:=1: 
     end_if;
     if has([x0,x1], infinity) then // if x0:= I + infinity etc.
        error("cannot handle this infinite interval");
     end_if;
     userinfo(3, "integration transformed to quadrature(".
                 expr2text(alg_expr, x = x0..x1).")"):
  end_if;

  //---------------------------------------
  // check for singularities at the borders
  // (We must feed in float(x0), float(x1),
  //  because alg_expr may be hold(f)(x) with
  //  a procedure f that handles only float input
  //---------------------------------------
  leftsingularity:= FALSE;
  if traperror(subs(alg_expr, x = float(x0),EvalChanges)) <> 0 then
     leftsingularity := TRUE;
  end_if:
  rightsingularity:= FALSE;
  if traperror(subs(alg_expr, x = float(x1),EvalChanges)) <> 0 then
     rightsingularity := TRUE;
  end_if:

  // After we have checked for the option Adaptive, we will use
  // <left/right singularity> to stabilizize singularities at
  // the boundaries further down below.

  // ---------------------------------------------------
  // Check options:
  // ---------------------------------------------------

  adaptive:= TRUE;   // default
  freeStyle:= TRUE:  // default: use GL=20,40,80,160
  maxcalls:= 0: // initialize maxcalls, will be set below
  useAbsErr:= FALSE; // did the user specify AbsoluteError = tol?

  for opt in [args(3..args(0))] do
    if has(opt, Gauss) or has(opt, GaussLegendre) or has(opt, GL)
    then method:= GL: freeStyle:= FALSE;
         if type(opt)<>"_equal" then
            error("specify method by 'method name = number of nodes'"); 
         end_if:
         if not testtype((n:= op(opt,2)), Type::PosInt) then 
            error("number of nodes must be a positive integer");
         end_if;
         next;
    end_if;
    if has(opt, GaussTschebyscheff) or has(opt, GaussChebyshev) 
    or has(opt, GT) or has(opt, GC)
    then method:= GT: freeStyle:= FALSE; 
         if type(opt)<>"_equal" then
            error("specify method by 'method name = number of nodes'"); 
         end_if:
         if not testtype((n:= op(opt,2)), Type::PosInt) then 
            error("number of nodes must be a positive integer");
         end_if;
         next;
    end_if;
    if has(opt, NewtonCotes) or has(opt, NC) 
    then method:= NC: freeStyle:= FALSE;
         if type(opt)<>"_equal" then
            error("specify method by 'method name = number of nodes'"); 
         end_if:
         if not testtype((n:= op(opt,2)), Type::PosInt) then 
            error("number of nodes must be a positive integer");
         end_if;
         next;
    end_if;
    if has(opt, AbsoluteError) then
       if type(opt) <> "_equal" then
          error("use AbsoluteError = some positive real number"); 
       end_if;
       tol:= float(op(opt,2));
       if type(tol) <> DOM_FLOAT or tol <= 0 then
          error("illegal tolerance for the AbsoluteError. ".
                "It must be positive, got ".expr2text(tol));
       end_if;
       useAbsErr:= TRUE;
       next;
    end_if;
    if opt = (Adaptive=FALSE) then adaptive:= FALSE; next; end_if:
    if opt = (Adaptive=TRUE) then next; end_if:
    if opt = Adaptive then next; end_if:
    if has(opt, MaxCalls) then
         if type(opt)<>"_equal"
           then error("use MaxCalls = positive integer or infinity"); 
         end_if:
         maxcalls:= op(opt,2):
         if maxcalls = infinity then 
            // replace infinity by RD_INF: note that
            // bool(fcalls < RD_INF) is much faster than
            // bool(fcalls < infinity) !!!
            maxcalls:= RD_INF; 
            next;
         end_if;
         if not testtype(maxcalls, Type::PosInt) 
           then error("use MaxCalls = positive integer or infinity"); 
         end_if:
         next;
    end_if;
    error("unknown option");
  end_for:

  if not adaptive and freeStyle then
     error("need to specify non-adaptive quadrature method");
  end_if;
  if not freeStyle and method = GT and adaptive then
     warning("can use Gauss-Tschebyscheff method in ".
             "non-adaptive mode only. Switching to ".
             "GaussTschebyscheff=".expr2text(n).", Adaptive=FALSE ; ".
             "requested precision will not be achieved");
     adaptive:= FALSE;
  end_if:

  // ---------------------------------------------------
  // set default method, if not specified by user

  if freeStyle then
     method:=GL:
     adaptive:=TRUE:
     if   DIGITS<=10 then  n:=20; 
     elif DIGITS<=50 then  n:=40; 
     elif DIGITS<=100 then n:=80; 
     else n:=160;
     end_if;
  end_if;

  if maxcalls=0 then maxcalls:= MAXDEPTH*n: end_if;
  // ---------------------------------------------------

 //----------------------------------------------------------------
 // The data (weights b[i] and abscissae c[i]) are provided for    
 // Gauss-Legendre quadrature, for Gauss-Tschebyscheff-Quadrature  
 // and for Newton-Cotes quadrature:                               
 // See GLdata.mu , GTdata.mu and NCdata.mu                        
 //----------------------------------------------------------------

 if method=GL then ([b,c]):=numeric::gldata(n,DIGITS) end_if;
 if method=GT then ([b,c]):=numeric::gtdata(n) end_if;
 if method=NC then ([b,c]):=numeric::ncdata(n) end_if;
 ([b,c]):=float([b,c]);

 if adaptive then
  //--------------------------------------------------
  // numerical stabilization near border singularities
  //--------------------------------------------------
  if leftsingularity and 
     not iszero(x0) and 
     not rightsingularity then
     // singularity at the left border. Need to stabilize
     // numerical evaluation near x0 by mapping x0 to 0:
     alg_expr := subs(alg_expr, x = x0 + x);
     [x0, x1] := [0, x1 - x0];
     userinfo(3, "integration transformed to quadrature(".
                 expr2text(alg_expr, x = x0..x1).")"):
  end_if;
  if rightsingularity and 
     not iszero(x1) and 
     not leftsingularity then
     // singularity at the right border. Need to stabilize
     // numerical evaluation near x1 by mapping x1 to 0:
     alg_expr := subs(alg_expr, x = x1 - x);
     [x0, x1] := [0, x1 - x0]:
     userinfo(3, "integration transformed to quadrature(".
                 expr2text(alg_expr, x = x0..x1).")"):
  end_if:
  if leftsingularity and rightsingularity then
     // Split the integral and map each border
     // to 0 separately:
     // int(f(x),x=x0..x1) = 
     //   int(f(x), x=x0..(x1-x0)/2)
     // + int(f(x), x=(x1-x0)/2..x1) =
     //   int(f(x0+(x1-x0)/2*y),y = 0..1)
     // + int(f(x1-(x1-x0)/2*y),y = 0..1)
     alg_expr:= (subs(alg_expr, x = x0 + (x1 - x0)/2*x) + 
                 subs(alg_expr, x = x1 - (x1 - x0)/2*x)
                )*(x1 - x0)/2:
     [x0, x1]:= [0, 1]:
  end_if:
 end_if:

 //-----------------define local subroutine ------------------------------
 // Quadsum computes                                                      
 //  Quadsum( f(x),x=x0..x1):=(x1-x0)*sum(b[i]*f(x0+c[i]*(x1-x0)),i=1..n) 
 // based on the quadrature formula                                       
 //    int(f(c),c=0..1)==sum( b[i]*f(c[i]),i=1..n)                        
 // with abscissae  c[i]  and weights  b[i]                               
 //-----------------------------------------------------------------------

 Quadsum:=proc(alg_expr,x0,x1,c)
 local L, i, result;
 save DIGITS;
 begin // important: x0, x1 may be exact. Compute L:= float(x1-x0), not
       // L:= float(x1)-float(x0). The latter may become zero!
       L:=float(x1-x0);
       if iszero(L) then return(0);end_if;
       if fcalls>=maxcalls then return(0) end_if;
       fcalls:= fcalls + n;
       x0:= float(x0):

/* This is an attempt to avoid the costs of the symbolic
   evaluation of alg_expr. Unfortunately, it does not
   work, if alg_expr=hold(some_function)(x)
       L*_plus(b[i]*float(eval(subsop(hold(subs(ALG_EXPR,X=FLOATX)),
                                      1 = alg_expr,
                                      [2, 1]= x,
                                      [2, 2]= x0 + c[i]
                                      )
                               ))$ i=1..n);
*/

   if traperror((result :=
          L*_plus(b[i]*float(subs(alg_expr,x=x0+c[i],EvalChanges))$ i=1..n);
      )) = 0 then
        return(result)
   else // the error might have been caused by a numerically
        // unstable formulation of the input (e.g.,
        // quadrature(arctanh(1 - x^2), x = 0..1)), which
        // hits the singularity arctanh(1) for small x due
        // do cancellation. Increase DIGITS to avoid the
        // cancellation.
        DIGITS:= 2*DIGITS;
        return(
           L*_plus(b[i]*float(subs(alg_expr,x=x0+c[i],EvalChanges))$ i=1..n)
        ):
   end_if;
 end_proc:

/*------------- define local subroutine ------------------------------------
 adaptiveQuad is a general adaptive algorithm based on a non-adaptive 
 quadrature rule Quadsum:
   int(f(x),x=0..1)==sum(b[i]*f(c[i],i=1..n)=:Quadsum(f,x=0..1)
 given by the quadrature data b[i] (weights) and c[i] (abscissae).
 First,  an estimate  Is:= Quadsum(f,x=x0..x1) is computed.
 Then, by recursive bisectioning of the interval, the method Quadsum is used 
 to compute the integrals over each half of the interval.
 Bisectioning is stopped, when  
    Quadsum(first half)+ Quadsum(second half)-Quadsum(complete interval)
 has become so small that this difference cannot be added to the estimate
 Is within float precision.
 Note:  let  p be the order of the quadrature rule, i.e.
    Error( Quadsum( f,  x=x0..x1 ) )  =  constant( f ) * (x1-x0)^p
 Then bisectioning will occur down to subintervals of length L, until  
       constant(f) * L^p == Is * 10^(-DIGITS)
 holds,  i.e.,  L = O( 10^(-DIGITS/p ) :  the costs grow
 exponentially with DIGITS !
 For very low precision low order methods Quadsum suffice.  
 FOR HIGH PRECISION A HIGH ORDER METHOD Quadsum IS NECESSARY !
 A reasonable (heuristic) choice seems Gauss-Legendre-Quadrature
 with n nodes which has order (2n+1). 
 Example: n=20, for which the abscissae c[j] and weights b[j]
 can be computed in a fraction of a second, if DIGITS is not too
 high. Each bisection step then produces quadrature values with errors
 whose sum is approximately the error of the quadrature on the double 
 interval multplied by a factor 
     2^(-40)==10^(-12)
 Thus, for slowly varying integrands, approximately DIGITS/12
 bisection steps are needed to reduce the error of Quadsum(whole interval)
 by DIGITS decimal places. This leads to
   2^(DIGITS/12)
 subintervals, each requiring n=20 evaluations of the integrand.
---------------------------------------------------------------------*/
  // Bisect interval x0..x1. Compare sum of quadratures
  // I21 = int(..,x=x0..xmid) and I22 = int(..,x=xmid..x1) with
  // I1 = int(..,x=x0..x1). Stop bisectioning, if no
  // significant change between I2=I21+I22 and I1.

  adaptiveQuad:=proc(alg_expr,x0,x1,c,bisectionlevel,I1)
     local mid,I21,I22,I2;
  begin
     // The following line passes the information, that another
     // call to adaptiveQuad has reached MAXDEPTH. Stop integrating
     // in all calls of adaptiveQuad:
     if problem then return(I1); end_if;
     if bisectionlevel>MAXDEPTH-10 then
        if interactive then
          warning("Precision goal not achieved after MAXDEPTH=".
                   expr2text(MAXDEPTH)." recursive calls!\n".
                  "There may be a singularity of ".expr2text(alg_expr).
                  " close to ".expr2text(x)."=".expr2text(float((x0+x1)/2)).".\n".
                  "Increase MAXDEPTH and try again for a more accurate result!");
        end_if;
        problem:= TRUE:
        return(I1);
     end_if;
     // use exact arguments x0, x1 for Quadsum and adaptiveQuad:
     mid:=(x0+x1)/2; 
     c:=map(c,_mult,0.5);
     I21:=Quadsum(alg_expr,x0,mid,c); 
     I22:=Quadsum(alg_expr,mid,x1,c);
     I2:=I21+I22;

     if has(I2, RD_INF) or
        has(I2, RD_NINF) or
        has(I2, RD_NAN) then
        error("numerical exception");
     end_if:

     // use global estimate Is of integral for stopping criterion
     if specfunc::abs(I2-I1) <= (specfunc::abs(Is)
                                +specfunc::abs(I1)
                                +specfunc::abs(I2))*macheps
     or fcalls>=maxcalls
     then return(I2) 
     else //-------------------------------------------------------------
          // The bisectionlevel gives the length of the present bisected 
          // interval = original interval / 2^(bisectionlevel)           
          //-------------------------------------------------------------
          // Call right half of interval first !! Note that infinite
          // intervals are mapped to int(f(1/x)/x^2,x=0..1). In this 
          // case the right half converges faster than the left half.
          // Make sure that right half produces good results, before
          // the left half produces fcalls>=maxcalls and stops the 
          // This way the inexact results returned with warnings are
          // still reasonably good!!

          return(adaptiveQuad(alg_expr,mid,x1,c,bisectionlevel+1,I22)
                +adaptiveQuad(alg_expr,x0,mid,c,bisectionlevel+1,I21));
     end_if;
  end_proc:
  //----------- end of adaptiveQuad ----------------------------

  // --------------------------------------------------------
  // The following code is experimental: instead of
  // inf(f(x), x = a .. b), consider the symmetrized
  // integral 
  //      int(f(x) + f(a+b-x), x = a..b) 
  // = 2* int(f(x) + f(a+b-x), x = (a+b)/2..b) 
  // = 2* int(f(x), x=a..b)
  // Advantage:
  // ----------
  //     This would automatically take into account symmetry/skew symmetry:
  //     If f(x) = f(-x):
  //         numeric::int(f(x), x = -b..b) = 2*numeric::int(f(x), x=0..b)
  //     If f(x) = -f(-x):
  //         numeric::int(f(x), x = -b..b) = numeric::int(0, x=0..b) = 0.0
  //     Further, the integration interval is halfed making it easier
  //     to reach the precision goal.
  // Disdvantage:
  // ------------
  //     The costs would be roughly doubled because the integrand
  //     twice as complex.
  // --------------------------------------------------------
  // if adaptive then
  //   alg_expr:= alg_expr + subs(alg_expr, x = x0 + x1 - x):
  //   x0:= (x0 + x1)/2;
  // end_if;
  // --------------------------------------------------------

  // now do the quadrature: 
  interactive:= stdlib::interactive();
  problem:= FALSE:
  // fcalls is the number of calls to the integrand
  fcalls:= 0; // each call to Quadsum increases fcalls by n
  alg_expr:=float(alg_expr);
  macheps:=float(10^(-DIGITS));
  c:=map(c,_mult,float(x1-x0));
  // In the adaptive mode, the precision goal is determined by
  // the rough estimate Is of the final result:
  if useAbsErr then
       Is:= tol/macheps; // non-adaptive estimate specified by the user
  else Is:= Quadsum(alg_expr,x0,x1,c); // non-adaptive rough estimate
       if has(Is, RD_INF) or
          has(Is, RD_NINF) or
          has(Is, RD_NAN) then
          error("numerical exception");
       end_if:
       // use global estimate Is of integral for stopping criterion
       if not adaptive then
         return(Is);
      end_if;
  end_if;
  userinfo(2, "rough estimate = ".expr2text(Is));
  for dummy from 1 to 2 do
    // 2 tries: if (a posteriori) the rough estimate Is
    // turns out to be of the correct order of magnitude,
    // return the adaptive value.
    // Otherwise, try again (dummy = 2) with the rough
    // estimate replaced by the more accurate value
    // computed in the first go (dummy = 1).
    Is0:= Is;
    if adaptive then // improve first estimate
       userinfo(2, "starting adaptive quadrature on the interval ".expr2text(x0..x1));
       Is:= adaptiveQuad(alg_expr,x0,x1,c,0,Is);
    end_if;
    if fcalls>=maxcalls then
       if interactive then
          warning("Precision goal not achieved after ".
                  expr2text(fcalls). " function calls!\n".
                  "Increase MaxCalls and try again for a more accurate result.");
       end_if;
       problem:= TRUE;
    end_if:
    if not interactive and problem then
        userinfo(1, "result after ".expr2text(fcalls)." function calls: ".
                expr2text(FAIL));
        return(FAIL)
    end_if;
    userinfo(1, "result after ".expr2text(fcalls)." function calls: ".
                expr2text(Is));
    if dummy=1 and specfunc::abs(Is)< specfunc::abs(Is0/100) then
       next; // try again
    end_if;
    return(Is);
  end_for:
end_proc:
//----------- end of Quadrature ----------------------------

