//---------------------------------------------------------
// This is a highly dubious function (requested by Schulefraktion)
// that employs some heuristics to **try to** find all real roots
// of a function (or an equation).
// The returned list contains multiple zeroes 
// according to their multiplicity.
//
// MyComment: Ich garantiere fuer gar nix (walter)
// 
//----------------------------------------------------------
// Calls:  numeric::allRealRoots(f, <x = a..b>, <NoWarning>)
//
// Returns: a list of real roots (float values)
//
// Parameters:
//    f    : an univariate expression or an equation
//    a..b : the search range; a, b may be finite or
//           +/- infinity. Default is -infinity..infinity
//    NoWarning: suppresses warnings about problems with
//               root isolation via interval arithmetic
//
// Example:
//   numeric::allRealRoots(x^10 = exp(x))
//
//               [-0.9127652716, 1.118325592, 35.77152064]
//
//   numeric::allRealRoots(sin(x) = exp(x), x = -20..20)
//
//               [-9.424858654, -6.281314366, -3.183063012]
//---------------------------------------------------------

numeric::allRealRoots:= proc(f, searchrange)
local x, searchrange_, is_poly, pf, coefftypes, eps,
      forig, rootintervals, problem, roots, r, xx,
      zero_counted, Arg, doWarn, 
      i, j, ff, multiplicity;
begin
  if args(0) < 1 then
     error("expecting at least one argument");
  end_if;

  if type(f) = "_equal" then
     f:= op(f, 1) - op(f, 2);
  end_if; 
  f:= expr(f); // in case of Factored objects
  forig:= f;

  x:= numeric::indets(f):

  if nops(x) = 0 then
     x:= `#x`;
  end_if;

  if nops(x) > 1 then 
     error("the function/equation may contain only one indeterminate. ".
           "Found the indeterminates ".expr2text(op(x)));
  end_if:

  x:= op(x):
  doWarn:= TRUE; // default
  searchrange_:= -infinity .. infinity; // default
  for i from 2 to args(0) do
     Arg:= args(i);
     if has(Arg, NoWarning) then
        doWarn:= FALSE;
        next;
     end_if;
     if type(Arg) <> "_equal" or
        type(op(Arg, 2)) <> "_range" then
        error("expecting a search range of the form  x = a..b");
     end_if;
     if x <> op(Arg, 1) and x <> `#x` then 
        error("the specified unknown is ".expr2text(op(Arg, 1)).", but ".
              "the function depends on ".expr2text(x));
     end_if:
     searchrange_:= op(Arg, 2);
  end_for;

  f:= numer(f);
  is_poly:= testtype(f, Type::PolyExpr(x));

  userinfo(1, "searching for roots of ".expr2text(f));

  // precision goal for numeric::realroots
  eps:= 10^(2 - DIGITS);

  //-----------------------------
  // The polynomial case
  //-----------------------------
  if is_poly then
     // use pf = poly(f) only to determine the coefficient types,
     // but do not pass it to the solver (poly expands!)
     case type(f)
     of "_mult" do
        coefftypes:= {}:
        for i from 1 to nops(f) do
          if not has(op(f, i), x) then
             next;
          end_if;
          pf:= poly(expr(op(f, i)), [x]);
          coefftypes:= coefftypes union map({coeff(pf)}, domtype);
        end_for:
        break;
     of "_power" do
        pf:= poly(expr(op(f, 1)), [x]);
        coefftypes:= map({coeff(pf)}, domtype);
        break;
     otherwise
        pf:= poly(expr(f), [x]);
        coefftypes:= map({coeff(pf)}, domtype);
     end_case;
     if coefftypes minus {DOM_INT, DOM_RAT} = {} then
        // if the coefficients are integers/rationals, then
        // numeric::realroots is just perfect (up to the fact
        // that multiple roots are counted only once). We need
        // to take care of the multiplicities:
        f:= polylib::sqrfree(f):
        userinfo(1, "result of squarefree factorization: ".expr2text(f));
        f:= Factored::convert_to(f, DOM_LIST);
        roots:= table():
        for i from 1 to (nops(f)-1)/2 do
            ff:= f[2*i];
            multiplicity:= f[2*i + 1];
            userinfo(1, "searching for roots of factor ".expr2text(ff).
                        " via numeric::realroots"):
            rootintervals:= numeric::realroots(ff, x = searchrange_,
                                               eps, Merge = FALSE);
            r:= map(rootintervals, r -> (r[1]+r[2])/2);
            userinfo(1, "following roots found: ".expr2text(r));
            for xx in r do
              for j from 1 to multiplicity do
                roots[nops(roots) + 1]:= xx;
              end_for:
            end_for:
         end_for:
        roots:= sort([roots[i] $ i = 1 .. nops(roots)]);
     else
        // if the coefficients are not integers/rationals, then
        // numerical::realroots applies float which may change the
        // roots drastically. Call numeric::polyroots instead, which often
        // yields much better numerical roots
        userinfo(1, "searching for polynomial roots via numeric::polyroots"):
        // get rid of non-real roots
        roots:= map(numeric::polyroots(f/*, Factor*/), numeric::complexRound);
        roots:= select(sort(roots), x -> domtype(x) = DOM_FLOAT);
        if op(searchrange_, 1) <> -infinity then
           roots:= select(roots, x -> bool(x >= op(searchrange_, 1)));
        end_if:
        if op(searchrange_, 2) <> infinity then
           roots:= select(roots, _leequal, op(searchrange_, 2));
        end_if:
     end_if;
     // get rid of non-real entries
     roots:= map(roots, numeric::complexRound);
     roots:= select(roots, testtype, DOM_FLOAT);
     // Check if the roots corrspond to some sort of singularity
     // (e.g., r = 0.0 is not a valid root for sin(x)/x or sin(x)*ln(x)/cos(x)). 
     for i from 1 to nops(roots) do
        if traperror(subs(forig, x = op(roots, i), EvalChanges)) <> 0 then
           // this does not look like a root
           roots:= subsop(roots, i = NIL);
        end_if;
     end_for:
     roots:= subs(roots, NIL = null(), EvalChanges);
     return(roots):
  end_if;

  //-----------------------------
  // The non-polynomial case
  //-----------------------------

  roots:= table():
  userinfo(1, "trying to factor");
  f:= factor(f):
  userinfo(1, "result of factorization: ".expr2text(f));
  f:= Factored::convert_to(f, DOM_LIST);
  for i from 1 to (nops(f)-1)/2 do
      ff:= f[2*i];
      multiplicity:= f[2*i + 1];
      if multiplicity < 0 then
         // this happens for the numerator of f
         next;
      end_if;

      userinfo(1, "searching for roots of factor ".expr2text(ff)):
      //-------------------------------------
      // Special case: the factor ff is a polynomial
      //-------------------------------------
      if testtype(ff, Type::PolyExpr(x)) then
      // userinfo(1, "calling numeric::allRealRoots for factor ".expr2text(ff));
         for xx in numeric::allRealRoots(ff, args(2..args(0))) do
           for j from 1 to multiplicity do
               roots[nops(roots) + 1]:= xx;
           end_for:
         end_for:
         next; // process the next factor
      end_if:

      //----------------------------------------------
      // First, compute search intervals for the roots
      // Heuristics: try numeric::realroots with a modest
      // precision goal of 0.01 and check whether the
      // interval isolation looks difficult (by comparing
      // Merge = FALSE and Merge = TRUE). If this indicates
      // no problem, then use the small precision goal
      // 10^(-DIGITS)
      //----------------------------------------------
      userinfo(1, "searching for non-polynomial roots via numeric::realroots"):
      if eps > 0.01 then
         rootintervals:= numeric::realroots(ff, x = searchrange_,
                                            eps, Merge = FALSE); 
      else
         rootintervals:= numeric::realroots(ff, x = searchrange_,
                                            0.01, Merge = FALSE); 
         // In the following, we check whether there are 3 adjacent
         // intervals that could be merged to a larger interval.
         // Looking for only 2 adjacent interval may not be a good
         // criterion: If one of the division points happens to be
         // close to a root, there will be 2 adjacent intervals. 
         // This does not indicate an isolation problem!
         problem:= FALSE;
         for j from 3 to nops(rootintervals) do
            if 
                specfunc::abs(  op(rootintervals[j-2],2) 
                              - op(rootintervals[j-1],1)) <= 
                10^(-DIGITS)*max(1, abs(op(rootintervals[j-1],1)))
            and 
                specfunc::abs(  op(rootintervals[j-1],2) 
                              - op(rootintervals[ j ],1)) <= 
                10^(-DIGITS)*max(1, abs(op(rootintervals[ j ],1)))
            then
               problem:= TRUE;
               break;
            end_if;
         end_for:

         if problem then
            if doWarn then
               warning("problem in isolating search intervals. Some roots may be lost"):
            end_if;
            // do not switch to the precision goal eps = 10^(2 - DIGITS)
         else
            // search again with precisiongoal eps
            rootintervals:= numeric::realroots(ff, x = searchrange_,
                                               eps, Merge = FALSE); 
         end_if;
      end_if;
      userinfo(1, "search intervals: ".expr2text(rootintervals));

      //----------------------------------------------
      // Second, use numeric::fsolve (for infinite intervals)
      // and numeric::realroot (for finite intervals) to check,
      // whether there really is a root in the search interval
      //----------------------------------------------
      // Due to enlargement of small intervals around the origin, 
      // there is danger of finding 0.0 in different rootintervals.
      // Make sure, 0.0 is counted only once:
      zero_counted:= FALSE;

      // search the rootintervals:
      for r in rootintervals do
         userinfo(2, "searching in search interval ".expr2text(r));
         //------------------
         // infinite interval
         //------------------
         if has(r, infinity) then
           // numeric::fsolve yields  xx = [x = floatvalue] or xx = FAIL 
           if traperror(( xx := numeric::fsolve(ff, x = r[1]..r[2], 
                                                RestrictedSearch))) = 0 and
              xx <> FAIL then
              userinfo(2, "root found: ".expr2text(xx));
              for j from 1 to multiplicity do
                  roots[nops(roots) + 1]:= op(xx[1], 2);
              end_for:
           end_if;
           userinfo(2, "no root found");
           next;
         end_if;
         //----------------
         // finite interval
         //----------------
         // numeric::realroot has problems with tiny search intervals.
         // Enlarge the search interval a bit:
         if specfunc::abs(r[1]) < 10.0^(-100*DIGITS) then
            r[1]:= specfunc::sign(r[1])*10.0^(-100*DIGITS);
         end_if;
         if specfunc::abs(r[2]) < 10.0^(-100*DIGITS) then
            r[2]:= specfunc::sign(r[2])*10.0^(-100*DIGITS);
         end_if;
         r[1]:= r[1]*(1 - specfunc::sign(r[1])* 10.0^(- DIGITS)); 
         r[2]:= r[2]*(1 + specfunc::sign(r[2])* 10.0^(- DIGITS)); 
         // numeric::realroot yields  xx = floatvalue or xx = FAIL 
         if traperror((xx := numeric::realroot(ff, x = r[1]..r[2]))) <> 0 or
            xx = FAIL then
            userinfo(2, "no root found");
            next;
         end_if;

         // check if we found a dubious point such as a sign changing
         // pole or any other sort of singularity:

        if traperror(subs(forig, x = xx, EvalChanges)) <> 0 then
           // this does not look like a root
           next;
        end_if;

        userinfo(2, "root found: ".expr2text(xx));
        if iszero(xx) then
           if zero_counted then
              next; // do not insert float(0.0) again
           end_if;
           zero_counted:= TRUE;
        end_if;
        // If the current root coincides with the last root,
        // then do not count it!
        if nops(roots) > 0 and 
           specfunc::abs(roots[nops(roots)] - xx) <= 
                  10^(-DIGITS)*specfunc::abs(xx) then
           next;
        end_if;
        for j from 1 to multiplicity do
            roots[nops(roots) + 1]:= xx;
        end_for:

      end_for:
  end_for:
  roots:= [roots[i] $ i = 1 .. nops(roots)];
  // get rid of non-real entries
  roots:= map(roots, numeric::complexRound);
  roots:= select(roots, testtype, DOM_FLOAT);
end_proc:
