/*----------------------------------------------------------------
  numeric::cubicSpline2d(
    X       : DOM_LIST,  // list of length n+1
    Y       : DOM_LIST,  // list of length m+1
    values  : DOM_ARRAY  // array(0..n, 0..m)
              or DOM_HFARRAY // hfarray(0..n, 0..m)
   <[xBoundaryType, yBoundaryType> 
                   // BoundaryType: 'NotAKnot' or 'Natural' or
                   // 'Periodic' or 'Complete = [p1, p2, ...]'
   <Symbolic>      // do not convert the data to floats
  )

Return value: a procedure S. 

The spline function S can be called in the following ways:
 *)  S(x, y)  with numerical values x, y:
              returns the value of the spline 
 *)  S(x, y, i, j)  with symbolic or numerical values x, y:
              returns the value of the spline assuming x, y
              to lie in patch (i, j)
 *)  S(x, y, [xx, yy])  with numerical values x, y and
              nonnegative integers xx, yy: returns the
              value of diff(S(x, y), x $ xx, y $ yy)
 *)  S(x, y, i, j, [xx, yy]) with symbolic or numerical values x, y:
              returns the value of diff(S(x, y), x $ xx, y $ yy)
              assuming x, y to lie in patch (i, j)

Some References:

J Bellman,  R., Kashev, B. G., Vasudevan, R.;
Dynamic programming and bicubic spline interpolation;
JMAA; 44; 1973; 160--174;

Birkhoff,  G.;
Piecewise bicubic interpolation and approximation in polygons;
adison 185--221;

Boor,  C. de;
Bicubic spline interpolation;
J. Math.\ Phys.;  41; 1962; 212--218;

41 TBT1787-2037 Hermite-Spline-Interpolation fr Funktionen von zwei
                Variablen / Egon Scheffold
41 TKY1885      Spline functions and multivariate interpolations / by
                B. D. Bojanov, H. A. Hakopian and A. A. Sahakian
41 TKY1720      Zweidimensionale Spline-Interpolations-Algorithmen /
                von Helmuth Spth
-----------------------------------------------------------------*/

numeric::cubicSpline2d := proc()
local xBC, yBC, symbolic, xArray, yArray, 
      z, yp0, ypm, xp0, xpn, Syx0, Syxn, A, hx, hy, hSet,
      n, m, mm, i, j, k, b, c, d, indexsearch;
option escape;
begin
   if args(0) < 3 then
      error("expecting at least 3 arguments"):
   end_if;

   // set the default modes:
   [xBC, yBC]:= [NotAKnot, NotAKnot]:
   symbolic:= FALSE:

   if domtype(args(1)) = DOM_LIST then
      n:= nops(args(1))-1;
      xArray:= array(0..n, args(1));
   else error("expecting a list of nodes as first argument")
   end_if;

   if domtype(args(2)) = DOM_LIST then
      m:=nops(args(2))-1;
      yArray:= array(0..m,args(2));
   else error("expecting a list of nodes as second argument")
   end_if;

   //-------------------------------------------------
   // check the data array args(3)
   //-------------------------------------------------
   if domtype(args(3)) <> DOM_ARRAY and
      domtype(args(3)) <> DOM_HFARRAY then
        error("expecting an array or an hfarray as third argument");
   elif op(args(3), [0, 1]) <> 2 then
        error("expecting a 2-dimensional array as third argument");
   elif op(args(3), [0, 2, 1]) <> 0 then
        error("the indices of the data array must start from 0");
   elif op(args(3), [0, 2, 2]) <> n then
        error("found ".expr2text(n+1)." x-nodes, ". 
              "expecting the data to be provided by an ".
              "array(0..". expr2text(n). ", 0..m). Received an ".
              "array(".expr2text(op(args(3), [0, 2]),op(args(3), [0,3])).")");
   elif op(args(3), [0, 3, 1]) <> 0 then
        error("the indices of the data array must start from 0");
   elif op(args(3), [0, 3, 2]) <> m then
        error("found ".expr2text(m+1)." y-nodes, ". 
              "expecting the data to be provided by an ".
              "array(0..n, 0..". expr2text(m). "). Received an ".
              "array(".expr2text(op(args(3), [0, 2]),op(args(3), [0,3])).")");
   end_if;
   z:= args(3);

   //-----------------------------------------------------
   // the grid data are read. Now, search for the options:
   //-----------------------------------------------------

   // store left complete boundary data in x direction in xp0
   xp0:= array(0..m, [0 $ m+1]): // initialize to avoid warnings
   // store right complete boundary data in x direction in xpn
   xpn:= array(0..m, [0 $ m+1]): // initialize to avoid warnings
   // store left complete boundary data in y direction in yp0
   yp0:= array(0..n, 0..3, [[0 $ 4] $ n+1]): // initialize to avoid warnings
   // store right complete boundary data in y direction in ypm
   ypm:= array(0..n, 0..3, [[0 $ 4] $ n+1]): // initialize to avoid warnings

   for i from 4 to args(0) do
       if args(i) = Symbolic then 
          symbolic:= TRUE;
          next;
       end_if;
       if domtype(args(i)) = DOM_LIST then
          if nops(args(i)) <> 2 then 
             error("expecting a list with 2 boundary types ".
                   "such as [NotAKnot, Natural], received ".
                   expr2text(args(i)));
          end_if;
          [xBC, yBC]:= args(i);
          if {NotAKnot, Natural, Periodic} 
             minus {xBC, yBC} = {} then
             next;
          end_if;
          if has(xBC, Complete) then
             if type(xBC) <> "_equal" then
                error("expecting a specification ".
                      "'Complete = [left boundary data, right boundary data]'");
             end_if;
             [xBC, xp0]:= [op(xBC)];
             if domtype(xp0) <> DOM_LIST or nops(xp0) <> 2 then
                error("expecting a specification ".
                      "'Complete = [left boundary data, right boundary data]'");
             end_if;
             [xp0, xpn]:= xp0;
             if domtype(xp0) <> DOM_LIST then
                error("expecting the left 'Complete' boundary data ".
                      "in x direction as a list");
             end_if;
             if domtype(xpn) <> DOM_LIST then
                error("expecting the right 'Complete' boundary data ".
                      "in x direction as a list");
             end_if;
             if nops(xp0) <> m+1 then
                error("expecting ".expr2text(m+1)." left 'Complete' boundary data ".
                      "in x direction, received ".expr2text(nops(xp0)));
             end_if;
             if nops(xpn) <> m+1 then
                error("expecting ".expr2text(m+1)." right 'Complete' boundary data ".
                      "in x direction, received ".expr2text(nops(xpn)));
             end_if;
             xp0:= array(0..m, xp0);
             xpn:= array(0..m, xpn);
          end_if;            

          if has(yBC, Complete) then
             if type(yBC) <> "_equal" then
                error("expecting a specification ".
                      "'Complete = [left boundary data, right boundary data]'");
             end_if;
             [yBC, yp0]:= [op(yBC)];
             if domtype(yp0) <> DOM_LIST or nops(yp0) <> 2 then
                error("expecting a specification ".
                      "'Complete = [left boundary data, right boundary data]'");
             end_if;
             [yp0, ypm]:= yp0;
             if domtype(yp0) <> DOM_LIST then
                error("expecting the left 'Complete' boundary data ".
                      "in y direction as a list");
             end_if;
             if domtype(ypm) <> DOM_LIST then
                error("expecting the right 'Complete' boundary data ".
                      "in y direction as a list");
             end_if;
             if nops(yp0) <> n+1 then
                error("expecting ".expr2text(n+1)." left 'Complete' boundary data ".
                      "in y direction, received ".expr2text(nops(yp0)));
             end_if;
             if nops(ypm) <> n+1 then
                error("expecting ".expr2text(n+1)." right 'Complete' boundary data ".
                      "in y direction, received ".expr2text(nops(ypm)));
             end_if;
             yp0:= array(0..n, 0..3, [[yp0[i+1], 0, 0, 0] $ i=0..n]);
             ypm:= array(0..n, 0..3, [[ypm[i+1], 0, 0, 0] $ i=0..n]);
          end_if;            
          // Complete boundary data are identified and stored in the
          // arrays xp0, xpn (of length 0..m), yp0, ypm (of length 0..n)
          next;
       end_if;
       error("unknown option, received ".expr2text(args(i)));
   end_for:

   if not symbolic then
      xArray:= map(xArray, float):
      yArray:= map(yArray, float):
      z:= map(z, float):
      if xBC = Complete then
         xp0:= map(xp0, float);
         xpn:= map(xpn, float);
      end_if;
      if yBC = Complete then
         yp0:= map(yp0, float);
         ypm:= map(ypm, float);
      end_if;
   end_if;

   (hx[i]:= xArray[i+1] - xArray[i];) $ i=0..n-1;
   (hy[i]:= yArray[i+1] - yArray[i];) $ i=0..m-1;

   if n = 0 then hx[0]:= 0; end_if; // just to avoid warnings. These
   if m = 0 then hy[0]:= 0; end_if; // values will never be used.

   //----------- check the x nodes -------------
   if not symbolic then
      // if symbolic, the nodes are *assumed* to be ordered
      hSet:= {hx[i] $ i = 0..n-1}:
      if n > 0 then
         if map(hSet,domtype) <> {DOM_FLOAT}
            then error("use the option 'Symbolic' for symbolic nodes");
         end_if;
         if map(hSet, x -> bool(x > 0)) <> {TRUE} then
            error("the x-nodes are not in ascending order");
         end_if;
         if map({op(hx)}, _not@iszero@op, 2)<> {TRUE} then
            error("the node coordinates must be distinct");
         end_if;
      end_if;
   end_if;
   //----------- check the y nodes -------------
   if not symbolic then
      // if symbolic, the nodes are *assumed* to be ordered
      hSet:= {hy[i] $ i = 0..m-1}:
      if m > 0 then
         if map(hSet,domtype) <> {DOM_FLOAT}
            then error("use the option 'Symbolic' for symbolic nodes");
         end_if;
         if map(hSet, x -> bool(x > 0)) <> {TRUE} then
            error("the y-nodes are not in ascending order");
         end_if;
         if map({op(hy)}, _not@iszero@op, 2)<> {TRUE} then
            error("the node coordinates must be distinct");
         end_if;
      end_if;
   end_if;

   /* -----------------------------------------------------------
    On the interval x[i] <= x < x[i+1], y[j] <= y < y[j+1},
    the 2 dimensional spline S is given by
      S(x, y) =
      A[i,j,0,0]     + A[i,j,0,1]    *Y + A[i,j,0,2]    *Y^2 + A[i,j,0,3]    *Y^3
    + A[i,j,1,0]*X   + A[i,j,1,1]*X  *Y + A[i,j,1,2]*X  *Y^2 + A[i,j,1,3]*X  *Y^3
    + A[i,j,2,0]*X^2 + A[i,j,2,1]*X^2*Y + A[i,j,2,2]*X^2*Y^2 + A[i,j,2,3]*X^2*Y^3
    + A[i,j,3,0]*X^3 + A[i,j,3,1]*X^3*Y + A[i,j,3,2]*X^3*Y^2 + A[i,j,3,3]*X^3*Y^3
    where X = x - x[i], Y = y - y[j].
    Fact:
    0)   S, S_x, S_y, S_xx, S_xy, S_yy are continuous at the nodes (x[i], y[j]).
    The algorithm:
    1)   S(x, y[j]) =
              A[i,j,0,0] + A[i,j,1,0]*X + A[i,j,2,0]*X^2 + A[i,j,3,0]*X^3
         is the 1-dimensional spline through the data 
         (x[i], A[i,j,0,0]) with i = 0..n and j fixed. 
         Thus, the data A[i,j,0,0], A[i,j,1,0], A[i,j,2,0], A[i,j,3,0],
         are determined by m + 1 1-dimensional spline interpolations along the
         first parameter lines.
    2.0) S(x[i], y) =
              A[i,j,0,0] + A[i,j,0,1] *Y + A[i,j,0,2] *Y^2 + A[i,j,0,3] *Y^3
         is the 1-dimensional spline through the data (y[j], A[i,j,0,0]) 
         with j = 0..m and i fixed. 
         Thus, the data A[i,j,0,1], A[i,j,0,2], A[i,j,0,3] can be computed
         by n + 1 spline interpolations along the second parameter lines.
    2.1) Sx(x[i], y) = 
             A[i,j,1,0] + A[i,j,1,1]*Y + A[i,j,1,2]*Y^2 + A[i,j,1,3]*Y^3
         is the 1-dimensional spline through the data (y[j], A[i,j,1,0]) 
         with j = 0..m and i fixed. Note that A[i,j,1,0] was compute in Step 1)
         Thus, the data A[i,j,1,1], A[i,j,1,2], A[i,j,1,3] can be computed
         by n + 1 spline interpolations along the second parameter lines.
    2.2) Sxx(x[i], y)/2 = 
             A[i,j,2,0] + A[i,j,2,1]*Y + A[i,j,2,2]*Y^2 + A[i,j,2,3]*Y^3
         is the 1-dimensional spline through the data (y[j], A[i,j,2,0]) 
         with j = 0..m and i fixed. Note that A[i,j,2,0] was compute in Step 1).
         Thus, the data A[i,j,2,1], A[i,j,2,2], A[i,j,2,3] can be computed
         by n + 1 spline interpolations along the second parameter lines.
    2.3) Sxxx(x[i], y)/6 = 
             A[i,j,3,0] + A[i,j,3,1]*Y + A[i,j,3,2]*Y^2 + A[i,j,3,3]*Y^3
         is the 1-dimensional spline through the data (y[j], A[i,j,3,0]) 
         with j = 0..m and i fixed. Note that A[i,j,3,0] was compute in Step 1).
         Thus, the data A[i,j,3,1], A[i,j,3,2], A[i,j,3,3] can be computed
         by n + 1 spline interpolations along the second parameter lines.

    The following code is a direct implementation of the steps 1), 2.0) .. 2.3)
   ----------------------------------------------------------- */

   //--------------------------------------------
   // initialize the array A (coefficient matrix)
   //--------------------------------------------
   A := array(0..n,0..m,0..3,0..3);

   for i from 0 to n do 
     for j from 0 to m do
       A[i,j,0,0] := z[i,j]; 
     end_for;
   end_for;  

   /* ----------------------------------------------------
      Step 1): compute 1 dimensional spline data along the 
      1st parameter lines. Store these data in the array A
      that holds the 2 dimensional spline data.

      The utility numeric::_cubicSpline computes the data
      of 1-dimensional splines.
   -----------------------------------------------------*/
   for j from 0 to m do
      // compute the spline data along the 1st parameter line
      // with fixed 2nd parameter index j
      // We pass xp0[j], xpn[j] in any case. These data won't be
      // used unless xBC = Complete:
 
      [b, c, d]:= numeric::_cubicSpline(
                  xBC, n, hx, [z[i, j] $ i=0..n], xp0[j], xpn[j]):
 
      // For the interpolations along the second parameter lines,
      // we need the following data with i=n. So we have to
      // extrapolate b[i], c[i], d[i] from i=n-1 to i=n
 
      if n > 0 then
        b[n]:= b[n-1] + 2*c[n-1]*hx[n-1] + 3*d[n-1]*hx[n-1]^2;
        c[n]:= c[n-1] + 3*d[n-1]*hx[n-1];
        d[n]:= d[n-1];
      end_if;
      (A[i, j, 1,0] := b[i];) $ i = 0..n;
      (A[i, j, 2,0] := c[i];) $ i = 0..n;
      (A[i, j, 3,0] := d[i];) $ i = 0..n;
   end_for:

   // For Complete boundary conditions in the y direction,
   // we also need interpolated values of the first y derivative
   if yBC = Complete then
       //-------------------------------------------------
       if xBC = Complete and n > 0 then
          // we need complete boundary conditions for S_y,
          // i.e., Syx0 = D([1, 2], S)(x0, y0)
          //       Syx0 = D([1, 2], S)(x.n, y0)
          // Compute them via a difference approximation:
          Syx0:= (yp0[1, 0] - yp0[ 0 , 0])/hx[0];
          Syxn:= (yp0[n, 0] - yp0[n-1, 0])/hx[n-1];
       else 
          // any value will do: it is not used, anyway
          Syx0:= 0: 
          Syxn:= 0:
       end_if;
       //-------------------------------------------------
       [b, c, d]:= numeric::_cubicSpline(
          xBC, n, hx, [yp0[i, 0] $ i = 0..n], Syx0, Syxn):
       if n > 0 then
         b[n]:= b[n-1] + 2*c[n-1]*hx[n-1] + 3*d[n-1]*hx[n-1]^2;
         c[n]:= c[n-1] + 3*d[n-1]*hx[n-1];
         d[n]:= d[n-1];
       end_if;
       (yp0[i, 1] := b[i];) $ i = 0..n;
       (yp0[i, 2] := c[i];) $ i = 0..n;
       (yp0[i, 3] := d[i];) $ i = 0..n;
       //-------------------------------------------------
       if xBC = Complete and n > 0 then
          // we need complete boundary conditions for S_y,
          // i.e., Syx0 = D([1, 2], S)(x0, y.m)
          //       Syxn = D([1, 2], S)(x.n, y.m)
          // Compute them via a difference approximation:
          Syx0:= (ypm[1, 0] - ypm[ 0 , 0])/hx[0];
          Syxn:= (ypm[n, 0] - ypm[n-1, 0])/hx[n-1];
       else 
          // any value will do: it is not used, anyway
          Syx0:= 0;
          Syxn:= 0;
       end_if;
       //-------------------------------------------------
       [b, c, d]:= numeric::_cubicSpline(
          xBC, n, hx, [ypm[i, 0] $ i = 0..n], Syx0, Syxn):
       if n > 0 then
         b[n]:= b[n-1] + 2*c[n-1]*hx[n-1] + 3*d[n-1]*hx[n-1]^2;
         c[n]:= c[n-1] + 3*d[n-1]*hx[n-1];
         d[n]:= d[n-1];
       end_if;
       (ypm[i, 1] := b[i];) $ i = 0..n;
       (ypm[i, 2] := c[i];) $ i = 0..n;
       (ypm[i, 3] := d[i];) $ i = 0..n;
       //-------------------------------------------------
   end_if;

   /* --------------------------------------------------------
      Step 2.0 - 2.3): compute 1 dimensional spline data along 
      the 2nd parameter lines. Store these data in the array A
      that holds the 2 dimensional spline data
   -------------------------------------------------------- */

   for k from 0 to 3 do
    //---------------------------------------------------------------
    // Consider the k-th x-derivative S.k(x, y) = diff(S(x, y), x $ k).
    // The data S.k(x[i], y[j]) were computed in step 1:
    //     S.k(x[i], y[j]) = A[i, j, k, 0].
    // For fixed i, use the table (y[j], S.k(x[i], y[j]) to compute the
    // spline S.k(x[i], y) by a 1-dimensional spline interpolation.
    // Once we have S.k(x[i], y), we know the y-derivates of the
    // x-derivatives and can fill in all 2-dimensional spline data.
    //
    // Complete boundary data in the y direction were computed in
    // Step 1) and stored in A[i, 0, k, 1], A[i, n, k, 1]:
    //---------------------------------------------------------------
    for i from 0 to n do
     // We pass the complete boundary conditions yp0[i,k], ypm[i,k]
     // in any case. These data won't be used unless yBC = Complete:

     [b, c, d]:= numeric::_cubicSpline(
        yBC, m, hy, [A[i, j, k, 0] $ j = 0 .. m], yp0[i, k], ypm[i, k]);

     // For m > 0, the data are only needed for j = 0 .. m-1.
     // However, for m = 0, the data must be set, too:

     mm:= max(m - 1, 0);;
     (A[i,j,k,1] := b[j];) $ j = 0..mm;
     (A[i,j,k,2] := c[j];) $ j = 0..mm;
     (A[i,j,k,3] := d[j];) $ j = 0..mm;
   end_for;
  end_for:

   //-----------------------------------------------
   // Utility called by the returned spline function
   //-----------------------------------------------
   indexsearch:=proc(lo,hi,Xe,x) 
   local mid;
   option noDebug;
   begin
      if hi <= lo+1 then return(lo) end_if;
      // note that specfunc::round is quite
      // a lot faster than specfunc::trunc
      mid:= specfunc::round((lo+hi)/2);
      if domtype(float(x-Xe[mid]))<>DOM_FLOAT then
         error("spline function contains symbolic nodes");
      end_if;
      if float(x-Xe[mid]) < 0 then
           indexsearch(lo,mid,Xe,x)
      else indexsearch(mid,hi,Xe,x)
      end_if;
   end_proc;

   // ------------------------------------
   // return the following spline function
   //-------------------------------------
   proc(x, y)
   local i,j, X, Y, xderiv, yderiv;
   option noDebug;
   begin
      if args(0) < 2 or args(0)  > 5 then 
         error("expecting no less than 2 and no more than 5 arguments"):
      end_if;
      if not symbolic then
         [x, y]:= [float(x), float(y)];
      end_if;
      [xderiv, yderiv]:= [0, 0];
      case args(0)
      of 2 do
      of 3 do
          //-------------------------------------------
          // symbolic return if x or y are symbolic (we
          // cannot decide which patch (x, y) is in)
          //-------------------------------------------
          if (domtype(float(x)) <> DOM_FLOAT) or 
             (domtype(float(y)) <> DOM_FLOAT) then
            if eval( op(op(_act_proc_env(),1),6)) = op(_act_proc_env(),1) then
               return(procname(args()));
            else
                 return(subsop(hold(ret)(args()),
                        0=op(_act_proc_env(),1), Unsimplified));
            end_if;
          end_if;
          //-------------------
          // symbolic nodes ?
          //-------------------
          if domtype(float(x-xArray[0]))<>DOM_FLOAT then
             error("spline function contains symbolic nodes");
          end_if;
          if domtype(float(y-yArray[0]))<>DOM_FLOAT then
             error("spline function contains symbolic nodes");
          end_if;

          //----------------------------------
          // search for the right patch (i, j)
          //----------------------------------
          if n = 0 then n:= 1; end_if;
          if float(x - xArray[0]) < 0 then
               i:= 0
          elif float(x - xArray[n-1]) >= 0 then
               i:= n - 1;
          else i:= indexsearch(0, n-1, xArray, x)
          end_if;

          if m = 0 then m:= 1; end_if;
          if float(y - yArray[0]) < 0 then
               j:=0
          elif float(y - yArray[m-1]) >= 0 then 
               j:= m - 1;
          else j:=indexsearch(0, m-1, yArray, y)
          end_if;

          if args(0) = 3 and 
             domtype(args(3)) = DOM_LIST and
             nops(args(3)) = 2 then
             [xderiv, yderiv] := args(3):
             if domtype(xderiv) <> DOM_INT or
                domtype(yderiv) <> DOM_INT then
                error("expecting a list of integers to specify which ".
                      "partial derivatives are to be returned. Received ".
                      expr2text(args(3)));
             end_if;
          end_if:
          break;
      of 4 do
      of 5 do
          [i, j]:= [args(3), args(4)];
          if domtype(i) <> DOM_INT then
             error("the 3rd argument must be an integer");
          end_if;
          i:= max(0, min(i, n-1));
          if domtype(j) <> DOM_INT then
             error("the 4th argument must be an integer");
          end_if;
          j:= max(0, min(j, m-1));
          if args(0) = 5 and 
             domtype(args(5)) = DOM_LIST and
             nops(args(5)) = 2 then
             [xderiv, yderiv] := args(5):
             if domtype(xderiv) <> DOM_INT or
                domtype(yderiv) <> DOM_INT then
                error("expecting a list of integers to specify which ".
                      "partial derivatives are to be returned. Received ".
                      expr2text(args(5)));
             end_if;
          end_if:
          break;
      end_case;

      //------------------------------------
      // evaluate the spline in patch (i, j)
      //------------------------------------
      X:= x - xArray[i];
      Y:= y - yArray[j];

      /* the direct representation: 24 multiplications, 16 powers, 15 additions
      Spline(x, y) = 
        A[i,j,0,0]     + A[i,j,0,1]*Y     + A[i,j,0,2]*Y^2     + A[i,j,0,3]*Y^3
      + A[i,j,1,0]*X   + A[i,j,1,1]*X*Y   + A[i,j,1,2]*X*Y^2   + A[i,j,1,3]*X*Y^3
      + A[i,j,2,0]*X^2 + A[i,j,2,1]*X^2*Y + A[i,j,2,2]*X^2*Y^2 + A[i,j,2,3]*X^2*Y^3
      + A[i,j,3,0]*X^3 + A[i,j,3,1]*X^3*Y + A[i,j,3,2]*X^3*Y^2 + A[i,j,3,3]*X^3*Y^3
      */
      // return diff(S(x, y), x $ xderiv, $ y yderiv)
      case [xderiv, yderiv]
      of [0, 0] do
         // use Horner representation: 15 multiplications, 0 powers, 15 additions
         return(context(  
               A[i,j,0,0] + Y*(A[i,j,0,1] + Y*(A[i,j,0,2] + Y*A[i,j,0,3]))
          + X*(A[i,j,1,0] + Y*(A[i,j,1,1] + Y*(A[i,j,1,2] + Y*A[i,j,1,3]))
          + X*(A[i,j,2,0] + Y*(A[i,j,2,1] + Y*(A[i,j,2,2] + Y*A[i,j,2,3]))
          + X*(A[i,j,3,0] + Y*(A[i,j,3,1] + Y*(A[i,j,3,2] + Y*A[i,j,3,3])))))));
      of [0, 1] do
         return(context(  
                A[i,j,0,1] + Y*(2*A[i,j,0,2] + 3*Y*A[i,j,0,3])
          + X*( A[i,j,1,1] + Y*(2*A[i,j,1,2] + 3*Y*A[i,j,1,3])
          + X*( A[i,j,2,1] + Y*(2*A[i,j,2,2] + 3*Y*A[i,j,2,3])
          + X*( A[i,j,3,1] + Y*(2*A[i,j,3,2] + 3*Y*A[i,j,3,3]))))));
      of [0, 2] do
         return(context(  
                2*A[i,j,0,2] + 6*Y*A[i,j,0,3] 
          + X*( 2*A[i,j,1,2] + 6*Y*A[i,j,1,3]
          + X*( 2*A[i,j,2,2] + 6*Y*A[i,j,2,3]
          + X*( 2*A[i,j,3,2] + 6*Y*A[i,j,3,3])))));
      of [0, 3] do
         return(context(  
            6*(A[i,j,0,3] 
          + X*(A[i,j,1,3]
          + X*(A[i,j,2,3]
          + X*(A[i,j,3,3]))))));
      of [1, 0] do
         return(context( 
                A[i,j,1,0] + Y*(A[i,j,1,1] + Y*(A[i,j,1,2] + Y*A[i,j,1,3]))
        + X*(2*(A[i,j,2,0] + Y*(A[i,j,2,1] + Y*(A[i,j,2,2] + Y*A[i,j,2,3])))
        + X*(3*(A[i,j,3,0] + Y*(A[i,j,3,1] + Y*(A[i,j,3,2] + Y*A[i,j,3,3])))))));
      of [1, 1] do
         return(context( 
                A[i,j,1,1] + Y*(2*A[i,j,1,2] + 3*Y*A[i,j,1,3])
        + X*(2*(A[i,j,2,1] + Y*(2*A[i,j,2,2] + 3*Y*A[i,j,2,3]))
        + X*(3*(A[i,j,3,1] + Y*(2*A[i,j,3,2] + 3*Y*A[i,j,3,3]))))));
      of [1, 2] do
         return(context( 
                2*A[i,j,1,2] + 6*Y*A[i,j,1,3]
        + X*(2*(2*A[i,j,2,2] + 6*Y*A[i,j,2,3])
        + X*(3*(2*A[i,j,3,2] + 6*Y*A[i,j,3,3])))));
      of [1, 3] do
         return(context( 
          6  *(  A[i,j,1,3]
        +   X*(2*A[i,j,2,3] +
        +   X* 3*A[i,j,3,3]))));
      of [2, 0] do
         return(context( 
           2 *(A[i,j,2,0] + Y*(A[i,j,2,1] + Y*(A[i,j,2,2] + Y*A[i,j,2,3])))
        + 6*X*(A[i,j,3,0] + Y*(A[i,j,3,1] + Y*(A[i,j,3,2] + Y*A[i,j,3,3])))));
      of [2, 1] do
         return(context( 
           2 *(A[i,j,2,1] + Y*(2*A[i,j,2,2] + 3*Y*A[i,j,2,3]))
        + 6*X*(A[i,j,3,1] + Y*(2*A[i,j,3,2] + 3*Y*A[i,j,3,3]))));
      of [2, 2] do
         return(context( 
           4 *(A[i,j,2,2] + 3*Y*A[i,j,2,3])
        +12*X*(A[i,j,3,2] + 3*Y*A[i,j,3,3])));
      of [2, 3] do
         return(context( 
          12  *A[i,j,2,3]
        + 36*X*A[i,j,3,3]));
      of [3, 0] do
         return(context( 
            6*(A[i,j,3,0] + Y*(A[i,j,3,1] + Y*(A[i,j,3,2] + Y*A[i,j,3,3])))));
      of [3, 1] do
         return(context( 
            6*(A[i,j,3,1] + Y*(2*A[i,j,3,2] + 3*Y*A[i,j,3,3]))));
      of [3, 2] do
         return(context( 
           12*(A[i,j,3,2] + 3*Y*A[i,j,3,3])));
      of [3, 3] do
         return(context( 
            36*A[i,j,3,3]));
      otherwise
         // either xderiv > 3 or yderiv > 3
         if symbolic then
              return(0)
         else return(float(0))
         end_if;
      end_case;
   end_proc;
end_proc:
