//------   file  det.mu  25.9.97    ---------------------
/*  Help file:

 det - determinant of a matrix

 Call:

 numeric::det(A <, Symbolic> <,hard_soft> 
              <, MinorExpansion>, <NoWarning> )
                                                                   
 Parameter:

  A         -- a  square real or complex matrix of domtype DOM_ARRAY,
               DOM_HFARRAY or of category Cat::Matrix
  hard_soft -- either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'
               If DIGITS <=15, the default is HardwareFloats
               If DIGITS  >15, the default is SoftwareFloats

                                                                   
Synopsis:

  numeric::det(A); returns the determinant of A. Non-float numerical
  entries of the matrix are evaluated to floats in the course of the
  computation. The algorithm is based on QR factorization using
  Householder transformations.

  Conversion to floats can be suppressed by the option Symbolic. In
  this case a symbolic LU factorization is used, non-numeric symbolic
  entries are admissible. However, the presence of such entries will
  slow down the computation drastically.

  With the option MinorExpansion recursive minor expansion along the
  first row is used. This option is useful for small dense matrices 
  with symbolic entries as well as for large sparse matrices.

  Examples: 

>> A:=array(1..3,1..3,[ [1,1,I] , [1,exp(1),I] , [1,2,2] ]):

>> numeric::det(A);
                        3.436563656 - 1.718281828 I

>> A:=array(1..3,1..3,[ [1/x,1,I/(2+x)] , [1,1,1] , [1,x,1] ]):     

>> numeric::det(A,Symbolic);
                                                 2
                        2 - (1 + I) x - (1 - I) x
                        --------------------------
                                x (x + 2)
>> numeric::det(A,MinorExpansion);

                           1     I      I x
                           - - ----- + ----- - 1
                           x   x + 2   x + 2

See also:  linalg::det

------------------------------------------------------------------*/

numeric::det:=proc(A)
local n, m, useHardwareFloats, HardwareFloatsRequested, 
      nowarning, d, symbolic, use_normal, normal2, method, B, 
      dont_use_normal_requested,
      i, j, k, result, real, h, s, ss, tmp, pivindex, 
      subdet, c, zeroes;
begin
   if args(0) < 1 then 
       error("expecting at least one argument") 
   end_if;

  //------------------
  // Check the matrix 
  //------------------
  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(args(1))) then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]),
                     [op(A)]);
       end_if;
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
  else
     error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;
  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  symbolic:= FALSE;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  nowarning:= FALSE;
  dont_use_normal_requested:= FALSE;
  normal2:= x -> normal(x, Expand = FALSE);

  for i from 2 to args(0) do
      case args(i)
      of Symbolic do method:= 1: // symbolic mode
                     symbolic:= TRUE;
                     useHardwareFloats:= FALSE;
                     break;
      of Hard do
      of HardwareFloats do
                     if method = 1 then
                        // Wrong usage by the user: print the following 
                        // warning even if NoWarning is used:
                        warning("ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'"):
                        method:= 1;
                        useHardwareFloats:= FALSE:
                     else
                        useHardwareFloats:= TRUE:
                        HardwareFloatsRequested:= TRUE;
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of MinorExpansion do
                     method:=3;  // minor expansion 
                     useHardwareFloats:= FALSE;
                     break;
      of NoWarning do
                     nowarning:= TRUE;
                     break;
      of "dont_use_normal" do
                     dont_use_normal_requested:= TRUE;
                     break;
      otherwise
         error("unknown option ".expr2text(args(i)));
      end_case;
  end_for;

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and not nowarning then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.");
        end_if;
        break;
     end_if;
     if not numeric::startHardwareFloats("det") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and not nowarning then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     /*-----------------------------------------------------------
     //  convert to hfarray
     -----------------------------------------------------------*/
     // The hfa module expects an hfarray as input.
     // If A contains symbols or numerical stuff that
     // cannot be represented by hardware floats, the
     // conversion to an hfarray will fail. Use traperror
     // to detect these cases and fall back to software floats
     if domtype(A) = DOM_HFARRAY then
        B:= A
     else
        if traperror((
             B:= hfarray(1..n, 1..m, [op(A)]);
           )) <> 0 then
           userinfo(1, "'HardwareFloats' failed"):
           useHardwareFloats:= FALSE;
           break;
        end_if;
     end_if:

     /*-----------------------------------------------------------
     //  call the hfa module
     -----------------------------------------------------------*/
     if traperror((d:= hfa::det(B))) = 0 then
        if has(d, RD_NAN) or
           has(d, RD_INF) or
           has(d, RD_NINF) then
           userinfo(1, "'HardwareFloats' failed"):
           useHardwareFloats:= FALSE;
        else
           // undo normalization:
           return(d);
        end_if:
     else
        userinfo(1, "'HardwareFloats' failed"):
     end_if:
  end_case;
  //---------------------------------
  // end of useHardwareFloats = TRUE
  //---------------------------------

  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  if HardwareFloatsRequested and not nowarning then
     warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
  end_if;

  userinfo(1, "using software floats"):
  // convert to domtype DOM_ARRAY for speed
  if domtype(A) = DOM_HFARRAY then
     assert(nops(A) = m*n):
     A:= array(1..n, 1..m, [op(A)]);
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     A:= expr(A);
  end_if;
  if domtype(A) <> DOM_ARRAY then
     error("the matrix must be an array, an hfarray, ".
           "or a matrix of category 'Cat::Matrix'");
  end_if:

  if method = 2 then
    B:= map(A, float):
    if {op(map(B, domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {} and
      not nowarning then
      warning("symbolic entry found, switching to symbolic mode");
      method := 1;
      if dont_use_normal_requested = FALSE then
         use_normal := TRUE
      end_if;
    else
      A:= B;
      if ({op(map(A,(iszero@Im)))} = {TRUE}) then  // is matrix real ? 
           real:= TRUE;
      else real:= FALSE;
      end_if:
   end_if;
  end_if;

  use_normal:= FALSE;
  if method=1 then
    if (indets(A)<>{})
    or has({op(map(A,domtype))},DOM_NIL)
    or has({op(map(A,domtype))},DOM_EXPR) then
      if dont_use_normal_requested = FALSE then
       use_normal:= TRUE
      end_if:
    end_if;
  end_if;

  if method=1 then
    userinfo(1,"symbolic mode, no pivoting unless necessary");
  end_if;
  if method=2 then
    userinfo(1,"numerical mode, using column pivoting");
  end_if;

  //--------------------------------------------------
  if method=1 then
     userinfo(1,"computing LU factorization");
     // enforce full evaluation
     A:= map(A, eval): 
     if (indets(A)<>{}) 
        or has({op(map(A,domtype))},DOM_NIL)
        or has({op(map(A,domtype))},DOM_EXPR)
        then 
             if dont_use_normal_requested then
               use_normal:=FALSE 
             else
               use_normal:=TRUE 
             end_if:
        else 
             use_normal:=FALSE;
        end_if;
     if use_normal then A[1,1]:=normal2(A[1,1]); end_if; 
     result:=1;
     for j from 1 to n-1 do
       if iszero(A[j,j]) then
           for pivindex from j+1 to n-1 do
               if not iszero(A[pivindex,j]) then break; end_if;
           end_for;
        else pivindex:=j;
        end_if;
        if pivindex<>j then
           result:=-result;
          (tmp:=A[j,k];A[j,k]:=A[pivindex,k];A[pivindex,k]:=tmp;) $ k=1..n;
        end_if;
        if iszero(A[j,j]) then return(0) end_if;
        result:=result*A[j,j];
        if use_normal
        then for i from j+1 to n do
                 tmp:=A[i,j]/A[j,j];
                 if not iszero(tmp) then
                 (A[i,k]:=normal2(A[i,k]-tmp*A[j,k])) $ k=j+1..n;
                 end_if;
             end_for;
        else for i from j+1 to n do
                 tmp:=A[i,j]/A[j,j];
                 if not iszero(tmp) then
                 (A[i,k]:=A[i,k]-tmp*A[j,k]) $ k=j+1..n;
                 end_if;
             end_for;
        end_if;
     end_for;
     result:=result*A[n,n]:
     if use_normal then return(normal2(result)):
                   else return(result):
     end_if;
  end_if; //method=1

  //--------------------------------------------------
  if method=2 then            
     userinfo(1,"computing QR factorization by Householder transformations");
     result:=1;
     for j from 1 to n-1 do
          // Householder step from the left to obtain upper triangular form. 
          s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..n);
          if iszero(s) then result:= result*A[j,j];
          else ss:=s+specfunc::abs(A[j,j])^2;
               s:=specfunc::sqrt(ss):
               h := 1 / (ss + specfunc::abs(A[j,j]) * s);
               if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
               A[j,j]:=A[j,j]+s; result:=result*s;
               for i from j+1 to n do
                   if real then   s:=_plus(A[k,j] *A[k,i]$ k=j..n) ;
                   else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..n) ;
                   end_if;
                   s:=s*h;
                   if not iszero(s) then
                     (A[k,i]:=A[k,i]-s*A[k,j]) $ k=j+1..n ;
                   end_if;
               end_for;
          end_if;
     end_for;
     return(result*A[n,n]);
  end_if; //method=2

  //--------------------------------------------------
  if method=3 then
     userinfo(1,"using minor expansion along first row");
     // enforce full evaluation
     A:= map(A, eval): 
     if not symbolic then
        A:= map(A, float);
     end_if;
  
     // define local subroutine  subdet = minor 
  
     subdet:=proc(r,c,n,digits) local result,i; option remember;
     begin
        if n=3 then return(A[r[1],c[1]]*A[r[2],c[2]]*A[r[3],c[3]]
                         + A[r[1],c[2]]*A[r[3],c[1]]*A[r[2],c[3]]
                         + A[r[2],c[1]]*A[r[1],c[3]]*A[r[3],c[2]]
                         - A[r[1],c[1]]*A[r[2],c[3]]*A[r[3],c[2]]
                         - A[r[1],c[2]]*A[r[2],c[1]]*A[r[3],c[3]]
                         - A[r[1],c[3]]*A[r[2],c[2]]*A[r[3],c[1]]
                          ) end_if;
        if n=2 then return(A[r[1],c[1]]*A[r[2],c[2]]
                          -A[r[1],c[2]]*A[r[2],c[1]]) end_if;
        if n=1 then return(A[r[1],c[1]]) end_if;

        if symbolic then 
           result:=0;
        else
           result:= float(0);
        end_if;
        
        for i from 1 to n do
           if not iszero(A[r[i],c[1]]) then
              result:=result+(-1)^(1+i)*A[r[i],c[1]]
              *subdet(subsop(r,i=null()),subsop(c,1=null()),n-1,digits);
           end_if;
        end_for;
        result;
    end_proc; //subdet

    //-------------------------------------------------------
    // find ordering c=[c[1],..,c[n]] of the columns s.th. 
    //    number of zeroes in column c[1]                  
    // >= number of zeroes in column c[2]                  
    // >=   ...                                            
    // Then expand recursively along column c[1] ...       
      
    for j from 1 to n do                        
      zeroes[j]:= nops(select([A[i, j] $ i = 1..n], iszero));
    end_for;                                    

    // reorder the columns
    c:= sort([j $ j=1..n ], (i,j) -> zeroes[i] >= zeroes[j]);

    // compute s = signum of column permutation 
    s:= 1: 
    for i from 1 to n - 1 do
      for j from i + 1 to n do
        if c[j] < c[i] then
           s := -s;
        end_if
      end_for
    end_for;
                                                
    return(s*subdet([i$ i=1..n],c,n,DIGITS));    
  
  end_if; //method=3
  //--------------------------------------------------

end_proc: //det
