//   


/* -----------------------------------------------------
   numeric::discont(f, x, range, precision, precision2)
  
    Try to determine the set of all x in range such that
    f is discontinous at x

    Isolated singularities are only computed up to precision

    Instead of several singularities none of which has a distance
    of more than precision2 from the other, only one singularity is
    returned

    Not reliable!
 -----------------------------------------------------*/

alias(DONTKNOW ={}):

numeric::discont :=
proc(f, equ: "_equal", precision: Type::Real, precision2): DOM_SET
  local i, s, r, x: DOM_IDENT, range: "_range",
  fsolve: DOM_PROC,
  fdiscontbranch: DOM_PROC,
  fdiscontnumeric: DOM_PROC,
  fdiscontnumeric2: DOM_PROC,
  fpreimage: DOM_PROC,
  frand:DOM_PROC,
  digits: DOM_INT;
begin

  if precision <= 0 then
     error("3rd argument: expecting a positive precision goal, ".
           "got ".expr2text(precision));
  end_if:
  //-----------------------------------------------------------
  // digits = 1 + ceil(-log(10, precision)), i.e.,
  // precision > 10^(-digits):
  digits:= max(10, 1 + ceil(-ln::float(precision)/2.302585093)); 
  //-----------------------------------------------------------
  frand:= frandom(1): // random generator to perturb bisectioning/trisectiong.
                      // Use seed = 1 to make numeric::discont look deterministic
                      // from a user's point of view.
 
  //---------------------------------------------
  // There are continuing problems with piecewise
  // objects that should have been converted to
  // numeric::piecewise objects by the plot
  // infrastructure. To cure this problem once
  // and for all, convert a piecewise to a
  // numeric::piecewise here
  //--------------
  if type(f) = piecewise then
     f:= numeric::piecewise::convert(f);
     if type(f) <> numeric::piecewise then
        error("could not convert piecewise to numeric::piecewise");
     end_if;
  end_if;

  //--------------
  // local methods
  //--------------

/* ----
  //----------------------------------------------------------
  // fsolve(f, x, range)
  // return both solutions and discontinuities of f in range 
  //----------------------------------------------------------
  fsolve:=
  proc(f, x: DOM_IDENT, range: "_range")
    local result, midpoint: DOM_PROC;
  save DIGITS;
  begin
    midpoint:= liste -> (liste[1] + liste[2])/2;
    // numeric::realroots requires precision >= 10^(-DIGITS).
    // Increase DIGITS accordingly
    DIGITS:= digits:
    if traperror
      ((result:= {op(map(numeric::realroots(f, x = range, precision),
                         midpoint)) })) = 0 then
      /// !!!!
      /// TO DO : realroots should react to precision2
      /// !!!!
      return(result)
    else
      return(DONTKNOW)
    end_if
  end_proc:
---- */

  // as long as realroots does not react to precision2, this may help
  fsolve:= (f, x,range) -> fdiscontnumeric(1/f, x, range);

  
  //----------------------------------------------------------
  //----------------------------------------------------------
  fdiscontnumeric:=
  proc(f, x: DOM_IDENT, range: "_range")
    local l,s, t, midpoint;
  begin
    l:= op(range, 2) - op(range,1): // length of the interval
    if l < precision2 then
        // start second form of discontnumeric
      return(fdiscontnumeric2(f, x, range))
    end_if;
      
    s:= interval(subs(f, x = DOM_INTERVAL(range)));
    if type(s) <> DOM_INTERVAL then
      return(DONTKNOW)
    end_if;
    
    // this recognizes some jumps, and it should find poles

    if has(s, {RD_INF, RD_NINF}) or op(s, 0) = hold(_union) then
      // perturb bisectioning to avoid the situation that
      // the discontinuity ends up on the splitting point
      // (it would remain in both subintervals for the rest
      //  of the iteration).
      t:= 1/2 + frand()/10^5:
      midpoint:= t*op(range, 1) + (1-t)*op(range, 2);
      if float(op(range, 2) - op(range, 1)) < precision then
        return({midpoint})
      else
        return(fdiscontnumeric(f, x, op(range, 1)..midpoint) union
               fdiscontnumeric(f, x, midpoint..op(range, 2)))
      end_if
    else
      return({})
    end_if
  end_proc:


  // fdiscontnumeric2
  //
  // checks whether the outer thirds of an interval contain singularities
  // note that it makes no sense to check two half-intervals here:
  // the singularity might be at the midpoint ...
  
  
  fdiscontnumeric2:=
  proc(f, x: DOM_IDENT, range: "_range")
    local
    s: DOM_LIST,
    ranges: DOM_LIST,
    len: DOM_FLOAT,
    l: DOM_FLOAT,
    r: DOM_FLOAT,
    i: DOM_INT,
    evAt: DOM_PROC,
    disc: DOM_PROC,
    t: DOM::FLOAT;
  begin

    // disc: decides whether a given interval representing f(r)
    // shows that there must be a discontinuity (pole or jump) in r
    // if the interval is unbounded, there must be a pole
    // if the interval consists of several parts, there must be a jump
    disc:=
    proc(iv)
    begin
      has(iv, {RD_INF, RD_NINF}) or bool(op(iv, 0) = hold(_union))
    end_proc;

    
    // evAt: evaluate f in the range r
    evAt:=
    proc(r)
    begin
     // interval::evalSlope(f, x = DOM_INTERVAL(r))
     // interval::evalBiCentered(f, x = DOM_INTERVAL(r))
// Stefan: 26.7.05: use evalMean
     // interval::evalMean(f, x = DOM_INTERVAL(r))
// Walter: 4.8.05: evalMean is way too slow, use subs:
        interval(subs(f, x = DOM_INTERVAL(r)))
    end_proc;
    
    len:= op(range, 2) - op(range, 1);
    l:= op(range, 1); // left border of the interval
    r:= op(range, 2); // right border of the interval
    // perturb bisectioning to avoid the situation that
    // the discontinuity ends up on the splitting point
    // (it would remain in both subintervals for the rest
    //  of the iteration).
    t:= (1/3 + frand()/10^6)*len:
    ranges:= [l .. l + t, l + t .. r -  t,   r - t .. r];
    s:= map(ranges, evAt);
  
    // two termination criteria
    //    (a) we detect one singularity in a small interval
    // or (b) we detect two singularities 
    if (len < precision and contains(map(s, disc), TRUE) > 0) or
       (disc(s[1]) and disc(s[3]))
        then
      // return midpoint
      return({(op(range, 1) + op(range, 2))/2})
    end_if;

    _union((if disc(s[i]) then
               fdiscontnumeric2(f, x, ranges[i])
            else {}
            end_if) $i=1..3);
    
  end_proc:
  
  //----------------------------------------------------------
  //----------------------------------------------------------
  //
  //  fpreimage(f, x, S, range) - compute the set of all x in range
  //                              such that f(x) in S
  //                    
  fpreimage:=
  proc(f, x, S, range)
    local a, s, iv;
  begin
    if domtype(S) <> DOM_SET then
      traperror((iv := hull(range);
                 iv := hull(subs(f, x=iv),EvalChanges);
                 if iszero(Im(iv)) and
                    op(iv, 1) <> RD_NINF and
                    op(iv, 2) <> RD_INF then
                   S := S intersect Dom::Interval([floor(op(iv, 1))],
                                                  [ceil(op(iv, 2))]);
                 end_if));
    end_if;
    if type(S) = DOM_SET then
      return(_union(fsolve(f - a, x, range) $a in S))
    end_if;
    FAIL
  end_proc:

  //----------------------------------------------------------
  //----------------------------------------------------------
  //
  //  fdiscontbranch(b)  - compute all discontinuities of
  //                       the branch b (of the picewise object f) 
  // 

  fdiscontbranch:=
  proc(b)
    local fdiscontcond, borders;
  begin

    // computes the set of border points of S
    // (= discontinuities of the indicator function of S)
    borders:=
    proc(S)
      local i: DOM_INT;
    begin
      case type(S)
        of "_union" do
        of "_minus" do  
        of "_intersect" do // to be careful ...  
          return(_union(borders(op(S, i) $i=1..nops(S))))
        of DOM_SET do   
          return(S)
        of Dom::Interval do  
          return({Dom::Interval::left(S), Dom::Interval::right(S)}
                 minus {-infinity, infinity}
                 )
        otherwise
          // this may be right or not 
          return({})
      end_case;
    end_proc;
      
    fdiscontcond:=
    proc(cond)
      local bo;
    begin
      case type(cond)
        of "_and" do
        of "_or" do
          _union(map(op(cond), fdiscontcond));
          break
        of "_not" do
          fdiscontcond(op(cond, 1));
          break
        of "_unequal" do
        of "_equal" do
        of "_less" do
        of "_leequal" do
          fsolve(op(cond, 1) - op(cond, 2), x, range);
          break
        of "_in" do
          _union(fsolve(op(cond, 1) - bo, x, range)
                 $bo in borders(op(cond, 2)));
          break
        otherwise
          {}
      end_case
    end_proc:
    
    fdiscontcond(op(b, 1)) union
    select(numeric::discont(op(b, 2), equ, precision, precision2),
           y -> subs(op(b, 1), x = y, EvalChanges))
  end_proc:
  
  //----------------------------------------------------------
  // The work starts:
  //----------------------------------------------------------

  //----------------------------------------------------------
  // analyzing second argument
  //----------------------------------------------------------
  [x, range]:= [op(equ)];  
  if type(x) <> DOM_IDENT or type(range) <> "_range" then
     error("Second argument must be of the form identifier = range")
  end_if;
  range:= map(range, float):

  //----------------------------------------------------------
  // special case: the range has length 0:
  //----------------------------------------------------------
  if iszero(op(range, 2) - op(range, 1)) then
     if traperror(subs(f, x= op(range,1),EvalChanges)) <> 0 then
          return({float(op(range,1))})
     else return({});
     end_if:
  end_if:

  //----------------------------------------------------------
  //  overloading for domains 
  //----------------------------------------------------------
  if f::dom::numericDiscont <> FAIL then
    return(f::dom::numericDiscont(args()))
  end_if;
  
  //----------------------------------------------------------
  // argument checking
  //----------------------------------------------------------
  if has(range, infinity) then
    error("Infinite ranges are not allowed")
  end_if;

  
  if args(0) <= 3 then
    precision2:= precision //default
  elif testtype(precision2, Type::Positive) = FALSE then
    error("Precision must be a positive number")
  end_if;

  //----------------------------------------------------------
  // convert range to float
  //----------------------------------------------------------
  range:= map(range, float);
  
  //----------------------------------------------------------
  // overloading for function environments 
  //----------------------------------------------------------
  if domtype(f) = DOM_EXPR and type((r:=eval(op(f,0))))=DOM_FUNC_ENV then
    if nops(f) =1 then
      if iszero(subs(Im(op(f, 1)), x = DOM_INTERVAL(range),EvalChanges)) then
        s:= r::realDiscont
      else
        s:= r::complexDiscont
      end_if
    else
      s:= FAIL
    end_if;
    if s<>FAIL then
      // f(g(x)) is continous unless f is not continous
      // at g(x) or g is not continous at x; let s be the set
      // of exceptional points
      s:= fpreimage(op(f), x, s, range)          
    end_if;
    if s <> FAIL then  
      return(s union
             numeric::discont(op(f), x = range, precision, precision2))            
    else
      // s = FAIL
      if (s:= r::numericDiscont(f, x = range, precision, precision2) ) <> FAIL
        then
        return(s)
      end_if
    end_if;
  end_if;

  //----------------------------------------------------------
  // do not handle RootOf's
  //----------------------------------------------------------
  if hastype(f, RootOf) then
    return(DONTKNOW)
  end_if;
  
  //----------------------------------------------------------
  //----------------------------------------------------------
  case type(f)
    of DOM_INT do
    of DOM_RAT do
    of DOM_IDENT do
    of "_index" do  
    of DOM_COMPLEX do
    of DOM_FLOAT do 
      return({})
    of "_plus" do
    of "_mult" do
      return(_union(numeric::discont(op(f,i), x = range, precision, precision2)
                    $i=1..nops(f)))
    of "_power" do
      // f=a^b
      // special cases for b integer and independent of x
      if not has(op(f,2), x) then
        if type(op(f, 2)) = DOM_INT then
          if op(f, 2) > 0 then
            return(numeric::discont(op(f,1), x = range, precision, precision2))
          else
            return(numeric::discont(op(f,1), x = range, precision, precision2)
                   union
                   fsolve(op(f, 1), x, range)  )
          end_if
        end_if;
        if interval(subs(Re(op(f, 1)), x = DOM_INTERVAL(range))) > 0 then
          // we are not on the branch cut
          return(numeric::discont(op(f,1), x = range, precision, precision2))
        end_if;
        // default: a(x)^b, where a might cross the branch cut
        return(numeric::discont(ln(op(f,1)), x = range, precision, precision2))
      else // ok? exp is continuous, but we might pass the branch cut
        return(numeric::discont(ln(op(f,1)), x = range, precision, precision2)
               union
               numeric::discont(op(f,2), x = range, precision, precision2)
               union
               fsolve(op(f, 1), x, range)  )
      end_if;

       // functions without singularities 
    of "besselJ" do
      f:=[op(f,2)];
    of "sin" do
    of "cos" do
    of "abs" do
    of "exp" do
    of "sinh" do
    of "cosh" do
    of "erf" do
    of "erfc" do
    of "Re" do
    of "Im" do  
       return(numeric::discont(op(f), x = range, precision, precision2))
   // functions with a discontinuity in 0 
    of "besselY" do
      f:=[op(f,2)];
       // fall through
    of "dirac" do
    of "heaviside" do
    of "sign" do
      assert(nops(f) = 1);
      return(numeric::discont(op(f), x = range, precision, precision2) union
             fsolve(op(f), x, range))
    // functions with a fdiscontinuity in 1 
    of "zeta" do
      return(numeric::discont(op(f, 1), x = range, precision, precision2) union
             fsolve(op(f, 1) - 1, x, range))
    // functions with discontinuities in all integers
    // cannot handle this in fdiscont
/*
    of "floor" do
    of "frac" do
    of "ceil" do
    of "trunc" do
      return(numeric::discont(op(f), x, range, precision) union
             fpreimage(op(f), x, Z_))
    of "round" do
        i:= genident();
        return(numeric::discont(op(f), x, range, precision) union
              fpreimage(op(f), x, Dom::ImageSet(i+1/2, i, Z_)))
    // functions with a discontinuity in -n 
    of "psi" do
      return(numeric::discont(op(f, 1), x = range, precision, precision2)
union
             fpreimage(op(f, 1), x,
                       Z_ intersect Dom::Interval(-infinity..[0])))
*/
    of "fact" do
    of "binomial" do  
       return(numeric::discont(rewrite(f, gamma), x = range, precision,
                               precision2))

    // logarithm: rewrite by ln
    // assume the base to be positive and not equal 1
    // either the base is constant and positive anyway, or it is an identifier
    // in the latter case, replace with a positive identifier   
    of "log" do
        return(numeric::discont(rewrite(f, ln), x = range, precision,
                                precision2))
    // integrals    
    of "int" do
      return(numeric::discont(op(f,1), x = range, precision2, precision2)
             union
             (if type(op(f,2))="_equal" then
                numeric::discont(op(f,[2,2,1]), x = range, precision,
                                 precision2)
                union numeric::discont(op(f,[2,2,2]), x = range, precision,
                                 precision2)
              else
                {}
              end_if))
    of numeric::piecewise do
      return(_union(map(op(f), fdiscontbranch)))
  end_case;

  //--------------------------------------------------------
  // last resort: try the DOM_INTERVAL floating-point method
  // Use traperror, because interval arithmetic is not
  // implemented for all special functions
  //--------------------------------------------------------
  if traperror((r:= fdiscontnumeric(f, x, range))) = 0 then
     return(r);
  end_if;

  return(DONTKNOW):
end_proc:

unalias(DONTKNOW):
// end of file 
