/*-------------------------------------------------------------------------------*/
/* Based on the diploma thesis                                                   */
/*   "Numerik dynamischer Systeme auf Lie-Gruppen, Renate Wagner, Dez.1998.      */
/* Code rewritten and optimized by W. Oevel, 1.1.1999                            */
/*-------------------------------------------------------------------------------*/
/* numeric::expMatrix - the exponential of a matrix                              */
/*                                                                               */
/* Aufruf:                                                                       */
/*                                                                               */
/* 1) numeric::expMatrix(A,   <hard_soft>,<method>,<ReturnType=d>,<NoWarning>)   */
/* 2) numeric::expMatrix(A, v,<hard_soft>,<method>,<ReturnType=d>,<NoWarning>)   */
/* 3) numeric::expMatrix(A, V,<hard_soft>,<method>,<ReturnType=d>,<NoWarning>)   */
/*                                                                               */
/* Parameter:                                                                    */
/*   A      : nxn Matrix (array(1..n,1..n,..)                                    */
/*   v      : n Vektor (list or array(1..n,..)                                   */
/*   V      : nxp Matrix (array(1..n,1..p,..)                                    */
/* method   : TaylorExpansion or Krylov or Interpolation or Diagonalization      */
/* hard_soft: either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'          */
/*            If DIGITS <=15, the default is HardwareFloats                      */
/*            If DIGITS  >15, the default is SoftwareFloats                      */
/* d        : either DOM_ARRAY or DOM_HFARRAY or Dom::DenseMatrix() or           */
/*            or Dom::Matrix()                                                   */
/* NoWarning: suppresses warnings                                                */
/*                                                                               */
/* Rueckgabewert:                                                                */
/*  1) exp(A)   = array(1..n,1..n,..)                                            */
/*  2) exp(A)*v = Liste oder array(1..n,..). Return type = input type            */
/*  3) exp(A)*V = array(1..n,1..p,..)                                            */
/*                                                                               */
/* Beschreibung:                                                                 */
/*                                                                               */
/* Die n x n Matrix A kann vom Typ DOM_ARRAY oder von der                        */
/* Kategorie Cat::Matrix sein.                                                   */
/* Der Vektor v kann eine Liste [v1,..,v.n] oder vom Typ DOM_ARRAY               */
/* (d.h., v=array(1..n,[v1,..,v.n])) sein.                                       */
/* Die n x p Matrix V kann DOM_ARRAY oder Cat::Matrix sein.                      */
/*                                                                               */
/* Die zur Verfuegung stehenden Methoden sind:                                   */
/*                                                                               */
/* - Default:                                                                    */
/*   ohne Angabe von <method> wird mittels einer Heuristik zwischen 2 Varianten  */
/*   von TaylorExpansion gewaehlt.                                               */
/* - TaylorExpansion:                                                            */
/*   Variante 1:                                                                 */
/*   Geeignetes Aufsummieren der Taylor-Reihe exp(A)*V = V+A*V+A^2*V/2!+...      */
/*   Falls kein V angegeben ist, so wird exp(A) = 1+A+A^2/2!+.. berechnet.       */
/*   Der Aufwand fuer exp(A) ist O(norm(A)*n^3), fuer exp(A)*V mit V= nxp-Matrix */
/*   Der Aufwand fuer exp(A) ist O(norm(A)*n^3), fuer exp(A)*V mit V= nxp-Matrix */
/*   ist er O(norm(A)*n^2*p). Damit unguenstig bei norm(A)>>1, guenstig bei      */
/*   1<<n, p<<n.                                                                 */ 
/*   Variante 2:                                                                 */
/*   Die Matrix A wird auf kleine Norm skaliert, dann wird die Taylor-Reihe      */
/*   aufsummiert und das Ergebnis wird durch repeated squaring zu exp(A) hoch-   */
/*   multipliziert. Gegebenenfalls wird zuletzt daraus exp(A)*v gebildet.        */
/*   Diese Version ist wenig geeignet, wenn die Dimension n sehr gross ist       */
/*   und nicht exp(A), sondern nur exp(A)*v benoetigt wird.                      */
/*   Der Aufwand ist O( ln(norm(A)*n^3 + n^2*p). Guenstig bei norm(A)>>1,        */
/*   unguenstig bei n>>1.                                                        */
/* - Krylov:                                                                     */
/*   Kann nur in der Version expMatrix(A,v,Krylov) oder expMatrix(A,V,Krylov)    */
/*   mit p=1 benutzt werden. Es wird das Arnoldi-Algorithmus verwendet.          */
/*   Geeignet fuer grosse Matrizen in folgenden Faellen:                         */
/*    * v ist von wenigen Eigenvektoren von A aufgespannt,                       */
/*    * A hat stark entartete Eigenwerte oder Cluster mit aehnlichen Eigenwerten.*/
/*   Achtung: der Arnoldi-Algorithmus neigt in einigen Faellen zur Instabilitaet.*/
/*   Achtung: Fehler ist duch norm(Delta exp(A)*v) <= norm(exp(A)*v)*10^(-DIGITS)*/
/*            gegeben. TaylorExpansion berechnet jede Komponente **einzeln** bis */
/*            auf relative Genauigkeit 10^(-DIGITS).                             */
/* - Interpolation:                                                              */
/*   Berechnet die Eigenwerte lambda von A und bestimmt ein Hermite-             */
/*   interpolierendes Polynom P mit P(lambda)=exp(lambda). Es folgt exp(A)=P(A). */
/*   Fuer exp(A)*V werden etwa n^3*p skalare Multiplikation benoetigt.           */
/*   Die Interpolation ist fuer n>>1 oft schlecht konditioniert und fuehrt zu    */
/*   starkem Anwachsen von Rundungsfehlern. Dies wird intern automatisch         */
/*   ueberprueft, expMatrix bricht bei schlechter Konditionierung mit einer      */
/*   Fehlermeldung ab.                                                           */
/* - Diagonalization:                                                            */
/*   Die Matrix A wird diagonalisiert: A = X diag(lambda1,lambda2,..) X^(-1).    */
/*   Das Ergebnis wird per exp(A) = X diag(exp(lambda1),exp(lambda2),..) X^(-1)  */
/*   berechnet, gegebenenfalls wird hieraus exp(A)*v, exp(A)*V ermittelt.        */
/*   Achtung: dies funktioniert nicht fuer beliebiges A! Nichtdiagonalisierbare  */
/*   Matrizen fuehren zu einer Fehlermeldung, bzw. es kommt zu einer grossen     */
/*   Verstaerkung von Rundungsfehlern, wenn die Nichtdiagonalisierbarkeit        */
/*   numerisch nicht ermittelt werden kann.                                      */
/*   Diese Methode ist aber FAIL safe fuer hermitesche oder schief-hermitesche   */
/*   Matrizen A.                                                                 */
/*                                                                               */
/* Die Algorithmen benutzen intern Polynomarithmetik.                            */
/*                                                                               */
/* Examples:                                                                     */
/*  >> A:= array(1..2,1..2,[[0,1],[-1,0]]):                                      */
/*  >> numeric::expMatrix(A);                                                    */
/*                                                                               */
/*                    +-                             -+                          */
/*                    |   0.5403023058, 0.8414709848  |                          */
/*                    |                               |                          */
/*                    |  -0.8414709848, 0.5403023058  |                          */
/*                    +-                             -+                          */
/*                                                                               */
/*  >> v1:= [1, 1]: v2:= array(1..2, [1, 1]):                                    */
/*  >> numeric::expMatrix(A, v1), numeric::expMatrix(A, v2, Krylov);             */
/*                                                                               */
/*                                    +-                         -+              */
/*       [1.38177329, -0.3011686789], | 1.38177329, -0.3011686789 |              */
/*                                    +-                         -+              */
/*-------------------------------------------------------------------------------*/

numeric::expMatrix:= proc(A)
local B, useHardwareFloats, HardwareFloatsRequested, returnType,
      n, Arg, expAv, expA, v, p, erg, return_type, i, j, k, s,
      norm_A, method, sollNorm, NN, NN1, NN2, kk, x, APoly, vPoly,
      MatrixMult, tol, ergPoly, ergPoly_v, Apower, Apower_v,
      vNorm, KrylovBase, H, arnoldi, last_m, m, Hmm, expHe1, Err, 
      norm_Erg, eigenvals, absolute_tol, c, cc, pp, tPoly, A_is_real, 
      v_is_real, d, d2, X, dummy, w, dowarn, myreturn,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
save DIGITS;
begin
  if args(0) < 1 then error("expecting at least one argument") end_if;
  if args(0) > 5 then error("expecting no more than 5 arguments") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  if domtype(A) = DOM_ARRAY  or
     domtype(A) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     originalrowrange:= 1..n:
     originalcolrange:= 1..m:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;

  //---------------------------------------------
  // Check the options
  //--------------------------------------------
  // set defaults
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:

  expAv:= FALSE;   /*  no vector specified, yet */
  method:= 0;      /*  no method specified, yet */
  p:= n;           /*  no vector v specified, yet*/
  return_type:= 0; /*  default: return 2-dim array */
  dowarn:= TRUE;   /*  default */

  if args(0)>1 then
     for Arg in [args(2..args(0))] do
       /*  convert DOM_LIST and Cat::Matrix objects to DOM_ARRAY: */
       if domtype(Arg) = DOM_LIST then 
               expAv:= TRUE;
               if nops(Arg)<>n then
                  error("dimensions of matrix and vector do not match");
               end_if;
               return_type:= 1;
               v:=array(1..n, 1..1);
               (v[i,1]:= Arg[i];) $ i=1..n;
               p:= 1;  /*  v = n x p - matrix */
               originalcolrange:= 1..p:
               next;
       end_if;
       if domtype(Arg)=DOM_ARRAY or
          domtype(Arg)=DOM_HFARRAY then
               expAv:= TRUE;
               case op(Arg, [0, 1])
               of 1 do return_type:= 2; /* return "Vector" = array(1..n,[..]) */
                       // v:= array(1..n,1..1);
                       // (v[i,1]:= Arg[i];) $ i=1..n;
                       v:= Arg;
                       if op(v, [0, 2, 1]) <> 1 then
                         rowrangechanged:= TRUE;
                         originalrowrange:= op(v, [0, 2]):
                       end_if;
                       if domtype(v) = DOM_ARRAY then
                          v:= subsop(v, 0 = (2, 
                                             1.. op(v,[0,2,2]) + 1 - op(v,[0,2,1]), 
                                             1..1));
                       else // domtype(v) = DOM_HFARRAY
                          v:= hfarray(1.. op(v,[0,2,2]) + 1 - op(v,[0,2,1]), 
                                      1 ..1, 
                                      [op(v)]);
                       end_if:
                       p:= 1;  /*  v = n x p - matrix */
                       break;
               of 2 do v:= Arg;
                       if op(v, [0, 2, 1]) <> 1 or
                          op(v, [0, 3, 1]) <> 1 then
                         rowrangechanged:= TRUE;
                         originalrowrange:= op(v, [0, 2]):
                         if domtype(v) = DOM_ARRAY then
                            v:= subsop(v, 0 = (2, 
                                               1.. op(v,[0,2,2]) + 1 - op(v,[0,2,1]),
                                               1.. op(v,[0,3,2]) + 1 - op(v,[0,3,1])
                                               ));
          
                         else // domtype(v) = DOM_HFARRAY
                            v:= hfarray(1.. op(v,[0,2,2]) + 1 - op(v,[0,2,1]), 
                                        1.. op(v,[0,3,2]) + 1 - op(v,[0,3,1]),
                                        [op(v)]);
                         end_if;
                       end_if:
                       p:= op(v,[0,3,2]); /* v = n x p - matrix */
                       break;
               otherwise error("wrong format of second argument");
               end_case;
               if n<>op(v,[0,2,2]) then
                  error("dimensions do not match");
               end_if;
               originalcolrange:= 1..p:
               next;
       end_if;
       if Arg::dom::hasProp(Cat::Matrix)=TRUE
          then expAv:= TRUE;
               v:= expr(Arg);
               if domtype(v)<>DOM_ARRAY and
                  domtype(v)<>DOM_HFARRAY then
                  error("the 2nd argument must be an array, a list, ".
                        "or a matrix of category 'Cat::Matrix'");
               end_if;
               if n<>op(v,[0,2,2]) then
                  error("dimensions do not match");
               end_if;
               p:=op(v,[0,3,2]); /* v = n x p - matrix */
               originalcolrange:= 1..p:
               next;
       end_if;
       case Arg 
       of Hard do
       of HardwareFloats do
                     HardwareFloatsRequested:= TRUE;
                     useHardwareFloats:= TRUE:
                     break;
       of Soft do
       of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
       of TaylorExpansion do
                      method:=0; /*  method0 splits up into method=1 */
                                 /*  or method=2, see below */
                      useHardwareFloats:= FALSE:
                      break; 
       of Krylov do  
                      method:=3;
                      useHardwareFloats:= FALSE:
                      break; 
       of Interpolation do
                      method:=4;
                      useHardwareFloats:= FALSE:
                      break;
       of Diagonalization do
                      method:=5;
                      useHardwareFloats:= FALSE:
                      break;
       of NoWarning do
                     dowarn:= FALSE;
                     break;
       otherwise
         if type(Arg) = "_equal" and
            lhs(Arg) = ReturnType then
              returnType:= rhs(Arg);
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(returnType).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(Arg));
         end_if;
       end_case;
     end_for;
  end_if;

  //--------------------------------------------------
  // Format the output! Below, all branches compute an
  // array erg that is used 'globally' by the 
  // following routine:
  //-------------------------
  myreturn:= proc()
  local erg_;
  begin 
     if erg::dom = returnType and
        (not rowrangechanged) and
        (not colrangechanged) then
        return(erg);
     end_if;
     case returnType
     of Dom::Matrix() do
          return(returnType(erg));
     of Dom::DenseMatrix() do
          return(returnType::create(erg));
     of DOM_ARRAY do
          if domtype(erg) = DOM_ARRAY then
            if op(erg, [0, 1]) = 1 then
               erg:= subsop(erg, 0 = (1, originalrowrange));
            end_if;
            if op(erg, [0, 1]) = 2 then
               erg:= subsop(erg, 0 = (2, originalrowrange, originalcolrange));
            end_if;
          else
            if op(erg, [0, 1]) = 1 then
             erg:= array(originalrowrange, [op(erg)]);
            end_if;
            if op(erg, [0, 1]) = 2 then
             erg:= array(originalrowrange, originalcolrange, [op(erg)]);
            end_if;
          end_if;
          return(erg);
     of DOM_HFARRAY do
          if op(erg, [0, 1]) = 1 then
             erg_:= hfarray(originalrowrange, [op(erg)]);
          end_if;
          if op(erg, [0, 1]) = 2 then
             erg_:= hfarray(originalrowrange, originalcolrange, [op(erg)]);
          end_if;
          if has(erg_, RD_NAN) or
             has(erg_, RD_INF) or
             has(erg_, RD_NINF) then
             if dowarn then
                warning("cannot return the result as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead.");
             end_if:
             if op(erg, [0, 1]) = 1 then
              erg:= array(originalrowrange, [op(erg)]);
             end_if;
             if op(erg, [0, 1]) = 2 then
              erg:= array(originalrowrange, originalcolrange, [op(erg)]);
             end_if;
          else
             erg:= erg_;
          end_if:
          return(erg);
     otherwise
          error("unexpected return type"):
     end_case;
  end_proc:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("expMatrix") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "using hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
        if traperror((B:= hfarray(1..m, 1..n, [op(A)]))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed, ".
                        "using 'SoftwareFloats' instead");
             end_if;
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
        end_if;
     else
        B:= A;
     end_if;

     if expAv then
        if domtype(v) <> DOM_HFARRAY then
          if traperror((w:= hfarray(1..n, 1..p, [op(v)]))) <> 0 or
                has(w, RD_NAN) or
                has(w, RD_INF) or
                has(w, RD_NINF) then
                  if HardwareFloatsRequested and dowarn then
                     warning("HardwareFloats failed, ".
                             "using 'SoftwareFloats' instead");
                  end_if;
                  userinfo(1, "'HardwareFloats' failed"):
                  useHardwareFloats:= FALSE;
                  break;
          end_if;
       else
          w:= v;
       end_if;
     else
       w:= null();
     end_if;

     if traperror((erg:= hfa::expMatrix(B, w))) <> 0 or
           has(erg, RD_NAN) or
           has(erg, RD_INF) or
           has(erg, RD_NINF) then
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed, ".
                        "using 'SoftwareFloats' instead");
             end_if;
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
     end_if;

     //------------------------------
     //-------- return --------------
     //------------------------------

     if erg = FAIL then
        return(FAIL);
     end_if:

     return(myreturn());
  end_case; // of useHardwareFloats = TRUE

  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  userinfo(1, "using software floats"):

  //---------------------------------------------
  // The "old" expMatrix code through release 2.0
  // It allowed
  //   return_type = 0 :  return as array(1..n, 1..p)
  //   return_type = 1 :  return as DOM_LIST
  //   return_type = 2 :  return as array(1..n)
  // In release 2.1: only allow

  return_type := 0;

  //---------------------------------------------
  /*  convert A to DOM_ARRAY */
  if A::dom::hasProp(Cat::Matrix) = TRUE then 
     A:= expr(A)
  end_if;
  if domtype(A)<>DOM_ARRAY and
     domtype(A)<>DOM_HFARRAY then 
     error("the matrix must be an array or of category 'Cat::Matrix'");
  end_if;

  if domtype(A) <> DOM_HFARRAY then
     A:= map(A, float); 
     if map({op(A)}, domtype) minus {DOM_FLOAT,DOM_COMPLEX}<>{} then 
        error("non-numerical entry in matrix")
     end_if;
  end_if;
  if domtype(A) = DOM_HFARRAY then
     A:= array(1..m, 1..n, [op(A)]);
  end_if:

  if expAv then
     if v::dom::hasProp(Cat::Matrix) = TRUE then 
        v:= expr(v)
     end_if;
     if domtype(v) <> DOM_HFARRAY then 
        v:= map(v,float); 
     end_if;
     if domtype(v) = DOM_HFARRAY then
        v:= array(1..n, 1..p, [op(v)]);
     end_if:
  end_if:


//------------------------------------------------------------------------------------
   /* -------------------------- */
   /* Special case: 1 x 1 matrix */
   /* -------------------------- */
  if n=1 then 
     if expAv
     then case return_type
          of 1 do erg:= [exp(A[1,1])*v[1,1]];
                  break;
          of 2 do erg:=array(1..1);
                  erg[1]:= exp(A[1,1])*v[1,1];
                  break;
          otherwise 
                  erg:= array(1..1,1..p);
                  (erg[1,j]:= exp(A[1,1])*v[1,j];) $ j=1..p;
          end_case;
     else erg:= array(1..1,1..1);
          erg[1,1]:= exp(A[1,1]);
     end_if;
     return(myreturn());
  end_if;
//------------------------------------------------------------------------------------
   /* ------------------------------------ */
   /* Special case: traceless 2 x 2 matrix */
   /* ------------------------------------ */
  if n=2 and A[2,2]=-A[1,1] then /* special formula for traceless 2x2 matrix */
     userinfo(1, "using special formula for traceless 2 x 2 matrices");
     d:= A[1,1]^2+A[1,2]*A[2,1];
     d:= float(specfunc::sqrt(d));
     if iszero(d) then s:=1 else s:= sinh(d)/d; end_if;
     c:= cosh(d);
     expA:= array(1..2,1..2):
     expA[1,1]:= c+s*A[1,1]: expA[1,2]:=   s*A[1,2]:  
     expA[2,1]:=   s*A[2,1]: expA[2,2]:= c+s*A[2,2]:  
     /* stabilize the smaller of the diagonal elements using
        det( exp(traceless) ) = 1 : */
     if specfunc::abs(expA[1,1]) > specfunc::abs(expA[2,2])
       then expA[2,2]:= (1+expA[2,1]*expA[1,2])/expA[1,1];
       else expA[1,1]:= (1+expA[2,1]*expA[1,2])/expA[2,2];
     end_if;
     if expAv
     then case return_type
          of 1 do erg:= [expA[1,1]*v[1,1]+expA[1,2]*v[2,1],
                         expA[2,1]*v[1,1]+expA[2,2]*v[2,1]];
                  break;
          of 2 do erg:= array(1..2,
                         [expA[1,1]*v[1,1]+expA[1,2]*v[2,1],
                          expA[2,1]*v[1,1]+expA[2,2]*v[2,1]]);
                  break;
          otherwise
                  erg:= array(1..2,1..p,
                         [[(expA[1,1]*v[1,j]+expA[1,2]*v[2,j]) $ j=1..p ],
                          [(expA[2,1]*v[1,j]+expA[2,2]*v[2,j]) $ j=1..p ]
                         ]);
          end_case;
     else erg:= expA;
     end_if;

     return(myreturn());
  end_if;
//------------------------------------------------------------------------------------
   /* ------------------------------------ */
   /* Special case: the matrix is in so(3) */
   /* ------------------------------------ */
  if n = 3 and           /* special formula for A in so(3)      */
     iszero(A[1,1]) and  /* For     [  0 -x3  x2]      [x1]     */
     iszero(A[2,2]) and  /*     A = [ x3  0  -x1], x = [x2]     */
     iszero(A[3,3]) and  /*         [-x2  x1  0 ]      [x3]     */
     A[1,2]=-A[2,1] and  /*                                     */  
     A[3,1]=-A[1,3] and  /* exp(A) = cos(d)*Eins + sin(d)/d *A  */
     A[3,2]=-A[2,3]      /*          + (1-cos(d))/d^2  x*x^T    */
     then                /* with d^2=x1^2+x^2+x^3               */
       userinfo(1,"using special formula for exp on so(3)");
       d2:= A[1,2]^2 + A[1,3]^2 + A[2,3]^2;
       d:= float(specfunc::sqrt(d2));
       c:= cos(d);
       if specfunc::abs(d2)< float(10^(-DIGITS)) then 
             s:= 1 - d2/6;
             cc:= 1/2 - d2/24 
        else s:=sin(d)/d;
             cc:= (1 - cos(d))/d2;
       end_if;
       expA:= array(1..3,1..3,[
           [c + cc* A[3,2]^2 ,s*A[1,2]+cc*A[3,2]*A[1,3],s*A[1,3]+cc*A[3,2]*A[2,1]],
           [s*A[2,1]+cc*A[1,3]*A[3,2],c +  cc*A[1,3]^2,s*A[2,3]+ cc*A[1,3]*A[2,1]],
           [s*A[3,1]+cc*A[2,1]*A[3,2],s*A[3,2]+cc*A[2,1]*A[1,3],c +  cc*A[2,1]^2 ]
                              ]);
       if expAv
       then case return_type
            of 1 do
               erg:= [expA[1,1]*v[1,1]+expA[1,2]*v[2,1]+expA[1,3]*v[3,1],
                      expA[2,1]*v[1,1]+expA[2,2]*v[2,1]+expA[2,3]*v[3,1],
                      expA[3,1]*v[1,1]+expA[3,2]*v[2,1]+expA[3,3]*v[3,1]];
               break;
            of 2 do 
               erg:= array(1..3, 
                           [expA[1,1]*v[1,1]+expA[1,2]*v[2,1]+expA[1,3]*v[3,1],
                            expA[2,1]*v[1,1]+expA[2,2]*v[2,1]+expA[2,3]*v[3,1],
                            expA[3,1]*v[1,1]+expA[3,2]*v[2,1]+expA[3,3]*v[3,1]]);
               break;
            otherwise 
               erg:= array(1..3,1..p,[
                  [(expA[1,1]*v[1,j]+expA[1,2]*v[2,j]+expA[1,3]*v[3,j]) $ j=1..p],
                  [(expA[2,1]*v[1,j]+expA[2,2]*v[2,j]+expA[2,3]*v[3,j]) $ j=1..p],
                  [(expA[3,1]*v[1,j]+expA[3,2]*v[2,j]+expA[3,3]*v[3,j]) $ j=1..p] 
                                             ]);
               break;
            end_case;
       else erg:= expA;
       end_if;

     return(myreturn());
  end_if;
//------------------------------------------------------------------------------------
   /* -------------------------------- */
   /* Special case: the vector is zero */
   /* -------------------------------- */
  if expAv and iszero(max(_plus(specfunc::abs(v[i,j]) $ j=1..p) $ i=1..n)) /*  norm(v)=0 */
  then case return_type
       of 1 do erg:= [float(0) $ p];
               break;
       of 2 do erg:= array(1..n, [float(0) $ n]);
               break;
       otherwise 
               erg:= array(1..n,1..p,[[float(0)$p]$n]);
       end_case;

     return(myreturn());
  end_if;
//------------------------------------------------------------------------------------
   /* -------------------------------- */
   /* Special case: the matrix is zero */
   /* -------------------------------- */
  norm_A:= max(_plus(specfunc::abs(A[i,j]) $ j=1..n) $ i=1..n); /*  norm(A) */
  if iszero(norm_A) then
     if expAv then 
        case return_type
        of 1 do erg:= [v[i,1] $ i=1..n];
                break; 
        of 2 do erg:= array(1..n, [v[i,1] $ i=1..n]);
                break; 
        otherwise 
                erg:= array(1..n,1..p,[[v[i,j] $ j=1..p] $ i=1..n]);
        end_case;
     else 
        erg:= array(1..n,1..n,[[float(0)$n]$n]);
        (erg[i,i]:= float(1);) $ i=1..n;
     end_if;

     return(myreturn());
  end_if;
//------------------------------------------------------------------------------------

  DIGITS:= DIGITS + 2; /*  increase working precision. There */
                       /*  can be further increases below    */

  /* --------------------------------------------------------------------------*/
  /*  Heuristics to estimate complexity of the two implemented Taylor methods. */
  /*  The number of terms in the Taylor expansion of exp(A) with               */
  /*  norm(A)=sollNorm is approximated by                                      */
  /*                                                                           */
  /*        (sollNorm)^i/i! <= exp(sollNorm)*10^(-DIGITS)                      */
  /*                                                                           */
  /*         -->   i >=  c(sollNorm)* DIGITS/ln(DIGITS)                        */
  /*         -->   i >=  c(sollNorm)* DIGITS/ln(DIGITS)                        */
  /*                                                                           */
  /*  with  c(0.01)=1.15, c(1.0)=3.0, c(3.0)=4.6, c(100.0)=38.9  (we use       */
  /*  sollNorm=3.0 in Version 1 and sollNorm=0.01 in Version 2).               */
  /*  This formula is exact for DIGITS=10 and works reasonably well for        */
  /*  DIGITS = 5 .. 1000. In the following we assume that A and v are dense.   */
  /*  The estimate will not be realistic for norm(A)<<sollNorm, because        */
  /*  c(sollNorm) should be replaced by c(norm(A)), which is unknown. Also for */
  /*  sparse A and v the estimate can be far from realistic.                   */
  /* --------------------------------------------------------------------------*/

  if method=0 then  /*  no method specified. Heuristics to choose between */
                    /*  Version 1 and Version 2 of TaylorExpansion        */

     /* ------------------------------------------------------------------------*/
     /*  complexity (Version 1 of TaylorExpansion)                              */
     /*    n^2                                  <- scale matrix                 */
     /*  + NN1*4.6*DIGITS/ln(1.0*DIGITS)*n^2*p; <- Taylor series v+A*v+A^2*v/2+ */
     /* ------------------------------------------------------------------------*/
     sollNorm:= 3.0;
     NN1:= max(1,norm_A/sollNorm);          /*  number of steps to undo scaling */

     /* ------------------------------------------------------------------------*/
     /*  complexity (Version 2 of Taylor Expansion)                             */
     /*     n^2                               <- scale matrix                   */
     /*   +(1.15*DIGITS/ln(1.0*DIGITS)-1)*n^3 <- Taylor series for scaled matrix*/
     /*   + NN2*n^3                           <- repeated squaring              */
     /*   + n^2*p;                            <- multiplication exp(A)*v        */
     /* ------------------------------------------------------------------------*/
     sollNorm:= 0.01;
     NN2:= max(0,ln(norm_A/sollNorm)/ln(2.0));  /*  steps in repeated squaring  */

     kk:= DIGITS/ln(float(DIGITS));
     userinfo(2, "estimating complexity of Taylor versions with/without repeated squaring:");
     userinfo(2, "  with/without = ".expr2text((NN2*n+p+(1.15*kk-1)*n)/(NN1*4.6*kk*p)));
     if NN1*4.6*kk*p < NN2*n + p + (1.15*kk-1)*n 
        then method:=1
        else method:=2;
     end_if;
  end_if;

  /* ---------------------------------------------------------------*/
  /*  Convert columns of A and v to polynomials. Important: get rid */
  /*  of float(0), otherwise A and v will always be dense!          */
  /* ---------------------------------------------------------------*/
  A:= subs(A, float(0)=0);
  x:= genident(): /*  unknown for polynomial representation */
  /*  APoly = list of polynomial columns */
  APoly:= [poly([[A[i,j],i] $ i=1..n],[x]) $ j=1..n];
  if expAv then  /*  convert array v to list of polynomials */
     v:= subs(v, float(0)=0);
     vPoly:=[poly([[v[i,j],i] $ i=1..n],[x]) $ j=1..p];
  end_if;

  /*  first version of MatrixMult only useful if B is sparse */
//MatrixMult:= proc(A,B)   /*  A and B = list of polynomials */
//local j,k,L,LL;  /*  The polynomials represent the columns */
//begin (L := [degree(nthterm(B[j],k)) $ k=1..nterms(B[j])];
//       LL:= [coeff(B[j])];
//       B[j]:= _plus(mapcoeffs(A[L[k]],_mult,LL[k]) $ k=1..nops(L));
//      ) $ j=1..nops(B); B;
//end_proc;

  /*  next version of MatrixMult is 50% faster, if B is dense */
  MatrixMult:= proc(A,B)    /*  A and B = list of polynomials */
  local j,k;        /*  The polynomials represent the columns */
  begin (B[j]:= _plus(mapcoeffs(A[k],_mult,coeff(B[j],k)) $ k=1..n);
        ) $ j=1..nops(B); B;
  end_proc;

  /*------------------------------------*/
  /* ---- now the real work starts ---- */
  /*------------------------------------*/

//-----------------------------------------------------------------------------
  /* -----------------------------------------------------------------*/
  /*  method1: Berechnung von exp(A)*v basierend auf der Taylor-Reihe */
  /*               exp(A)*v = v + A*v + A^2*v/2! + ...                */
  /* -----------------------------------------------------------------*/
  if method=1 then
     userinfo(1, "using Taylor expansion without repeated squaring");

     /*  put v= unit-matrix, if no v is given. This */
     /*  allows to compute exp(A) with this version */
     if not expAv then 
        p:=n; 
        vPoly:= [poly([[float(1), j]], [x]) $ j=1..n ]; 
     end_if;

     /* -------------------------------------------------------------------*/
     /*  scale matrix down to norm(A)<=sollNorm (=small). Large sollNorm   */
     /*  decreases the costs, but makes the summation of the Taylor series */
     /*  numerically unstable, if the matrix has large eigenvalues which   */
     /*  do not lie on the positive real axis. sollNorm:=3.0 seems to be a */
     /*  reasonable compromise between speed and stability.                */
     /* -------------------------------------------------------------------*/
     sollNorm:=3.0;
     if norm_A>sollNorm
        then NN:=ceil(norm_A/sollNorm);
             /*  scale A -> A/NN, i.e. norm((new)A)<= sollNorm */
             APoly:=map(APoly,mapcoeffs,_mult,float(1/NN));
             norm_A:= norm_A/NN;
        else NN:=1; 
     end_if;

     tol:= float(10^(-DIGITS));
     tol:= exp(norm_A)*tol;
     /*  How many terms of the Taylor series do we need   */
     /*  to achieve a relative precision <= 10^(-DIGITS)? */
     kk:=0; repeat kk:=kk+1; tol:=tol*kk until norm_A^kk<=tol end_repeat;

     DIGITS:= DIGITS + max(0,ceil(ln(float(kk*(0.01+NN)))/ln(10.0)));

     /*  Compute (1+A+A^2/2+..)*(1+A+A^2/2+..)*...*(1+A+A^2/2+..)*v */
     ergPoly:= vPoly;
     (Apower_v:= ergPoly;
      (/*  compute next Apower_v = A^i*v/i! and add to ergPoly */
       Apower_v:=map(MatrixMult(APoly,Apower_v),mapcoeffs,_mult,1/i);
       ergPoly:= zip(ergPoly,Apower_v,_plus);
       ) $ i=1..kk;
      ) $ j=1..NN;

     case return_type 
       of 1 do
            erg:= [coeff(ergPoly[1],i) $ i=1..n];
            break;
       of 2 do 
            erg:= array(1..n,[coeff(ergPoly[1],i) $ i=1..n]);
            break;
       otherwise
            erg:= array(1..n,1..p,[[coeff(ergPoly[j],i) $ j=1..p] $i=1..n]);
     end_case;
     // The zero entries are 0. Convert to float(0):
     erg:= map(erg, float);
     // return(erg);
  end_if;

//-----------------------------------------------------------------------------
  /* ------------------------------------------------------------------------*/
  /*  method 2: Berechnung von exp(A) basierend auf der Taylor-Reihe von     */
  /*  exp(A/2^NN) mit 2^NN=ceil(norm(A)/sollNorm) und repeated squaring      */
  /*  dieser Matrix. There is no need for stabilizing measures in computing  */
  /*  exp(A/2^NN), since norm(A/2^NN)<= small sollNorm                     . */
  /* ------------------------------------------------------------------------*/
  if method=2 then
     userinfo(1, "using Taylor expansion with repeated squaring");

     tol:=float(10^(-DIGITS));
     sollNorm:=0.01;
     if norm_A>sollNorm
        then NN:=ceil(ln(norm_A/sollNorm)/ln(2.0));
             /*  scale A -> A/2^NN, i.e. norm((new)A)<= sollNorm */
             APoly:=map(APoly,mapcoeffs,_mult,float(1/2^NN));
             norm_A:= norm_A/2^NN;
        else NN:=0; 
     end_if;

     /*  How many terms of the Taylor series do we need */
     /*  to achieve a relative precision <= tol?        */
     tol:= exp(norm_A)*tol;
     kk:=0; repeat kk:=kk+1; tol:=tol*kk until norm_A^kk<=tol end_repeat;

     DIGITS:= DIGITS + max(0,ceil(ln(float(kk*(0.01+NN)))/ln(10.0)));
     /*  start to compute Taylor series ergPoly = 1 + A + A^2/2! + ... */
     Apower:= APoly;
     ergPoly:= [poly([[float(1),j]],[x])+Apower[j] $ j=1..n];  /*  1 + A */
     (/*  compute next Apower = A^k/k! and add to ergPoly */
       Apower:= map(MatrixMult(APoly,Apower),mapcoeffs,_mult,1/k);
       ergPoly:= zip(ergPoly,Apower,_plus);
     ) $ k=2..kk;

     /*  repeated squaring on ergPoly=exp(A/2^NN) to compute (ergPoly)^(2^NN) */
     (ergPoly:= MatrixMult(ergPoly,ergPoly);) $ i=1..NN;

     if expAv then 
          ergPoly_v:=MatrixMult(ergPoly,vPoly);
          case return_type
          of 1 do 
               erg:= [coeff(ergPoly_v[1],i)$ i=1..n];
               break;
          of 2 do 
               erg:= array(1..n,[coeff(ergPoly_v[1],i) $ i=1..n]);
               break;
          otherwise  
               erg:= array(1..n,1..p,[[coeff(ergPoly_v[j],i)$ j=1..p] $ i=1..n]);
          end_case;
     else 
          erg:= array(1..n,1..n,[[coeff(ergPoly[j],i)$ j=1..n] $ i=1..n]);
     end_if;
     // The zero entries are 0. Convert to float(0):
     erg:= map(erg, float);
     // return(erg);
  end_if;

//-----------------------------------------------------------------------------
  /*  ---------------------------------------------------------------*/
  /*  method 3: Berechnung von exp(A)*v basierend auf Krylovraeumen. */
  /*  ---------------------------------------------------------------*/
  if method=3 then
    userinfo(1, "using Krylov method");

    if (not expAv) or (p <> 1) then
       error("the 'Krylov' method is only implemented for exp(Matrix)*Vector");
    end_if;

    if map({op(v)}, domtype) minus {DOM_FLOAT,DOM_COMPLEX}<> {}
     then error("non-numerical entry in vector")
    end_if;

    /*  ----------------------------------------------------------------*/
    /*  For an n x n input matrix A the Arnoldi-algorithm computes an   */
    /*  upper Hessenberg matrix Hmm of dimension m x m and approximates */
    /*  exp(A)*v by                                                     */
    /*     exp(A)*v = norm(v)* V_m * exp(Hmm) * e_1                     */
    /*  with the orthonormal n x m matrix V_m spanned by the Krylov     */
    /*  vectors v,A*v,A^2v,..,A^(m-1)*v and the first unit m-vector     */
    /*  e1:=[1,0$m-1].                                                  */
    /*  In principle one can try m = 1,2,3,...,n until the approximation*/
    /*  is good enough. This is expensive, if the process does not stop */
    /*  with m<<n. Therefore we do not use all m in [1,2,3,..,n],  but  */
    /*  start with m=1 and then increase m -> sqrt(2)*m. With costs     */
    /*  O(m^3) this ensures that unsuccesful attempts do not increase   */
    /*  the total costs by more than a factor of 2, if m runs to n.     */
    /*  ----------------------------------------------------------------*/

    tol:= float(10^(-DIGITS));
    vNorm:=float(specfunc::sqrt(_plus(specfunc::abs(v[i,1])^2 $ i=1..n))):
    KrylovBase[1]:=mapcoeffs(vPoly[1],_mult,1/vNorm); /*  vv = v/norm(v) */

    /* --------------------------------------------------------*/
    /*  compute Krylov basis v[j]=KrylovBase[j] and upper mxm  */
    /*  Hessenberg matrix H by the Arnoldi algorithm:          */
    /*                v[1] := v/norm(v);                       */
    /*                for j=1..m do                            */
    /*                    w = A*v[j];                          */
    /*                    for k=1..j do                        */
    /*                        H[k,j]:= <w,v[k]>;               */
    /*                        w:= w - H[k,j]*v[k];             */
    /*                    end_for;                             */
    /*                    if norm(w)<eps then Abbruch; end_if; */
    /*                    H[j+1,j]:= norm(w);                  */
    /*                    v[j+1]:= w/H[j+1,j];                 */
    /*                end_for;                                 */
    /*  Result: with orthonormal KrylovBase V=[v[1],..,v[m]]   */
    /*                                                         */
    /*   A*V = V*H + H[m+1,m]*v[m+1]*e_m^T,  H=V^T*A*V         */
    /*                                                         */
    /*      exp(A)*v =(approx)= norm(v)*V*exp(H)*e1            */
    /* --------------------------------------------------------*/

    /*  H = storage for Hessenberg matrix used in arnoldi. Its upper left m+1 x m */
    /*  part is used in arnoldi(..,m). Row m+1 is used for estimating the error.  */
    H:= array(1..n+1,1..n,[[0$n]$n+1]); 

    /* subroutine arnoldi: transforms matrix A (=APoly) to upper MxM Hessenberg    */
    /* matrix H and computes orthonormal nxM transformation matrix V = KrylovBase: */

    arnoldi:=proc(last_m,M) 
    local i, j, k, wPoly, wNorm;
    begin
      for j from last_m+1 to M do
        wPoly:= MatrixMult(APoly,[KrylovBase[j]])[1];
        wNorm:= float(specfunc::sqrt(_plus(specfunc::abs(coeff(wPoly,k))^2$ k=1..n)));
        (H[k,j]:= /* scalar_product(wPoly,KrylovBase[k])*/
            _plus(_mult(conjugate(coeff(KrylovBase[k],i)),coeff(wPoly,i)) $ i=1..n);
         wPoly:= (wPoly-mapcoeffs(KrylovBase[k],_mult,H[k,j])); /* w:=w-vv*H[k,j] */
        ) $ k=1..j;
        /* H[j+1,j] = norm(w) */
        H[j+1,j]:=float(specfunc::sqrt(_plus(specfunc::abs(coeff(wPoly,k))^2$ k=1..n)));
        /* if new norm(w) < norm(old w)*tol, then new w is */
        /* dominated by roundoff and it is time to stop!   */
        if specfunc::abs(H[j+1,j])<=wNorm*tol then M:=j; m:=j; break; end_if; /* Abbruch */
        KrylovBase[j+1]:= mapcoeffs(wPoly,_mult,1/H[j+1,j]);
      end_for;
      /* return upper M x M part of Hessenberg matrix H */
      array(1..M,1..M,[[H[i,k] $ k=1..M] $ i=1..M]);
    end_proc;

    last_m:=0; m:=1;

    while m<=n do 
      /* Compute the Krylov base vectors spanned by A^(last_m+1)*v,..,A^m*v    */
      /* and the corresponding upper Hessenberg matrix Hmm:= arnoldi(last_m,m).*/
      /* Then compute exp(Hmm)*e1 as a list: */
      Hmm:=arnoldi(last_m,m); 
        /* Warning: arnoldi can change m as a side effect. Need to call arnoldi */
        /* first, before passing it to expMatrix. Otherwise vector [1,0$m-1]    */
        /* might have wrong dimension.*/
      userinfo(1, "calling expMatrix with ".expr2text(m)." x ".
                                            expr2text(m). " Hessenberg matrix");
      expHe1:= [op(numeric::expMatrix(Hmm,[1,0$m-1]))];
      /* Compute exp(A)*v:=Krylovbase*e^Hmm*e_1*Norm(v)*/
      ergPoly:= _plus(mapcoeffs(KrylovBase[j],_mult,expHe1[j]) $ j=1..m);
      /* Bestimme die Norm des Ergebnisses.                                       */
      norm_Erg:=float(specfunc::sqrt(_plus(specfunc::abs(coeff(ergPoly,i))^2 $ i=1..n)));
      if m<n then
         /* estimate error norm(exp(A)*v - KrylovBase*exp(Hmm)*e1) */
         Err:= specfunc::abs(expHe1[m])*H[m+1,m];
         userinfo(2, "estimate of absolute error: ".expr2text(Err*vNorm));
         if Err<=norm_Erg*tol then break end_if; /* Abbruchkriterium */
      end_if;
      if m=n then break end_if; /* ultimatives Abbruchkriterium */
      last_m:=m;
      /* increase m by a factor of sqrt(2) */
      m:= min(ceil(1.41*m),n);
    end_while;

    userinfo(1, expr2text(n)." x ".expr2text(n)." matrix reduced to ".
                expr2text(m)." x ".expr2text(m). " Hessenberg matrix");
    ergPoly:= mapcoeffs(ergPoly,_mult,vNorm);
    case return_type
      of 1 do 
           erg:= [coeff(ergPoly,i) $ i=1..n];
           break;
      of 2 do
           erg:= array(1..n,[coeff(ergPoly,i) $ i=1..n]);
           break;
      otherwise
           erg:= array(1..n,1..1,[[coeff(ergPoly,i)]$ i=1..n]);
    end_case;
     // The zero entries are 0. Convert to float(0):
     erg:= map(erg, float);
    // return(erg):
  end_if;

//-----------------------------------------------------------------------------
  /*-------------------------------------------------------------------*/
  /* method 4: compute exp(A), exp(A)*v by polynomial interpolation.   */
  /* Idea:  if    p(lambda) = exp(lambda) for all eigenvalues          */
  /*        and  p'(lambda) = exp(lambda) for all double eigenvalues   */
  /*        and p''(lambda) = exp(lambda) for all triple eigenvalues   */
  /*        etc.                                                       */
  /*        then p(A) = exp(A).                                        */
  /*        Hence, compute interpolating polynomial p through          */
  /*        [lambda1,exp(lambda1)], [lambda2,exp(lambda2)], ...        */
  /*-------------------------------------------------------------------*/
  if method=4 then
      userinfo(1, "interpolating exp(x) by a polynomial:");

      tol:= float(10^(-DIGITS));
      eigenvals:= map(numeric::eigenvalues(A),float):
      DIGITS:= 2*DIGITS+2+round(n/10);
      absolute_tol:=max(op(map(eigenvals,specfunc::abs)))*tol;
      (if specfunc::abs(eigenvals[j])<=absolute_tol then
          eigenvals[j]:=float(0) end_if; ) $ j=1..n;
      userinfo(2, "interpolation nodes = eigenvalues = ".expr2text(eigenvals));
      /* compute coefficients c[i] of interpolating polynomial */
      /*  pp(A) = c[0]+c[1]*A+..+c[m]*A^m                      */
      m:=n-1;
      /* converting list to array(0..m) */
      eigenvals:= array(0..m, eigenvals);
      c:=map(eigenvals,exp);
      (for j from m downto k do
          if specfunc::abs(eigenvals[j]-eigenvals[j-k])<=absolute_tol
             then c[j]:=c[j]/k; /*Vorsicht!*/
             else c[j]:=(c[j]-c[j-1])/(eigenvals[j]-eigenvals[j-k]);
          end_if;
       end_for:) $ k=1..m;
      (for j from m-1 downto k-1 do
          c[j]:=c[j]-eigenvals[j+1-k]*c[j+1] end_for: ) $ k=1..m;
      userinfo(2, "coefficients of the interpolating polynomial: ");
      userinfo(2, expr2text(op(c)));

      /* heuristic check, whether interpolation polynomial is ok */
      /* within numerical precision. We should have              */
      /*              pp(eigenval)=exp(eigenval)                 */
      /* Evaluation via Horner scheme:                           */
      (pp:= c[m]: (pp:= eigenvals[i]*pp+c[m-j];) $ j=1..m;
       if specfunc::abs(pp-exp(eigenvals[i]))>absolute_tol
          then error("interpolation is numerically unstable");
      end_if; ) $ i=0..m;

      /* Horner schemes:                                     */
      /* evaluate (c[0]+c[1]*A+c[2]*A^2+..+c[m]*A^m)*v by    */
      /*     t:= c[m]*v;                                     */
      /*     for i from m downto 0 do t:=A*t+c[i]*v end_for; */
      /* evaluate c[0]+c[1]*A+c[2]*A^2+..+c[m]*A^m by        */
      /*     t:= c[m]*A;                                     */
      /*     for i from m downto 1 do t:=A*t+c[i]*A end_for; */
      /*     t:= t+ c[0];                                    */
      if expAv then 
              tPoly:= map(vPoly,mapcoeffs,_mult,c[m]);
              (tPoly:= zip(MatrixMult(APoly,tPoly),
                          map(vPoly,mapcoeffs,_mult,c[m-i]),_plus);
               ) $ i=1..m;
         else tPoly:= map(APoly,mapcoeffs,_mult,c[m]);
              (tPoly:= zip(MatrixMult(APoly, tPoly),
                          map(APoly,mapcoeffs,_mult,c[m-i]),_plus)
              ) $ i=1..m-1;
              (tPoly[k]:= tPoly[k] + poly([[c[0],k]],[x]);) $ k=1..n;
      end_if;

    /* If matrix A is real, then exp(A) is real, too. Get rid   */
    /* of imaginary numerical junk by applying Re to the result */
    if has(map({coeff(APoly[j]) $ j=1..n},domtype),DOM_COMPLEX)
       then A_is_real:= FALSE else A_is_real:= TRUE;
    end_if;
    if expAv and has(map({coeff(vPoly[j]) $ j=1..p},domtype),DOM_COMPLEX)
       then v_is_real:= FALSE else v_is_real:= TRUE;
    end_if;

    if expAv then 
       case return_type
       of 1 do
            erg:= [coeff(tPoly[1],i) $ i=1..n];
            break;
       of 2 do
            erg:= array(1..n,[coeff(tPoly[1],i) $  i=1..n]);
            break;
       otherwise
            erg:= array(1..n,1..p,[[coeff(tPoly[j],i) $ j=1..p] $ i=1..n]);
       end_case;
       if A_is_real and v_is_real then
          erg:= map(erg,Re);
       end_if;
    else
       erg:= array(1..n,1..n,[[coeff(tPoly[j],i) $ j=1..n] $ i=1..n]);
       if A_is_real then
          erg:= map(erg, Re);
       end_if;
    end_if;
    // The zero entries are 0. Convert to float(0):
    erg:= map(erg, float);
    // return(erg);
  end_if;

//-----------------------------------------------------------------------------
  /*------------------------------------------------------------*/
  /* method 5: compute exp(A), exp(A)*v  by diagonalization     */
  /*          A  = X * diag(l1,l2,..)           * X^(-1)        */
  /*       exp(A)= X * diag(exp(l1),exp(l2),..) * X^(-1)        */
  /* Warning: if no diagonalization exists, then X will not be  */
  /*          invertible and one gets an error when computing   */
  /*          X^(-1) or X^(-1) will be wrong because of         */
  /*          numerical instability;                            */
  /* This method is fail safe for hermitean or skew-hermitean A */
  /*------------------------------------------------------------*/
  if method=5 then
    userinfo(1, "diagonalizing matrix");
    DIGITS:= DIGITS + round(ln(n)/ln(10.0));
    /* If matrix A is real, then exp(A) is real, too. Get rid   */
    /* of imaginary numerical junk by applying Re to the result */
    if has(map({coeff(APoly[j]) $ j=1..n},domtype),DOM_COMPLEX)
       then A_is_real:= FALSE else A_is_real:= TRUE;
    end_if;
    if expAv and has(map({coeff(vPoly[j]) $ j=1..p},domtype),DOM_COMPLEX)
       then v_is_real:= FALSE else v_is_real:= TRUE;
    end_if;
    ([d,X,dummy]):= numeric::eigenvectors(A,NoErrors);
    d:=map(d,float@exp):

    if expAv then 
         A:=numeric::matlinsolve(X,v)[1];
         ((A[i,j]:=d[i]*A[i,j]) $ i=1..n) $ j=1..p ;
         A:= array(1..n,1..p,[[_plus(X[i,k]*A[k,j]$ k =1..n) $ j=1..p] $ i=1..n]);
         if A_is_real and v_is_real then 
            A:= map(A,Re)
         end_if;
         case return_type
            of 1 do 
                 erg:= [A[i,1] $ i=1..n];
                 break;
            of 2 do
                 erg:= array(1..n,[A[i,1] $ i=1..n]);
                 break;
            otherwise
                 erg:= A;
         end_case;
    else A:=numeric::inverse(X):
         ((A[i,j]:=d[i]*A[i,j]) $ j=1..n) $ i=1..n;
         A:= array(1..n,1..n,[[_plus(X[i,k]*A[k,j]$ k=1..n) $ j=1..n] $ i=1..n]);
         if A_is_real then 
              erg:= map(A,Re);
         else erg:= A; 
         end_if;
    end_if;
    // return(erg);
  end_if;
  //-----------------------------------------------------------------------------

  //-------------------------
  //------- return ----------
  //-------------------------
  return(myreturn());

end_proc:
