/* W. Oevel, 17.2.98 */
/* 29.10.01: Scilab eingebaut */
/* 10.08.07: Scilab durch eigene Routinen ersetzt */
/*-------------------------------------------------------------------------------*/
/* numeric::fMatrix - functional calculus for matrices                           */
/*                                                                               */
/* Aufruf:  numeric::fMatrix(f, A, param1, param2, ..,                           */
/*                           <hard_soft>, <ReturnType = d>,  <NoWarning>)        */
/*                                                                               */
/* Parameter:                                                                    */
/*   f                  : procedure, f(a, param1, param2, ...) with a in C       */
/*                                   must produce scalar values                  */
/*   A                  : diagonalizable numerical square matrix                 */
/*                        (array, hfarray, or Cat::Matrix)                       */
/*   param1, param2, .. : numerical or symbolical parameters                     */
/*   hard_soft          : Hard, HardwareFloat, Soft, SoftwareFloats              */
/*   d                  : DOM_ARRAY, Dom::DenseMatrix() or Dom::Matrix()         */
/*                                                                               */
/* Rueckgabewert:                                                                */
/*   f(A, param1, param2,..) = array, hfarray, densematrix or matrix             */
/*   FAIL is returned if the routine thinks that A is not diagonalizable         */
/*                                                                               */
/* Beschreibung:                                                                 */
/*                                                                               */
/*   Die Matrix A wird diagonalisiert: A = X diag(lambda1,lambda2,..) X^(-1).    */
/*   Das Ergebnis wird per                                                       */
/*   f(A,p1,p2,..) = X diag(f(lambda1,p1,p2,..),f(lambda2,p1,p2,..),..) X^(-1)   */
/*   berechnet.                                                                  */
/*   Achtung: dies funktioniert nicht fuer beliebiges A! Nichtdiagonalisierbare  */
/*   Matrizen fuehren zu einer Fehlermeldung, bzw. es kommt zu einer grossen     */
/*   Verstaerkung von Rundungsfehlern, wenn die Nichtdiagonalisierbarkeit        */
/*   numerisch nicht ermittelt werden kann.                                      */
/*   Diese Methode ist aber FAIL safe fuer hermitesche oder schief-hermitesche   */
/*   Matrizen A.                                                                 */
/*                                                                               */
/* Examples:                                                                     */
/*  >> A:= array(1..2,1..2,[[0,1],[-1,0]]):                                      */
/*  >> B:= numeric::fMatrix(sqrt,A);                                             */
/*                                                                               */
/*                   +-                             -+                           */
/*                   |   0.7071067811, 0.7071067811  |                           */
/*                   |                               |                           */
/*                   |  -0.7071067812, 0.7071067811  |                           */
/*                   +-                             -+                           */
/*                                                                               */
/*  >> numeric::fMatrix(_power, B, 2);                                           */
/*                                                                               */
/*    +-                                                                  -+     */
/*    |  3.833233541e-20 - 2.439454887e-19 I, 10.0e-1 - 5.421010862e-20 I  |     */
/*    |                                                                    |     */
/*    |     - 10.0e-1 + 2.710505431e-20 I,          4.065758146e-20        |     */
/*    +-                                                                  -+     */
/*                                                                               */
/*  >> numeric::fMatrix(exp@_mult, A, t);                                        */
/*                                                                               */
/*    +-                                                                    -+   */
/*    |    0.5 exp(-I t) + 0.5 exp(I t),   0.5 I exp(-I t) - 0.5 I exp(I t)  |   */
/*    |                                                                      |   */
/*    |  0.5 I exp(I t) - 0.5 I exp(-I t),   0.5 exp(-I t) + 0.5 exp(I t)    |   */
/*    +-                                                                    -+   */
/*                                                                               */
/*-------------------------------------------------------------------------------*/

numeric::fMatrix:=proc(f,A)
local Args, useHardwareFloats, HardwareFloatsRequested,
      request, returnType, 
      B, m, n, d, dd, X, dummy, i, j, k, dowarn, nowarn,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
begin
  if args(0) < 2 then error("not enough arguments") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  if domtype(A) = DOM_ARRAY or
     domtype(A) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("second argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     originalrowrange:= 1..n:
     originalcolrange:= 1..m:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults

  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  request:= null();
  dowarn:= TRUE;
  nowarn:= null();

  // Args is the list of further parameters for f
  // and the options for numeric::fMatrix.
  // Search for the options and eliminate them
  // from the list:
  Args:= [args(i) $ i = 3..args(0)];
  for i from nops(Args) downto 1 do
      case Args[i]
      of Hard do
      of HardwareFloats do
                     request:= HardwareFloats;
                     useHardwareFloats:= TRUE:
                     HardwareFloatsRequested:= TRUE;
                     delete Args[i];
                     break;
      of Soft do
      of SoftwareFloats do 
                     request:= SoftwareFloats;
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     delete Args[i];
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     nowarn:= NoWarning;
                     delete Args[i];
                     break;
      otherwise
         if type(Args[i]) = "_equal" and
             lhs(Args[i]) = ReturnType then
              returnType:= rhs(Args[i]);
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(op(Args[i], 2)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
              delete Args[i];
         end_if;
      end_case;
  end_for:

  //------------------------------------------------------------
  // Trivial case: 1 x 1 matrices
  //------------------------------------------------------------

  if n=1 then 
     B := array(1..1,1..1,[[f(float(A[1,1]), op(Args))]]);
  else

  //------------------------------------------------------------
  // Step 1: compute the diagonalization
  //------------------------------------------------------------

  userinfo(1, "diagonalizing matrix");
  // If request = null(), numeric::eigenvectors decides on 
  // its own whether to use HardwareFloats or SoftwareFloats

  if useHardwareFloats then
    [d, X, dummy]:= numeric::eigenvectors(
                A, request, NoErrors, ReturnType = DOM_HFARRAY, nowarn);
  else
    [d, X, dummy]:= numeric::eigenvectors(
                A, request, NoErrors, ReturnType = DOM_ARRAY, nowarn);
  end_if:

  userinfo(1, "the eigenvalues are ".expr2text(d));
  userinfo(1, "applying the function to the eigenvalues:");

  //------------------------------------------------------------
  // Step 2: apply the function to the eigenvalues
  //------------------------------------------------------------

  d:=map(d, f, op(Args)):
  userinfo(1, expr2text(d));

  //------------------------------------------------------------
  // Step 3:  result = X*diag(f(eigenvalues))*X^(-1)
  //------------------------------------------------------------

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  // Shall X*d*X^(-1) be computed by HardwareFloats?
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("inverse") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "using hardware floats"):

     if domtype(X) <> DOM_HFARRAY then
        if traperror((B:= hfarray(1..n, 1..n, [op(X)]))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed, ".
                        "using 'SoftwareFloats' instead");
             end_if;
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
        end_if;
        X:= B;
     else
        B:= X;
     end_if;

     if traperror((
            B:= hfa::inverse(B)
        )) <> 0 or
        has(B, RD_NAN) or
        has(B, RD_INF) or
        has(B, RD_NINF) then
          if HardwareFloatsRequested and dowarn then
             warning("HardwareFloats failed, ".
                     "using 'SoftwareFloats' instead");
          end_if;
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if;

     if B = FAIL then
        return(FAIL);
     end_if:


     if map({op(d)}, domtype) minus {DOM_FLOAT, DOM_COMPLEX} = {} then
        dd:= hfarray(1..n, 1..n):
        (dd[i, i]:= d[i]) $ i = 1..n;
        B:= X*(dd*B);
     else
        B:= array(1..n, 1..n, [[_plus(X[i,k]*d[k]*B[k, j] $ k = 1..n) $ j = 1..n] $ i = 1..n]);
     end_if;
     if has(B, RD_NAN) or
        has(B, RD_INF) or
        has(B, RD_NINF) then
           if HardwareFloatsRequested and dowarn then
              warning("HardwareFloats failed, ".
                      "using 'SoftwareFloats' instead");
           end_if;
           userinfo(1, "'HardwareFloats' failed"):
           useHardwareFloats:= FALSE;
           break;
     end_if;
     break;
  end_case:

  case useHardwareFloats // use case because we want to use break below
  of FALSE do
     userinfo(1, "using 'SoftwareFloats'");
     userinfo(1, "inverting eigenvector matrix");
     A:= numeric::inverse(X, SoftwareFloats, ReturnType = DOM_ARRAY, nowarn):
     if A = FAIL then
        return(FAIL);
     end_if:
     for i from 1 to n do
       (A[i,j]:=d[i]*A[i,j]) $ j=1..n;
     end_for;
     userinfo(1, "building result");
     B:= array(1..n,1..n,[[_plus(X[i,k]*A[k,j]$k=1..n)$j=1..n]$i=1..n]);
     break;
 end_case:
 end_if; // if n = 1

  //------- return ----------
  //-------------------------
  if B::dom = returnType and
     (not rowrangechanged) and
     (not colrangechanged) then
       return(B):
  end_if:
  case returnType
     // mind the ordering: process DOM_{HF}ARRAY first to
     // avoid unnecessary loading of the DOMAIN package
  of DOM_ARRAY do
     if domtype(B) <> DOM_ARRAY or rowrangechanged or colrangechanged then
        B:= array(originalrowrange, originalcolrange, [op(B)]);
     end_if;
     return(B);
  of DOM_HFARRAY do
     if domtype(B) <> DOM_HFARRAY then
        if traperror((
            A:= hfarray(originalrowrange, originalcolrange, [op(B)])
           )) <> 0 or
           has(A, RD_NAN) or
           has(A, RD_INF) or
           has(A, RD_NINF) then
              if dowarn then
                 warning("cannot return the resulting matrix as an hfarray. ".
                         "Returning it as an array of type DOM_ARRAY instead");
              end_if:
              return(array(originalrowrange, originalcolrange, [op(B)]));
        end_if:
        return(A);
     elif rowrangechanged or colrangechanged then
      //B:= subsop(B, 0 = (2, originalrowrange, originalcolrange));
        B:= hfarray(originalrowrange, originalcolrange, [op(B)]);
     end_if;
     return(B);
  of Dom::Matrix() do
       return(returnType(B));
  of Dom::DenseMatrix() do
       return(returnType::create(B));
  otherwise
       error("unexpected return type"):
  end_case;
end_proc:
