
//------   file  factorCh.mu   29.9.97  ---------------------
/*  Help file:

 factorCholesky - numerical Cholesky factorization of a matrix

 Call:

 numeric::factorCholesky( A <, hard_soft> <,Symmetric> <, Symbolic>
                          <, NoCheck> <, ReturnType = typ )
                                                                   
 Parameter:

  A         -- a square real or complex matrix of domtype DOM_ARRAY
               or Cat::Matrix
  hard_soft -- either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'
               If DIGITS <=15, the default is HardwareFloats
               If DIGITS  >15, the default is SoftwareFloats

  typ       -- either DOM_ARRAY, DOM_HFARRAY, densematrix or matrix
                                                                   
Synopsis:

  numeric::factorCholesky(A); returns the lower triangular
  Cholesky factor L of the factorization A=LL^H, where L^H 
  denotes the Hermitean transpose of L. The result is of 
  domtype DOM_ARRAY. It is normalized such that the diagonal
  of L is real and positive.

  numeric::factorCholesky(A, Symmetric); returns the lower triangular
  Cholesky factor L of the factorization A=LL^T, where L^T 
  denotes the usual transpose of L. The result is of 
  domtype DOM_ARRAY. 
                                                                  
  Non-float numerical entries are admissible, they will be 
  evaluated to floats. Conversion to floats may be suppressed
  by using the option Symbolic.
 
  If the matrix is not Hermitean or not positive definite 
  and the option <Symmetric> is not used, then
  the computation will terminate with an error message.
  Checking of these properties can be suppressed with the option
  NoCheck. This option should be used, whenever the matrix contains 
  indeterminates. If numeric symbolic entries such as PI, sqrt(2) etc.
  are present, then these properties are checked numerically.

  Examples: 

>> A := array( 1..2 , 1..2 , [ [1,2*I],[-2*I,5] ] ):

>> numeric::factorCholesky(A);                                

                            +-              -+
                            |    1.0,    0   |
                            |                |
                            |  - 2.0 I, 1.0  |
                            +-              -+

>> numeric::factorCholesky(A,Symbolic);

                              +-          -+
                              |    1,   0  |
                              |            |
                              |  - 2 I, 1  |
                              +-          -+

>> A := array( 1..2 , 1..2 , [ [-1 ,1],[1,1] ] ):

>> numeric::factorCholesky(A);                                

Error: matrix is not positive definite within working precision [nume\
ric::factorCholesky]

>> numeric::factorCholesky(A, Symmetric);                                

                     +-                        -+
                     |   I,          0          |
                     |                          |
                     |  - I, 1.414213562 + 0 I  |
                     +-                        -+

See also: linalg::factorCholesky, numeric::factorLU , numeric::factorQR 
                                                                  */
//-----------------------------------------------------------------
numeric::factorCholesky:=proc(A)
local useHardwareFloats, HardwareFloatsRequested, 
      returnType, symbolic_mode, symmetricDecomp, 
      dowarn, specialwarning, warning_given, do_check,
      m, n, real,i,j,k,B,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
begin
  if args(0) < 1 then error("expecting at least one argument") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  if domtype(args(1)) = DOM_ARRAY or
     domtype(args(1)) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     originalrowrange:= 1..n:
     originalcolrange:= 1..m:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  do_check:=TRUE; 
  symbolic_mode:=FALSE; 
  symmetricDecomp:= FALSE;
  dowarn:= TRUE;
  specialwarning:= FAIL;

  for i from 2 to args(0) do
      case args(i)
      of Symbolic do symbolic_mode := TRUE;
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Hard do
      of HardwareFloats do
                     if symbolic_mode then
                        specialwarning:= 
                                "ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'":
                        useHardwareFloats:= FALSE:
                     else
                        useHardwareFloats:= TRUE:
                        HardwareFloatsRequested:= TRUE;
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Symmetric do
                     symmetricDecomp:= TRUE;  
                     break;
      of NoCheck do do_check := FALSE; 
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("factorCh") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "using hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
        if traperror((B:= hfarray(1..m, 1..n, [op(A)]))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
        end_if;
     else
        B:= A;
     end_if;

     if traperror((B:= hfa::factorCh(B, symmetricDecomp, do_check))) <> 0 or
           domtype(B) = DOM_STRING or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             if domtype(B) = DOM_STRING then
                error(B);
             end_if:
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
     end_if;

     //------------------------------
     //-------- return --------------
     //------------------------------

     if B = FAIL then
        return(FAIL);
     end_if:

     // proceed to the end of this routine. There, the
     // matrix is converted to the requested return type.
     // Further, rowrange and colrange are adapted if necessary.
     A:= B:
  end_case;
  //---------------------------------
  //----- end of HardwareFloats -----
  //---------------------------------

  case useHardwareFloats
  of FALSE do
  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  userinfo(1, "using software floats"):

  // Do use domtype(A) <> DOM_ARRAY here to avoid
  // unnecessary loading of the DOMAINS package
  if domtype(A) <> DOM_ARRAY and
     domtype(A) <> DOM_HFARRAY and
     A::dom::hasProp(Cat::Matrix)=TRUE then
     // convert to domtype DOM_ARRAY for speed
     A:= expr(A);
  end_if;
  if domtype(A) <> DOM_ARRAY and
     domtype(A) <> DOM_HFARRAY then
     error("the matrix must be an array, an hfarray, or of category 'Cat::Matrix'");
  end_if:

  if domtype(A) = DOM_ARRAY and
     symbolic_mode and 
     do_check and
     (  (indets(A)<>{}) or has({op(map(A,domtype))}, DOM_NIL) )
  then do_check:=FALSE; 
       if dowarn then
         warning("symbolic indeterminate found, switching to option 'NoCheck'");
       end_if:
  end_if;

  if domtype(A) = DOM_ARRAY and
     not symbolic_mode then
    B:=map(A,float);
    if ({op(map(B,domtype))} minus {DOM_FLOAT,DOM_COMPLEX}) <> {}
      then symbolic_mode:=TRUE;
           do_check:=FALSE;
           if dowarn then
             warning("symbolic entry found, switching to options 'Symbolic', 'NoCheck'");
           end_if:
      else A:=B;
    end_if:
  end_if;

  if domtype(A) = DOM_HFARRAY then
     // convert to DOM_ARRAY to make indexed
     // reading and writing faster
     A:= array(1..n, 1..m, [op(A)]):
  end_if:

  if symmetricDecomp then 
     for i from 1 to n do
       for j from 1 to i-1 do
          if A[i,j]<>A[j,i] then error("the matrix is not symmetric") end_if;
       end_for;
     end_for;
     for i from 1 to n do
       (A[i,j]:=(A[i,j]-_plus(A[i,k]*A[j,k] $ k=1..j-1))/A[j,j]) $ j=1..i-1;
        A[i,i]:=A[i,i]-_plus(A[i,k]^2 $ k=1..i-1);
        A[i,i]:=(A[i,i])^(1/2);
       if symbolic_mode then
            (A[i,j]:= 0) $ j=i+1..n;
       else (A[i,j]:= float(0)) $ j=i+1..n;
       end_if;
        if i<n and iszero(A[i,i]) then 
            error("a regular symmetric factorization does not exist") 
        end_if;
     end_for;
  else 
     real:=TRUE;
     if do_check then
       for i from 1 to n do
         for j from 1 to i-1 do
           if iszero(Im(A[i,j]))
              then if A[i,j]<>A[j,i] then error("the matrix is not hermitean") end_if;
              else if A[i,j]<>conjugate(A[j,i]) then error("the matrix is not hermitean") end_if:
                   real:= FALSE;
           end_if;
         end_for;
         if not iszero(Im(A[i,i]))then error("the matrix is not hermitean"); end_if;
       end_for;
     end_if;
     warning_given:= FALSE;

     for i from 1 to n do
       if real
       then (A[i,j]:= (A[i,j]-_plus(A[i,k]*A[j,k] $ k=1..j-1))/A[j,j])
                          $ j=1..i-1;
       else (A[i,j]:= (A[i,j]-_plus(A[i,k]*conjugate(A[j,k]) $ k=1..j-1)
                      )/A[j,j]) $ j=1..i-1;
       end_if;
       if symbolic_mode
         then A[i,i]:= A[i,i] - _plus(abs(A[i,k])^2 $ k=1..i-1);
         else A[i,i]:= A[i,i] - _plus(specfunc::abs(A[i,k])^2 $ k=1..i-1);
       end_if;
       if do_check then
         if traperror(bool(A[i,i] <= 0))<>0 then 
             if dowarn and not warning_given then
                warning("positiveness checked numerically");
                warning_given:=TRUE;
              end_if;
              if float(A[i,i]) <= 0 then
                 error("the matrix is not positive definite within working precision");
              end_if;
         else if A[i,i] <= 0 then
                 if symbolic_mode
                  then error("the matrix is not positive definite");
                  else error("the matrix is not positive definite within working precision");
                 end_if;
              end_if;
         end_if;
       end_if;
       A[i,i]:=(A[i,i])^(1/2);
       if symbolic_mode then
            (A[i,j]:= 0) $ j=i+1..n;
       else (A[i,j]:= float(0)) $ j=i+1..n;
       end_if;
     end_for;
  end_if;
  end_case; // or useHardwareFloats = FALSE

  //-------------------------
  //------- return ----------
  //-------------------------
  if A = FAIL then
     return(FAIL)
  end_if:

  if A::dom = returnType and
     not (rowrangechanged or colrangechanged) then
     return(A);
  end_if;
  
  case returnType
     // mind the ordering: process DOM_{HF}ARRAY first to
     // avoid unnecessary loading of the DOMAIN package
  of DOM_ARRAY do
       if A::dom <> DOM_ARRAY then
          A:= array(originalrowrange, originalcolrange, [op(A)]);
       elif rowrangechanged or colrangechanged then
          A:= subsop(A, 0 = (2, originalrowrange, originalcolrange));
       end_if;
       return(A);
  of DOM_HFARRAY do
       if A::dom <> DOM_HFARRAY then
          if traperror((B:= 
                hfarray(originalrowrange, originalcolrange, [op(A)])
          ))<> 0 then
             if dowarn then
                warning("cannot return the Cholesky factor as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead");
             end_if:
             return(array(originalrowrange, originalcolrange, [op(A)]));
          else
             return(B):
          end_if:
       elif rowrangechanged or colrangechanged then
        //A:= subsop(A, 0 = (2, originalrowrange, originalcolrange));
          A:= hfarray(originalrowrange, originalcolrange, [op(A)]);
       end_if;
       return(A);
  of Dom::Matrix() do
       return(returnType(A));
  of Dom::DenseMatrix() do
       return(returnType::create(A));
  otherwise
       error("unexpected return type"):
  end_case;

end_proc:
