
//------   file  factorLU.mu  ---------------------
/*  Help file:

 factorLU - LU factorization of a matrix

 Call:

 numeric::factorLU( A 
                    <, Symbolic>  
                    <, hard_soft> 
                    <, ReturnType = typ 
                    <, NoWarning>
                   )
                                                                   
 Parameter:

  A         --  an m x n matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
  hard_soft -- either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'
               If DIGITS <=15, the default is HardwareFloats
               If DIGITS  >15, the default is SoftwareFloats

  typ       -- either DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix() or Dom::Matrix()

Synopsis:

  The LU factorization of a real/complex matrix is PA=LU, where
  L is lower triangular, normalized to 1 along the diagonal,
  and R is upper triangular.  P is a permutation matrix 
  corresponding to row exchanges.

  numeric::factorLU(A) returns the list [L,U,p], where L and R are
  of domtype DOM_ARRAY. The list of integers p=[ p[1],p[2], .. ]
  represents the row exchanges in pivoting steps, i.e., B = PA = LU, 
  where B[i,j] = A[p[i],j]. Multiplication y=Px of the permutation
  matrix P with a vector x is realized by y[i]:=x[p[i]].
                                                                  
  Non-float numerical entries are allowed in the matrix, they
  will be evaluated to floats in the course of the computation.
  Non-numeric symbolic entries will lead to an error, unless
  the option Symbolic is specified.

  The option Symbolic prevents conversion of the matrix entries
  to floats. No row exchanges will be performed unless necessary.

  Examples: 

>> A:=array(1..3,1..3,[[1,2,3],[2,4,6],[4,8,9]]):
>> ([L,U,p]):=numeric::factorLU(A);

          -- +-            -+  +-                -+            --
          |  |    1,  0, 0  |  |  4.0, 8.0,  9.0  |             |
          |  |              |  |                  |             |
          |  |   0.5, 1, 0  |, |   0,   0,   1.5  |, [3, 2, 1]  |
          |  |              |  |                  |             |
          |  |  0.25, 0, 1  |  |   0,   0,  0.75  |             |
          -- +-            -+  +-                -+            --

>> A:= array(1..3,1..2,[[1,2],[2,3],[3,4]]):
>> ([L,U,p]):=numeric::factorLU(A);

 -- +-                      -+  +-                   -+            --
 |  |        1,       0,  0  |  |  3.0,      4.0      |             |
 |  |                        |  |                     |             |
 |  |  0.3333333333,  1,  0  |, |   0,  0.6666666666  |, [3, 1, 2]  |
 |  |                        |  |                     |             |
 |  |  0.6666666666, 0.5, 1  |  |   0,        0       |             |
 -- +-                      -+  +-                   -+            --

See also:  linalg::factorLU, numeric::factorQR, numeric::factorCholesky
                                                                  */
//-----------------------------------------------------------------
numeric::factorLU:=proc(A)
local useHardwareFloats, HardwareFloatsRequested, 
      returnType, dowarn, specialwarning,
      m, n, method, B, i, j, k, tmp, pivot, pivindex, rownorm, L, p,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
begin

   if args(0) < 1 then error("expecting at least one argument") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  if domtype(args(1)) = DOM_ARRAY or
     domtype(args(1)) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [m, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [m, n]:= A::dom::matdim(A):
     originalrowrange:= 1..m:
     originalcolrange:= 1..n:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array or a matrix of category 'Cat::Matrix'"):
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  dowarn:= TRUE;
  specialwarning:= FAIL;

  for i from 2 to args(0) do
      case args(i)
      of Symbolic do method:= 1:
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Hard do
      of HardwareFloats do
                     if method = 1 then
                        specialwarning :=
                                "ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'":
                        useHardwareFloats:= FALSE:
                        HardwareFloatsRequested:= FALSE;
                     else
                        useHardwareFloats:= TRUE:
                        HardwareFloatsRequested:= TRUE;
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("factorLU") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "using hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
        if traperror((B:= hfarray(1..m, 1..n, [op(A)]))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
        end_if;
     else
        B:= A;
     end_if;

     if traperror((B:= hfa::factorLU(B))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             if domtype(B) = DOM_STRING then
                error(B);
             end_if:
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
     end_if;

     if B = FAIL then
        return(FAIL);
     end_if:

     // proceed to the end of this routine. There, the
     // matrix is converted to the requested return type.
     // Further, rowrange and colrange are adapted if necessary.
     [L, A, p]:= B:

  end_case; // of useHardwareFloats = TRUE

  //---------------------------------
  //----- end of HardwareFloats -----
  //---------------------------------

  case useHardwareFloats
  of FALSE do
     //---------------------------------
     //---- use SoftwareFloats ---------
     //---------------------------------

     userinfo(1, "using software floats"):

     // Do use domtype(A) <> DOM_ARRAY here to avoid
     // unnecessary loading of the DOMAINS package
     if domtype(A) <> DOM_ARRAY and
        domtype(A) <> DOM_HFARRAY and
        A::dom::hasProp(Cat::Matrix)=TRUE then
        // convert to domtype DOM_ARRAY for speed
        A:= expr(A);
     end_if;
     if domtype(A) <> DOM_ARRAY and
        domtype(A) <> DOM_HFARRAY then
        error("the matrix must be an array, an hfarray, or of category 'Cat::Matrix'");
     end_if:

     if op(A,[0,1])<>2 then error("the argument is not a matrix");end_if;

     if method=2 and A::dom <> DOM_HFARRAY then
        B:=map(A,float):
        if {op(map(B,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
           if dowarn then
              warning("symbolic entry found, switching to option Symbolic");
           end_if;
           method:=1;
        else A:=B;
        end_if:
     end_if;

     if A::dom = DOM_HFARRAY then
        A:= array(op(A, [0, 2]), op(A, [0, 3]), [op(A)]);
     end_if:
 
     p:=[i $ i=1..m]: // initialize list to store row permutations 

     if method=1 then
        userinfo(1,"symbolic mode, avoiding pivoting if possible ");
     end_if;
     if method=2 then            
        userinfo(1,"using column pivoting");
     end_if;

     for j from 1 to min(m-1,n) do
       if method=1 then
          for pivindex from j to m do
             if not iszero(A[pivindex,j]) then break; end_if;
          end_for;
          if pivindex > m then next end_if; // no nontrivial pivot element 
       end_if;
       if method=2 then
          pivot:=0;
          pivindex:=j;
          for i from j to m do 
              if iszero(A[i,j]) then next end_if;
              if j=n
              then if (tmp:= specfunc::abs(A[i,j]))>pivot
                      then pivot:= tmp; pivindex:= i;
                   end_if;
              else rownorm:=max(specfunc::abs(A[i,k]) $ k=j+1..n );
                   if not iszero(rownorm)
                   then specfunc::abs(A[i,j])/rownorm;
                        if pivot<last(1) then pivot:=last(1); pivindex:=i; end_if;
                   else pivindex:= i; break;
                   end_if;
              end_if;
          end_for;
       end_if;
  
       if pivindex<>j then
         (tmp:=A[j,k];
          A[j,k]:=A[pivindex,k];
          A[pivindex,k]:=tmp;) $ k=1..n;
         tmp:=p[j];
         p[j]:=p[pivindex];
         p[pivindex]:=tmp;
       end_if;
       if not iszero(A[j,j]) then
          for i from j+1 to m do
              A[i,j]:=A[i,j]/A[j,j]:
              if not iszero(A[i,j]) then
                (A[i,k]:=A[i,k]-A[i,j]*A[j,k]) $ k=j+1..n;
              end_if;
          end_for;
       end_if;
     end_for;
  
     /* LU data are stored in lower and upper part of A. Extract L and U: */
     if method = 1 then
          L:=array(1..m,1..m,[[0$m]$m]); 
          (L[i,i]:= 1) $ i=1..m;
     else L:=array(1..m,1..m,[[float(0)$m]$m]); 
          (L[i,i]:= float(1)) $ i=1..m;
     end_if;
     
     for j from 1 to min(m-1, n) do
       (L[i,j]:= A[i,j]) $ i=j+1..m;
     end_for:
     for j from 1 to n do
        if method = 1 then
             (A[i,j]:= 0) $ i=j+1..m 
        else (A[i,j]:= float(0)) $ i=j+1..m 
        end_if;
     end_for:

   end_case; // of useHardwareFloats = FALSE
   //-------------------------
   //------- return ----------
   //-------------------------
   if L = FAIL or A = FAIL or p = FAIL then
      return(FAIL);
   end_if; 

   if L::dom = returnType and
      A::dom = returnType and
      not (rowrangechanged or colrangechanged) then
      return([L, A, p]);
   end_if;

   if rowrangechanged then
       p:= map(p, i -> i + op(originalrowrange, 1) - 1);
   end_if:

   case returnType
     // mind the ordering: process DOM_{HF}ARRAY first to
     // avoid unnecessary loading of the DOMAIN package
   of DOM_ARRAY do
       if L::dom <> DOM_ARRAY then
          L:= array(originalrowrange, originalrowrange, [op(L)]);
       elif rowrangechanged or colrangechanged then
          L:= subsop(L, 0 = (2, originalrowrange, originalrowrange));
       end_if;
       if A::dom <> DOM_ARRAY then
          A:= array(originalrowrange, originalcolrange, [op(A)]);
       elif rowrangechanged or colrangechanged then
          A:= subsop(A, 0 = (2, originalrowrange, originalcolrange));
       end_if;
       return([L, A, p]);
   of DOM_HFARRAY do
       if L::dom <> DOM_HFARRAY then
          if traperror((B:=
                hfarray(originalrowrange, originalrowrange, [op(L)])
          ))<> 0 then
             if dowarn then
                warning("cannot return the L factor as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead");
             end_if:
             L:= array(originalrowrange, originalrowrange, [op(L)]);
          else
             L:= B:
          end_if:
       elif rowrangechanged or colrangechanged then
          L:= hfarray(originalrowrange, originalrowrange, [op(L)]);
       end_if;
       if A::dom <> DOM_HFARRAY then
          if traperror((B:=
                hfarray(originalrowrange, originalcolrange, [op(A)])
          ))<> 0 then
             if dowarn then
                warning("cannot return the U factor as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead");
             end_if:
             A:= array(originalrowrange, originalcolrange, [op(A)]);
          else
             A:= B:
          end_if:
       elif rowrangechanged or colrangechanged then
          A:= hfarray(originalrowrange, originalcolrange, [op(A)]);
       end_if;
       return([L, A, p]);
   of Dom::Matrix() do
        return([returnType(L), returnType(A), p]);
   of Dom::DenseMatrix() do
        return([returnType::create(L), returnType::create(A), p]);
   otherwise
        error("unexpected return type"):
   end_case;
   return(FAIL);
end_proc:
