
//------   file  factorQR.mu  19.9.97         ---------------------
//  auf nichtquadratische Matrizen erweitert 8.2.99 W. Oevel  
//  option QDet eingefuegt: 9.7.99 W. Oevel 
//  Scilab eingebaut, QDet wieder entfernt: 24.10.01, W. Oevel
/*  Help file:

 factorQR - QR factorization of a matrix

 Call:

 numeric::factorQR( A 
                    <, Symbolic>
                    <, hard_soft>
                    <, ReturnType = typ>
                    <, NoWarning>
                    // <, QDet>  wieder entfernt
                   )
                                                                   
 Parameter:

  A  --  a matrix (DOM_ARRAY, DOM_HFARRAY or category Cat::Matrix)
  hard_soft -- either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'
               If DIGITS <=15, the default is HardwareFloats
               If DIGITS  >15, the default is SoftwareFloats

  typ       -- either DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix() or Dom::Matrix()
  "dont_use_normal" -- undocumented option to suppress the use of normal
                                                                   
Synopsis:

  The QR factorization of a real/complex m x n matrix is A=QR, where
  Q is a unitary/orthogonal m x m and R is an upper triangular 
  m x n matrix.

  numeric::factorQR(A); returns the list [Q,R] with Q and R of
  domtype DOM_ARRAY.
                                                                  
  Non-float numerical entries are allowed in the matrix, they
  will be evaluated to floats in the course of the computation.
  Non-numeric symbolic entries will lead to a warning. Householder
  transformations are used to compute the factorization. The
  R factor is normalized such that its diagonal entries are
  real and nonnegative.

  The option Symbolic prevents conversion of the matrix entries
  to floats. In this case symbolic entries are allowed, Gram-Schmidt
  orthonormalization is used to compute the factorization.

-----    New (9.7.99)  -----
  With the option QDet, the output changes to [Q,R,d] with d = det(Q).
  Presently this option is only available for the numerical
  factorization (i.e., there is no effect when using option Symbolic).

  This option is not documented (for internal use only)
-----------------------

  For invertible n x n matrices A the result without option Symbolic
  is the float version of the result with option Symbolic.
  For non-invertible or non-square A the factorization is not
  unique and option Symbolic may return a different result
  than the numerical factorization without option Symbolic.

  Examples: 

>> A:= matrix([ [1,1,I] , [1,exp(1),I] , [1,2,I] ]):

>> ([Q,R]):=numeric::factorQR(A);

    -- +-                                              -+
    |  | 0.5773502691, -0.7424322034,  - 0.339795659  I |
    |  |                                                |
    |  | 0.5773502691,  0.6654877745,  - 0.4730673192 I |,
    |  |                                                |
    |  | 0.5773502691,  0.07694442889,   0.8128629783 I |
    -- +-                                              -+
           +-                                           -+ --
           | 1.732050807, 3.301451553, 1.732050807 I     |  |
           |                                             |  |
           |      0,      1.220439979, 1.394208422e-19 I |  |
           |                                             |  |
           |      0,           0,      6.38099974e-20    |  |
           +-                                           -+ --
   
>> Q*R;
                       +-                       -+
                       | 1.0,     1.0,     1.0 I |
                       |                         |
                       | 1.0, 2.718281828, 1.0 I |
                       |                         |
                       | 1.0,     2.0,     1.0 I |
                       +-                       -+


>> A:= matrix([ [1,1,I] , [1,1,I] , [1,2,I] ]):

>> ([Q,R]):=numeric::factorQR(A,Symbolic);

   -- +-                         -+                              --
   |  |  1/2     1/2  1/2    1/2  |                               |
   |  | 3       2    3      2     |  +-                       -+  |
   |  | ----, - ---------,  ----  |  |            1/2          |  |
   |  |  3          6        2    |  |  1/2    4 3         1/2 |  |
   |  |                           |  | 3   ,   ------,  I 3    |  |
   |  |  1/2     1/2  1/2     1/2 |  |           3             |  |
   |  | 3       2    3       2    |  |                         |  |
   |  | ----, - ---------, - ---- |, |        1/2  1/2         |  |
   |  |  3          6         2   |  |       2    3            |  |
   |  |                           |  |   0,  ---------,    0   |  |
   |  |  1/2    1/2  1/2          |  |           3             |  |
   |  | 3      2    3             |  |                         |  |
   |  | ----,  ---------,     0   |  |   0,      0,        0   |  |
   |  |  3         3              |  +-                       -+  |
   -- +-                         -+                              --

>> Q*R;
                               +-         -+
                               |  1, 1, I  |
                               |           |
                               |  1, 1, I  |
                               |           |
                               |  1, 2, I  |
                               +-         -+

>> ([Q,R,d]):= numeric::factorQR(A, QDet);

-- +-                                                  -+
|  |  0.5773502691, -0.4082482904,   - 0.7071067811 I   |
|  |                                                    |
|  |  0.5773502691, -0.4082482904,    0.7071067811 I    |,
|  |                                                    |
|  |  0.5773502691,  0.8164965809, - 5.421010862e-20 I  |
-- +-                                                  -+

   +-                                              -+      --
   |  1.732050807,  2.309401076,   1.732050807 I    |       |
   |                                                |       |
   |       0,      0.8164965809, 7.666467083e-20 I  |, - I  |
   |                                                |       |
   |       0,            0,        1.32787105e-19   |       |
   +-                                              -+      --



See also:  linalg::factorQR, numeric::factorLU, numeric::factorCholesky
                                                                  */
//-----------------------------------------------------------------
numeric::factorQR:=proc(A)
local useHardwareFloats, HardwareFloatsRequested,
      returnType, dowarn, specialwarning,
      n, m, mn, method, use_normal, normal2, B, rreal, 
      dont_use_normal_requested,
      i, j, k, l, d, h, s, ss, q, r, Q, R,
      // showQdet, Qdet,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
begin

  if args(0) < 1 then error("expecting at least one argument") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  if domtype(args(1)) = DOM_ARRAY or
     domtype(args(1)) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     originalrowrange:= 1..n:
     originalcolrange:= 1..m:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array or a matrix of category 'Cat::Matrix'"):
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  /*
  showQdet:= FALSE;
  */
  dowarn:= TRUE;
  specialwarning:= FAIL;
  dont_use_normal_requested:= FALSE;

  for i from 2 to args(0) do
      case args(i)
      of Symbolic do method:= 1:
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Hard do
      of HardwareFloats do
                     if method = 1 then
                        specialwarning:=
                               "ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'":
                        useHardwareFloats:= FALSE:
                        HardwareFloatsRequested:= FALSE;
                     else
                        useHardwareFloats:= TRUE:
                        HardwareFloatsRequested:= TRUE;
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      /*
      of QDet do     showQdet:= TRUE; 
                     break;
      */
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      of "dont_use_normal" do
                     dont_use_normal_requested:= TRUE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------

  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("factorQR") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "using hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
        if traperror((B:= hfarray(1..n, 1..m, [op(A)]))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
        end_if;
     else
        B:= A;
     end_if;

     if traperror((B:= hfa::factorQR(B))) <> 0 or
           has(B, RD_NAN) or
           has(B, RD_INF) or
           has(B, RD_NINF) then
             if domtype(B) = DOM_STRING then
                error(B);
             end_if:
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             if HardwareFloatsRequested and dowarn then
                warning("HardwareFloats failed. Using SoftwareFloats instead");
             end_if:
             break;
     end_if;

     if B = FAIL then
        return(FAIL);
     end_if:

     // proceed to the end of this routine. There, the
     // matrix is converted to the requested return type.
     // Further, rowrange and colrange are adapted if necessary.
     [Q, R]:= B:

  end_case; // of useHardwareFloats = TRUE
  //---------------------------------
  //----- end of HardwareFloats -----
  //---------------------------------

  case useHardwareFloats
  of FALSE do
     //---------------------------------
     //---- use SoftwareFloats ---------
     //---------------------------------
   
      userinfo(1, "using software floats"):
   
     // Do use domtype(A) <> DOM_ARRAY here to avoid
     // unnecessary loading of the DOMAINS package
     if domtype(A) <> DOM_ARRAY and
        domtype(A) <> DOM_HFARRAY and
        A::dom::hasProp(Cat::Matrix)=TRUE then
        // convert to domtype DOM_ARRAY for speed
        A:= expr(A);
     end_if;
     if domtype(A) <> DOM_ARRAY and
        domtype(A) <> DOM_HFARRAY then
        error("the matrix must be an array, an hfarray, or of category 'Cat::Matrix'");
     end_if:
   
     if op(A,[0,1])<>2 then error("the argument is not a matrix");end_if;
   
     use_normal:= FALSE;

     if method=1 and A::dom <> DOM_HFARRAY then
      if (indets(A)<>{}) 
         or has({op(map(A,domtype))},DOM_NIL)
         or has({op(map(A,domtype))},DOM_EXPR)
        then 
           if not dont_use_normal_requested then
              use_normal:=TRUE 
           end_if;
      end_if;
     end_if;

     if method=2 and A::dom <> DOM_HFARRAY then            
        B:=map(A,float):
        if {op(map(B,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {} then 
            if dowarn then
                 warning("symbolic entry found, switching to option Symbolic");
            end_if;
            method:=1;
            if not dont_use_normal_requested then
              use_normal:= TRUE;
            end_if;
        else A:=B;
        end_if:
     end_if;

     if A::dom = DOM_HFARRAY then
        A:= array(op(A, [0, 2]), op(A, [0, 3]), [op(A)]);
     end_if:

     if {op(map(A,(iszero@Im)))} = {TRUE}     // is matrix real ? 
       then rreal:=TRUE else rreal:=FALSE
     end_if:

if method=1 then
   normal2:= x -> normal(x, Expand = FALSE);
   userinfo(1,"using Gram-Schmidt orthonormalization");
   R:=array(1..n,1..m,[[0$m]$n]); // initialize upper factor 
   if m>=n                   // initialize orthogonal factor 
      then Q:=array(1..n,1..n,[[A[i,j] $j=1..n]$i=1..n]);
      else Q:=array(1..n,1..n,[[0$n]$n]):
           ((Q[i,j]:= A[i,j];) $ j=1..m) $ i=1..n;
   end_if;
   for j from 1 to n do
      if j<=m then
       (Q[i,j]:= Q[i,j]-_plus(Q[i,k]*R[k,j]/r[k] $ k=1..j-1 );) $ i=1..n;
       if use_normal then (Q[i,j]:= normal2(Q[i,j]);) $ i=1..n; end_if;
      end_if;
      
      r[j]:=_plus(conjugate(Q[i,j])*Q[i,j] $ i=1..n );
      if use_normal then r[j]:= normal2(r[j]); end_if;
      if iszero(r[j]) then       // insert an arbitrary column. Try the 
         for k from 1 to n do    // standard columns q=(0,..,1,..,0)^T  
            q:=[0$n]; q[k]:=1;   // until q is independent of the first 
            for i from 1 to n do // j-1 orthonormalized columns of Q    
               q[i]:=q[i]-_plus(Q[i,l]*conjugate(Q[k,l])/r[l] $ l=1..j-1);
               if use_normal then q[i]:= normal2(q[i]); end_if;
            end_for;
            r[j]:=_plus( conjugate(q[i])*q[i] $ i=1..n );
            if use_normal then r[j]:= normal2(r[j]); end_if;
            if not iszero(r[j]) then
               if domtype(r[j]) = DOM_FLOAT or 
                  (domtype(r[j]) = DOM_COMPLEX and op(r[j],1) = DOM_FLOAT)
                  then
                    // If the matrix contains floats, we should
                    // double check that this column can be accepted.
                    if specfunc::abs(r[j]) < max(1/(m + n)/10^2, 10^(-DIGITS)) then
                       next; // try the next value of k
                    end_if;
                end_if;
                // in the symbolic case any nonzero r[j] is OK
                (Q[i,j]:=q[i]) $ i=1..n; break;
             end_if;
         end_for;
         if iszero(r[j]) then error("could not orthonormalize the columns") end_if;
         if j<=m then R[j,j]:=0; end_if;
      else 
         if j<=m then R[j,j]:= sqrt(r[j]) end_if;
      end_if;
      (R[j,k]:=_plus(conjugate(Q[i,j])*A[i,k] $ i=1..n );) $ k=j+1..m;
      if use_normal then
        (R[j,k]:=normal2(R[j,k]);) $ k=j+1..m;
      end_if;
   end_for;
   r:=[sqrt(r[i]) $ i=1..n];
   for j from 1 to n do
       (Q[i,j]:=Q[i,j]/r[j]) $ i=1..n;
       (R[j,i]:=R[j,i]/r[j]) $ i=j+1..m;
   end_for;
end_if;

if method=2 then            
   userinfo(1,"using Householder transformations");
   mn:= min(m,n);

   d:=[float(0)$n]:   // initialize list to store diagonal 
   h:=[float(0)$mn]:  // initialize list to store factors  

   for j from 1 to mn do
        // Householder step from the left to obtain upper   
        // triangular form. Householder data are stored in  
        // lower triangular part of A.                      
        s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..n);
        if iszero(s) then d[j]:=A[j,j];
        else ss:=s+specfunc::abs(A[j,j])^2;
             s:=specfunc::sqrt(ss):
             h[j] := 1 / (ss + specfunc::abs(A[j,j]) * s);
             if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
             A[j,j]:=s+A[j,j]; d[j]:=-s;
             for i from j+1 to m do
                 if rreal then  s:=_plus(A[k,j] *A[k,i]$ k=j..n);
                 else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..n);
                 end_if;
                 s:=s*h[j];
                 (A[k,i]:=A[k,i]-s*A[k,j]) $ k=j..n ;
             end_for;
        end_if;
   end_for;

   // construct Q from the Householder data stored in the lower part of A
   Q:=array(1..n,1..n,[[float(0)$n]$n]); 
   (Q[i,i]:=float(1)) $ i=1..n;
   for j from mn downto 1 do
     if not iszero(h[j]) then
        for i from j+1 to n do
           if rreal then  s:=_plus(A[k,j]*Q[k,i] $ k=j..n) ;
           else s:=_plus(conjugate(A[k,j])*Q[k,i] $ k=j..n) ;
           end_if;
           s:=s*h[j];
           (Q[k,i]:=Q[k,i]-s*A[k,j]) $ k=j..n ;
        end_for;
       (Q[k,j]:=A[k,j]/d[j]) $ k=j..n;
       Q[j,j]:=Q[j,j]+1;
     end_if;
   end_for;
   /*
    if showQdet then 
       Qdet:=(-1)^(mn-nops(select(h,iszero))); 
    end_if;
   */

   // construct R from the upper part of A and diagonal d 
   for j from 1 to mn do
     (A[i,j]:= float(0)) $ i=j+1..n;
      A[j,j]:=specfunc::abs(d[j]);
      if not iszero(A[j,j]) then     
         s:=sign(d[j]);                 // normalize diagonal of 
         (A[j,i]:=A[j,i]/s) $ i=j+1..m; // R to be real positive 
         (Q[i,j]:=Q[i,j]*s) $ i=1..n;
         /*
         if s<>1 and showQdet then 
            Qdet:= s*Qdet; 
         end_if;
         */
      end_if;
   end_for;

/*
   if showQdet
      then return([Q, A, Qdet]);
      else return([Q, A]):
   end_if;
*/

    R:= A:

end_if;

   end_case; // of useHardwareFloats = FALSE

   //-------------------------
   //------- return ----------
   //-------------------------
   if Q = FAIL or R = FAIL then
     return(FAIL)
   end_if:

   if R::dom = returnType and
      Q::dom = returnType and
      not (rowrangechanged or colrangechanged) then
      return([Q, R]);
   end_if;

   case returnType
     // mind the ordering: process DOM_{HF}ARRAY first to
     // avoid unnecessary loading of the DOMAIN package
   of DOM_ARRAY do
       if Q::dom <> DOM_ARRAY then
          Q:= array(originalrowrange, originalrowrange, [op(Q)]);
       elif rowrangechanged or colrangechanged then
          Q:= subsop(Q, 0 = (2, originalrowrange, originalrowrange));
       end_if;
       if R::dom <> DOM_ARRAY then
          R:= array(originalrowrange, originalcolrange, [op(R)]);
       elif rowrangechanged or colrangechanged then
          R:= subsop(R, 0 = (2, originalrowrange, originalcolrange));
       end_if;
       return([Q, R]);
   of DOM_HFARRAY do
       if Q::dom <> DOM_HFARRAY then
          if traperror((B:=
                hfarray(originalrowrange, originalrowrange, [op(Q)])
          ))<> 0 then
             if dowarn then
                warning("cannot return the Q factor as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead");
             end_if:
             Q:= array(originalrowrange, originalrowrange, [op(Q)]);
          else
             Q:= B:
          end_if:
       elif rowrangechanged or colrangechanged then
          Q:= hfarray(originalrowrange, originalrowrange, [op(Q)]);
       end_if;
       if R::dom <> DOM_HFARRAY then
          if traperror((B:=
                hfarray(originalrowrange, originalcolrange, [op(R)])
          ))<> 0 then
             if dowarn then
                warning("cannot return the R factor as an hfarray. ".
                        "Returning it as an array of type DOM_ARRAY instead");
             end_if:
             R:= array(originalrowrange, originalcolrange, [op(R)]);
          else
             R:= B:
          end_if:
       elif rowrangechanged or colrangechanged then
          R:= hfarray(originalrowrange, originalcolrange, [op(R)]);
       end_if;
       return([Q, R]);
   of Dom::Matrix() do
        return([returnType(Q), returnType(R)]);
   of Dom::DenseMatrix() do
        return([returnType::create(Q), returnType::create(R)]);
   otherwise
        error("unexpected return type"):
   end_case;

end_proc:
