/*
   numeric::product=product:float - 
     numerical evaluation of infinite product(f(k),k=a..b)

  Examples:

  >> product(1-4^-i, i = 1..infinity)

            infinity
            -----            i
             | |   (1 - (1/4) )
             | |
            i = 1

  >> float(%)

          0.6885375371

  >> numeric::product(i^2/(i^2 - 1), i = 2..infinity)
  
              2.0

  >> numeric::product(i[1], i[1] = 1..6)

             720.0

  >> numeric::product(exp(r), r in RootOf(x^2 - 4))
  
              1.0

*/


numeric::product := proc(f, x)
  local
    k,r,a,b,
    roots,j;
  save DIGITS;
begin
  if testargs() then
    if args(0) <> 2 then
      error("Expecting two arguments");
    end_if;
    if not contains({"_in", "_equal"}, type(x)) then
      error("Second argument: expecting a range specification ".
            "such as 'k=a..b' or 'x in RootOf(...)'")
    end_if;
  end_if;

  k := op(x,1);
  if not contains({"_index", DOM_IDENT}, type(k)) then
    error("The index must be an identifier or an indexed identifier")
  end_if;
  
  if numeric::indets(f) minus {k} <> {} then
    return(hold(numeric::product)(float(f), x))
  end_if;
  
  r := op(x,2);
  if type(r) = RootOf then
    // Second parameter is: k = RootOf(a,b)
    a := float(op(r,1));
    b := op(r,2);
    
    // Catch the case where the equation contains additional indeterminates
    if (indets(a, PolyExpr) minus {b}) <> {} then
      return(hold(numeric::product)(float(f), k = RootOf(a,b)))
    end_if;
    
    // Increase precision to offset error accumulation in _mult
    DIGITS := DIGITS + length(degree(a));

    if type(f) = "_power" and not has(op(f,2), k) then
      // The special case where f is a monomial is already handled in prod_rootof.
      if traperror((roots := numeric::polyroots(a))) <> 0 then
        return(hold(numeric::product)(float(f), k = RootOf(a,b)))
      end_if;
      return(numeric::complexRound(_mult(float(subs(op(f,1), k = j, EvalChanges)) $ j in roots))^op(f,2));
    else
      // Attempt to calculate roots of a polynomial
      // numeric::polyroots returns a list with roots occuring according to
      // their multiplicity. An error occurs when
      //  - a is not a polynomial (e.g. it contains something like f(x)) or
      //  - a is the zero polynomial
      if traperror((r := numeric::polyroots(a))) <> 0 then
        return(hold(numeric::product)(float(f), k = RootOf(a,b)))
      end_if;
      return(numeric::complexRound(_mult(float(subs(f,k=j,EvalChanges)) $ j in r)));
    end_if;
  elif domtype(r) = DOM_SET then
    //r := float(r);
    a := _mult(float(subs(f, k = j, EvalChanges)) $ j in r);
    return(numeric::complexRound(a))
  elif type(r) = "_range" then
    a := op(r,1);
    b := op(r,2);
    
    if map({a,b}, domtype) minus {DOM_INT, domtype(infinity)}<>{} then
      return(hold(numeric::product)(float(f), k = a..b))
    end_if;
    
    if a > b then
      error("Left bound is larger than right bound")
    end_if;

    if domtype(a) = DOM_INT then
      if domtype(b) = DOM_INT and
         b - a < 10^5 then
        r := _mult(float(subs(f, k = j, EvalChanges)) $ j = a..b);
        return(numeric::complexRound(r))
      elif b = infinity then
        r := numeric::product_infinite(subs(f, k = k+a-1), k);
        if r<>FAIL then
          return(r)
        end_if;
      end_if;
    elif a = -infinity then
      if domtype(b) = DOM_INT then
        r := numeric::product_infinite(subs(f, k = -k+b+1), k);
        if r<>FAIL then
          return(r)
        end_if;
      elif b = infinity then
        r := float(subs(f, k=0, EvalChanges))*numeric::product_infinite(f * subs(f, k = -k), k);
        if r<>FAIL then
          return(r)
        end_if;
      end_if;
    end_if;
    
    // Couldn't evaluate the product / the product didn't converge.
    return(hold(numeric::product)(float(f), k = a..b))
  end_if;
  
  error("Second argument: expecting a range specification ".
        "such as 'k=a..b', 'k in {...}' or 'x in RootOf(...)'");
end_proc:

product::float:= numeric::product:


/*

Float evaluation of product f(k), k = 1..infinity

Analyze the asymptotic behaviour of the factors. If the limit appears
to be 1.0, calculate the product using numeric::sum.

*/
numeric::product_infinite :=
proc(f,k)
  local
    S,P,
    conv,
    a,b,j,N,soc,
    g;
begin
  conv := UNKNOWN;
  
  if type(k) = "_index" then
    g := genident();
    f := subs(f, k=g);
    k := g;
  end_if;
  
/* ------------------------------------------
  // Try to find the limit of the factors symbolically, but don't waste
  // too much time on it.
  if not hastype(f, DOM_FLOAT, {DOM_COMPLEX}) then
    if traperror((L := limit(f, k = infinity)), 10) = 0 then
      if has(L,undefined) then
        return(FAIL);
      elif is(L = 1) = TRUE then
        conv := 1;
      elif is(abs(L) < 1) = TRUE then
        return(0.0);
      elif is(abs(L) > 1) = TRUE then
        return(FAIL); // doesn't converge
      end_if;
    end_if;
  end_if;
------------------------------------------ */
  
  // If limit() wasn't successful, use a heuristic
  if conv = UNKNOWN then
    userinfo(3, "Use heuristic to test convergence of factors");

    // Heuristic: If the factors get close enough to 1.0 at some point,
    //    and the distance to 1.0 is decreasing, we assume that the
    //    factors converge to 1.0 (if they don't, numeric::sum later on
    //    will fail).
    //
    // Look at a number of consecutive factors. This avoids an incorrect
    // decision when the factors are oscillating wildly.
    a := array(1..5);
    soc := 2*DIGITS;
    for j from 1 to 5 do
      for N from 1 to 5 do
        a[N] := abs(float(1-subs(float(f),k=soc+N, EvalChanges)));
        if not domtype(a[N]) in {DOM_FLOAT,DOM_COMPLEX} then
          return(FAIL)
        end_if;
      end_for;
      
      b := [abs(float(1-subs(float(f),k=2*soc+N,EvalChanges))) $ N = 1..3];

      // The values a and b are subject to severe cancellation.
      // If the b values are too small, they cannot be trusted 
      // and are likely to be pure roundoff. Hence, assume
      // convergence if abs(max(op(b))) < 10^(-DIGITS).
      
      if 0.5 > max(op(a)) then
        if min(op(a)) > max(op(b)) or 
           iszero(max(op(b))) or
           abs(max(op(b))) < 10^(-DIGITS) then
          break;
        end_if;
      end_if;
      
      soc := soc * 10;
    end_for:
    
    if j > 5 then
      // If the factors are small and decreasing in absolute value,
      // we simply assume that the product converges to zero.
      //
      // Note that there may be products where the factors are strictly increasing
      // but limited by 1-epsilon. These products are also zero, but we can't
      // test this numerically.
      a := [abs(float(subs(float(f),k=soc+N,EvalChanges))) $ N = 1..3];
      b := [abs(float(subs(float(f),k=2*soc+N,EvalChanges))) $ N = 1..3];
      if 1 - 10^(-DIGITS) > max(op(a)) and min(op(a)) >= max(op(b)) then
        return(0.0);
      end_if;
      
      return(FAIL) // don't know whether this converges
    end_if;
  else
    // We know from limit() that the factors converge to 1.0.
    // There is a danger of missing an initial 0.0 factor, however, so we
    // still split off a number of factors determined by trial and error.
    soc := 2*DIGITS;
  end_if;
  
  // Heuristic: split of the first factors, in the hope that it
  //  a) improves convergence (this is actually not necessarily the case for Levin acceleration!) and
  //  b) finds factors that are exactly zero
  P := _mult(float(subs(f, k=j, EvalChanges)) $ j = 1..soc);
  f := subs(f, k = k+soc);
  
  if not testtype(P, Type::Numeric) then
    return(FAIL)
  elif iszero(P) then
    return(0.0)
  end_if;
  
  // The limit of factors appears to be 1.0
  //
  // !!! Workaround (PARI problem):
  //     If epsilon is a rational expression, float(1-epsilon) becomes stationary != 1
  //     as epsilon approaches zero. This means that float(ln(1-epsilon)) becomes
  //     stationary != 0.
  //     numeric::sum correctly notices that such a series doesn't converge.
  //     So: Convert epsilon to float, then add 1 again (float additions round as one
  //     expects), then take the ln.
  if traperror((
       S := numeric::sum(ln(1+hold(float)(f-1)), k=1..infinity);
     )) <> 0 then
      return(FAIL);
  end_if:
  if testtype(S, Type::Numeric) then
    return(P*exp(S));
  end_if;
  
  return(FAIL);
end_proc:

