/* Numerical solver for n nonlinear equations in n unknowns
  using Newton-Raphson/modified Newton-Raphson/Gradient method */

/* 
   Call:          numeric::fsolve(eqs, var, Option);
   Parameters:  
       eqs = equation/expression (in one variable) or
             list or set of equations/expressions (if there
             are many equations). Expressions are interpreted
             as equations expr=0.
       vars= identifier or indexed identifier 
             or  list or set of (indexed) identifiers.
             Instead of identifiers also equations of the form
                   variable = numerical value       (starting point)
                   variable = leftbound..rightbound (search range)
             may be used.

       The number of equations must not be > number of unknowns.

   Options:
      RestrictedSearch   - look for solutions in search range only
      UnrestrictedSearch - look for solutions outside search range, too
      MultiSolutions     - allow sequences of solutions to be returned.
                           Only useful with search ranges and
                           RestrictedSearch.
                           Without MultiSolutions, this may return FAIL,
                           if no solution inside the search range is found.
                           With MultiSolutions, all found solutions
                           are returned
      Random             - without this option numeric::fsolve produces
                           deterministic results. With this option several
                           calls with the same input may lead to different
                           solutions
   Examples:

   numeric::fsolve( x=tan(x), x);
   numeric::fsolve( x=tan(x), x=1);
   numeric::fsolve( x=tan(x), x=1..2);
   numeric::fsolve([x=tan(x)], x=1..2);
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x,y});
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x=1,y=-0.2});
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x=1,y=-1..1});
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x=0..2,y=-1..1});
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x=-1..0,y=-1..0},
                   RestrictedSearch);
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x=-1..0,y=-1..0},
                   RestrictedSearch, Random);
   DIGITS:= 10:
   rough_approximation:= numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, {x,y});
   DIGITS:= 1000:
   numeric::fsolve({x^2+y=sin(x), x^2=y^3+1}, rough_approximation);
   DIGITS:= 10:
*/
   

numeric::fsolve:= proc(equations, searchintervals)
local macheps, machepssqr, m, n, f, ff, F, vars, Jacobian, scale,
      i, j, x, b,
      leftSearchBounds, rightSearchBounds, 
      leftStartBounds, rightStartBounds, 
      complexSearch, firstRun, lastRun, RandomMode, 
      symbolicCheck, dummies, Normsqr, ScaledNormsqr, 
      rand, randomvector, Shift, NewtonStep, modifiedNewtonStep, 
      NewtonStrategy, Step, lastShift, Abbruch, new_x,
      checkrange, startingpoint, oldnorm, newnorm, totalSteps,
      insideSolutions, outsideSolutions, EliminateDuplicates, 
      SingleRun, Run, maxRun, SearchLengthTable, maxSearchLength, 
      StepsInThisRun, RestrictedMode, ShowAllSolutions, tmp,
      UnRestrictRequest, useHardwareFloats;
save DIGITS;
begin
  if args(0)<2 then error("not enough arguments"); end_if;

  /* --------------------------------------- */
  /* search arguments 3..args(0) for options */
  /* --------------------------------------- */
  //Step:= modifiedNewtonStep: /* default */
  //RestrictedMode:= FALSE;    /* default */
  RestrictedMode:= FALSE;      /* default, if no search range is given */
  UnRestrictRequest:= FALSE;   /* default, if UnrestrictedSearch is not given*/
  ShowAllSolutions:= FALSE;    /* default */
  RandomMode:= FALSE;          /* default */
  // do not use save SEED, because SEED **must** be
  // modified inside fsolve, if option Random is used

  (case args(i)
      //of Newton do           Step:= NewtonStep: break;
      of Random do             RandomMode:= TRUE: break;
      of RestrictedSearch do   RestrictedMode:= TRUE; break;
      of UnrestrictedSearch do UnRestrictRequest:= TRUE:
                               RestrictedMode:= FALSE; break;
      of MultiSolutions     do ShowAllSolutions:= TRUE; break;
      otherwise              error("unknown option");
   end_case;) $ i=3..args(0);
  if not RandomMode then
    save hold(SEED);
    SEED:= 1:  /* default: from the outside, fsolve is deterministic,
                  because the internal random always uses the same SEED */
  end_if:

  // ----  pre-processing of args(1)..args(2) -----
  if equations::dom::hasProp(Cat::Matrix)=TRUE then
     equations:= expr(equations):
  end_if:
  case domtype(equations)
  of DOM_SET do
  of DOM_ARRAY do
     equations:= [op(equations)]; 
  end_case;
  if domtype(equations) <> DOM_LIST then equations:= [ equations ]; end_if;
  // take care of DOM_POLY input:
  equations:= map(equations, expr);
  if domtype(searchintervals)=DOM_SET 
     then searchintervals:=[op(searchintervals)];
  end_if;
  if domtype(searchintervals)<>DOM_LIST
     then searchintervals:=[searchintervals];
  end_if;
  macheps:= float(10^(-DIGITS));
  machepssqr:= macheps^2: 
  DIGITS:= DIGITS+3;
  n:= nops(equations);
  m:= nops(searchintervals);
  if m < n then 
    error("cannot handle case 'more equations than unknowns'");
  end_if;

  // convert set/list of equations left=right into list of expressions left-right 
  f:= map([op(equations)], proc(eq) begin
                           if type(eq)="_equal"
                              then return(float(op(eq,1)-op(eq,2)))
                              else return(float(eq))
                           end_if;
                           end_proc);

  // fill up with trivial equations, if more unknowns than equations:
  if m > n then 
      f:= [op(f), float(0) $m-n]; 
      n:= m;
  end_if;

  // Automatische Generierung der Unbestimmten einbauen ??? 
  // vars:=indets(equations);
  // vars:=vars minus select(vars,testtype,Type::Constant);

  // the unknowns vars = [var1,var2,..] 
  vars:= [0$m];       // initialize container
  leftSearchBounds:= [0$m]; // initialize container
  rightSearchBounds:= [0$m];// initialize container
  leftStartBounds:= [0$m];  // initialize container
  rightStartBounds:= [0$m]; // initialize container
  (if type(searchintervals[i]) = "_equal" 
   then vars[i]:= op(searchintervals[i],1);
        if type(op(searchintervals[i],2))<> "_range" 
        then tmp:= float(op(searchintervals[i],2));
             if not has({DOM_FLOAT,DOM_COMPLEX}, domtype(tmp))
                then error("illegal starting point");
             end_if;
             leftStartBounds[i]:= tmp; 
             rightStartBounds[i]:= tmp;
             leftSearchBounds[i]:= -infinity; 
             rightSearchBounds[i]:= infinity;
        else leftSearchBounds[i] := float(op(searchintervals[i],[2,1]));
             rightSearchBounds[i]:= float(op(searchintervals[i],[2,2]));
             if {domtype(leftSearchBounds[i]), domtype(rightSearchBounds[i])}
                minus {DOM_FLOAT,DOM_COMPLEX, stdlib::Infinity} <> {} 
                then error("illegal search range"); 
             end_if;
             leftStartBounds[i]:= leftSearchBounds[i];
             rightStartBounds[i]:= rightSearchBounds[i];
             if not UnRestrictRequest then
               RestrictedMode := TRUE:
             end_if;
        end_if;
   else vars[i]:= searchintervals[i];
        leftSearchBounds[i] := -infinity;
        rightSearchBounds[i]:= infinity;
        leftStartBounds[i]:= leftSearchBounds[i];
        rightStartBounds[i]:= rightSearchBounds[i];
   end_if;
   if domtype(vars[i])<>DOM_IDENT and type(vars[i])<>"_index" 
      then error("illegal unknown") 
   end_if;
   ) $ i=1..m;

   //-----------------------------------------------------------------------
   // Since MuPAD 4.0, float(infinity) = RD_INF, float(-infinity) = RD_NINF.
   // The code below expects +/-infinity as boundaries, not RD_INF/RD_NINF:
   //-----------------------------------------------------------------------
   leftStartBounds:= subs(leftStartBounds, [RD_NINF = -infinity, RD_INF = infinity]):
   rightStartBounds:= subs(rightStartBounds, [RD_INF = infinity, RD_NINF = -infinity]):

   //-----------------------------------------------------------------------
   // Also for complex starting points, the search bounds
   // were set to -infinity, infinity. So query start bounds instead:
   complexSearch:= FALSE; // default
   if has(map(leftStartBounds, domtype), DOM_COMPLEX) or
      has(map(rightStartBounds, domtype), DOM_COMPLEX) 
      then complexSearch:= TRUE;
           RestrictedMode:= FALSE;
   end_if;

   if map({op(zip(rightStartBounds,leftStartBounds,_subtract))},iszero)={TRUE}
      then SingleRun:= TRUE;
           RestrictedMode:= FALSE;
      else SingleRun:= FALSE;
   end_if;

   if not complexSearch and
      has(zip(leftSearchBounds,rightSearchBounds,bool@_leequal), FALSE)
          then error("search range x=a..b with a>b found")
   end_if;

   // ----------------------------------------------------------
   // symbolic pre-processing: check, whether the symbolic 0 is
   // a solution (this is the preferred solution to be returned).
   // We may do this only if 0 is in the search interval(s) or
   // if there is no starting point <> 0:
   // ----------------------------------------------------------
   symbolicCheck:= TRUE;
   // check that there are no starting points <> 0
   if has(map({op(zip(rightStartBounds, leftStartBounds, _subtract))}, iszero), TRUE) then
      // a starting point is given for at least one of the variables is given
      if select(leftStartBounds, _not@iszero) <> [] then
           // at least one starting point is <> 0
           symbolicCheck:= FALSE: 
      end_if;
   end_if;
   // check that all search intervals contain 0
   if symbolicCheck and
     ( SingleRun or
       RandomMode or
       complexSearch or
       select(leftSearchBounds, _not@_leequal, 0) <> [] or
       select(rightSearchBounds, _less, 0) <> []
     )
      then
           symbolicCheck:= FALSE: 
   end_if;
   // do the symbolic check
   if symbolicCheck then
        // check whether the symbolic origin is a solution
        if traperror((
        b:=  map({op( subs(f, [vars[i] = 0 $ i = 1..nops(vars)], EvalChanges))}, iszero) = {TRUE} 
                    )) = 0 then
           if b then
              return([vars[i] = float(0) $ i = 1..nops(vars)])
           end_if;
        end_if;
        if nops(vars) = 1 then
            if  traperror((
            b:=  map({op( subs(f, [vars[i] = 1 $ i = 1..nops(vars)], EvalChanges))}, iszero) = {TRUE} 
                    )) = 0 then
               if b then
                 return([vars[i] = float(1) $ i = 1..nops(vars)])
              end_if;
            end_if;
        end_if;
        if nops(vars) = 1 then
            if  traperror((
            b:=  map({op( subs(f, [vars[i] = -1 $ i = 1..nops(vars)], EvalChanges))}, iszero) = {TRUE} 
                    )) = 0 then
               if b then
                 return([vars[i] = float(-1) $ i = 1..nops(vars)])
              end_if;
            end_if;
        end_if;
   end_if;

   // ----------------------------------------------------------
   // use numeric::realroot for search over a real finite range
   // Trap potential errors, if f is not real-valued etc.
   // ----------------------------------------------------------
   if n=1 and 
     (not complexSearch) and
     leftSearchBounds[1]<>rightSearchBounds[1] and
     not has([domtype(leftSearchBounds[1]),
              domtype(rightSearchBounds[1])], stdlib::Infinity)
     then
       userinfo(1, "calling numeric::realroot");
       if traperror( (
            x:= numeric::realroot(f[1],
                vars[1]= leftSearchBounds[1]..rightSearchBounds[1])
           )) = 0 
          and x <> FAIL then 
         return([vars[1]=x])
       end_if;
   end_if;

   //--------------------------------------------------------------
   // Enlarge the search range by a relative amount macheps,
   // to allow for round-off. Otherwise, solutions on the
   // border might not be found.
   // Do this *after* setting SingleRun:= TRUE/FALSE.
   //--------------------------------------------------------------

   if not complexSearch then
      for i from 1 to n do
          tmp:= leftSearchBounds[i]:
          if domtype(tmp) = DOM_FLOAT then // beware of +infinity
             leftSearchBounds[i]:= tmp-10*specfunc::abs(tmp)*macheps-machepssqr;
          end_if:
          tmp:= rightSearchBounds[i]:
          if domtype(tmp) = DOM_FLOAT then // beware of -infinity
             rightSearchBounds[i]:=tmp+10*specfunc::abs(tmp)*macheps+machepssqr;
          end_if:
      end_for:
   end_if;

  //---- compute the symbolic Jacobian of the functions f=[f[1],..,f[n]] -----
  Jacobian:= map(array(1..n,1..m,[[diff(f[i],vars[j])$j=1..m]$i=1..n]),float);

  // stop, if the equations contain objects that cannot be differentiated:
  if has(Jacobian, hold(diff)) or has(Jacobian, hold(D)) then
     error("cannot differentiate equation");
  end_if;

  //---- F(x) = numerical evaluation of f at numerical point x. ----
  F:=proc(x) begin 
     subs(f,zip(vars,x,
                (var,value)->(var=value))); 
     eval(float(%));  
  end_proc;

  // dirty: for MuPAD 1.5 we will need global identifiers to call linsolve. 
  // Need to be constructed only once 
  dummies:= [genident("dx") $ i=1..m]: 

  //---- Normsqr(x)=<x,x> ----
  Normsqr:=proc(f,scale) local i; begin 
     _plus(specfunc::abs(f[i])^2 $ i=1..nops(f));
  end_proc;

  // ScaledNormsqr(x)= <Scale^(-1)*f , Scale^(-1)*f> 
  ScaledNormsqr:=proc(f,scale) local i; begin 
     _plus(specfunc::abs(f[i]/scale[i])^2 $ i=1..nops(f));
  end_proc;

  /* Check whether numerical point x is in search range.  */
  checkrange:= proc(x) begin 
     if not RestrictedMode then return(TRUE) end_if;
     if has(map(x, domtype), DOM_COMPLEX) then return(FALSE) end_if;
     if has(zip(leftSearchBounds,x,bool@_leequal),FALSE) then return(FALSE) end_if;
     if has(zip(x,rightSearchBounds,bool@_leequal),FALSE) then return(FALSE) end_if;
     TRUE
  end_proc;


  //---------------------------------------------------------------
  // Shift(..) uses one of the linear solvers numeric::linsolve
  // or numeric::matlinsolve. Unfortunately, with Scilab,
  // numeric::linsolve is slower than with SoftwareFloats.
  // Further, if the dimension is small (say less than 7 x 7),
  // SoftwareFloats are faster because of the overhead of the
  // link to Scilab. Hence, use Scilab only for large systems
  // via numeric::matlinsolve. Otherwise, call numeric::linsolve,
  // because it allows sparse input.
 
  useHardwareFloats:=
              if DIGITS < 16 and m*n > 50 and
                 numeric::startHardwareFloats() then
                 TRUE;
              else 
                 FALSE
              end_if;
  //---------------------------------------------------------------

  /* --------------------------------------------------- */
  /* Compute the Newton vector dx = (f'(x))^(-1)*f(x).   */
  /* or the Gradient shift     dx = <g,g>/<f'g,f'g> g    */
  /*                           with g = f'(x)^T f(x)     */
  /* --------------------------------------------------- */

  Shift:=proc(x)  
  local NewtonShift, GradientShift, Newton_norm, GS_norm, JJ, ffTf, i, j, OK;
  begin //Use trapperror, because overflow may occur:
        if traperror((ff:= F(x)))<>0 then return(FAIL) end_if;
        if map({op(ff)},domtype) minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT} <>{}
          then error("equations seem to contain illegal symbolic objects");
        end_if;
        if iszero( _plus(specfunc::abs(ff[i]) $ i=1..m) ) then 
            // F(x)=0 -> x = solution. Notify other procedures by lastShift=0.
            lastShift:= float(0); return(x) 
        end_if;
        // substitute values into Jacobian of f 
        JJ:=map(subs(Jacobian,zip(vars,x,
                                  (var,value)->(var=value))),
                eval);
        /* balance the equations by scaling f[i] -> f[i]/scale[i]   */
        /* such that all rows in Jacobian at point x have same norm */
        (scale[i]:= _plus( specfunc::abs(JJ[i,j]) $ j=1..m); ) $ i=1..n;
        (if iszero(scale[i]) then scale[i]:=1.0 end_if;) $ i=1..n;
        (ff[i]:= ff[i]/scale[i];) $ i=1..n;
        ((JJ[i,j]:= JJ[i,j]/scale[i];) $ i=1..n;) $ j=1..m;
        /*------------------------------------------------------------------*/
        if useHardwareFloats then
           NewtonShift:= numeric::matlinsolve(JJ, ff, ReturnType = DOM_ARRAY);
           if NewtonShift<>FAIL and iszero(NewtonShift[2]) then
              NewtonShift:=[op(NewtonShift[1])];  
              OK := TRUE;
           else
              OK := FALSE;
           end_if;
        else
           // compute dx = (f'(x))^(-1) f(x) by solving f'(x) dx = f(x)  
           // After the introduction of Scilab, numeric::linsolve is not
           // the fastest solver anymore. In fact, it is now slower than
           // before the introduction of Scilab ;-(
           NewtonShift:= numeric::linsolve(
                          [(_plus(JJ[i,j]*dummies[j]$j=1..m)=ff[i])$i=1..n],
                          dummies, SoftwareFloats);
           if NewtonShift<>FAIL and nops(NewtonShift)=n then 
                   // This is the code for use with numeric::linsolve
                   // linsolve returns solved equations [dx[1]=..,dx[2]=..,..].  
                   // Get rid of dx[i] (i.e., dummy[i]) by picking solution from 
                   // right hand side of the solved equations: 
              NewtonShift:= map(NewtonShift, op, 2);
              OK := TRUE;
           else
              OK := FALSE;
           end_if;
        end_if;
        /*------------------------------------------------------------------*/
        GradientShift:= [_plus(conjugate(JJ[j,i])*ff[j] $ j=1..n) $ i=1..m];
        // Beware: in fsolve([y = x, y = x + 1], [x, y]), the GradientShift
        // tends to zero, but reaches some final tiny value because of round-off.
        // As a result, it passes the tests further down below and fsolve returns
        // a solution. Hence, we must eliminate round-off trash here:
        for i from 1 to m do 
           if specfunc::abs(GradientShift[i]) < 
              macheps * _plus(specfunc::abs(JJ[j, i]*ff[j]) $ j = 1..n) then
              GradientShift[i]:= float(0);
           end_if;
        end_for:
        GS_norm:= Normsqr(GradientShift);  /* = <g,g> = < f'^T f, f'^T f> */
        /* ffTf = <f'g,f'g> = <f'f'^T f , f'f'^T f> */
        ffTf:= _plus(specfunc::abs(_plus(JJ[i,j]*GradientShift[j] $ j=1..m))^2 $ i=1..n); 
        /*------------------------------------------------------------------*/

        if OK then
           Newton_norm:= Normsqr(NewtonShift);
           /* check whether f'^(-1) is ill-conditioned */
           /* <GradientShift,GradientShift> = <g,g>^3/<f'g,f'g>^2 */
           if Newton_norm * machepssqr *ffTf^2 <= GS_norm^3 then 
              NewtonStrategy:= TRUE;
              lastShift:= Newton_norm;
              return(NewtonShift);
           end_if;
        end_if;
        if ffTf>machepssqr*GS_norm then 
           NewtonStrategy:= FALSE;
           GradientShift:= map(GradientShift,_mult,GS_norm/ffTf);
           lastShift:= GS_norm^3/ffTf^2;
           return(GradientShift);
        end_if;
        return(FAIL);
  end_proc;
  /* ----------------------------------------- */
  /*   NewtonStep(x) = x - dx, dx = Shift(x)   */
  /* ----------------------------------------- */
  NewtonStep:=proc(x) local dx, new_x;
  begin /* lastShift = global variable to control convergence     */
        dx:= Shift(x);
        if dx=FAIL then return(FAIL) end_if; 
        if iszero(lastShift) then return(x) end_if;
        if dx<>FAIL then return(zip(x,dx,_subtract)) end_if;
  end_proc;
  /* ------------------------------------------------ */
  /* modifiedNewtonStep(x) = x - t*dx, dx = Shift(x)  */
  /* ------------------------------------------------ */
  modifiedNewtonStep:=proc(x)
  local dx, i, new_x, t;
  begin dx:= Shift(x); 
        if dx=FAIL then return(FAIL) end_if; 
        if iszero(lastShift) then return(x) end_if;
        if lastShift<=machepssqr*(1+Normsqr(x)) then return(x); end_if;
        oldnorm:= Normsqr(ff); /* scaled ff provided by Shift(x) */
        t:=1.0: i:=1;
        while 2^i < 10^DIGITS do
         if i=1 then new_x:= zip(x, dx,_subtract);
                else new_x:= zip(x, map(dx,_mult,t),_subtract);
         end_if;
         //Here is danger of overflow/underflow, so use traperror:
         if traperror((newnorm:= ScaledNormsqr(F(new_x),scale)))<>0
         then return(FAIL); end_if;
         if ( (NewtonStrategy=TRUE) and (newnorm<=oldnorm) )
         or ( (NewtonStrategy=FALSE) and (newnorm<0.9*oldnorm))
            then return(new_x) 
            else t:=t/2;
         end_if;
         i:=i+1;
       end_while;
       FAIL // any acceptable point was returned before 
  end_proc;

  /*------------------------------------------------------------------- */
  /* rand = generator of random floats in [0.0, 1.0]. Sequence is started
    by global system variable SEED. Fix SEED before starting rand inside
    fsolve -> the result of fsolve is deterministic !! 
    Do not use the fast frand, because it depens on DIGITS!
  */
  /*------------------------------------------------------------------- */
  // Do not use the following random generator, because
  // the final result would depend on DIGITS, i.e., the
  // user might geht a different result after increasing
  // DIGITS ;-(
  // rand:= float@random(0..10^DIGITS-1)/10^DIGITS;
  // The following should suffice:
  rand:= float@random(0..10^13-1)/10^13;
  /*------------------------------------------------------------------- */
  randomvector:=[0$m];
  ( if has({leftStartBounds[i],rightStartBounds[i]}, infinity)
    then randomvector[i]:= 
        100*tan@(float(arctan(leftStartBounds[i] /100)) +
                 float(arctan(rightStartBounds[i]/100) 
                      -arctan(leftStartBounds[i] /100))*rand)
    else randomvector[i]:= 
         float(leftStartBounds[i])+ 
         float(rightStartBounds[i]-leftStartBounds[i])*rand
    end_if; ) $ i=1..m;
  /*------------------------------------------------------------------- */
  startingpoint:=proc() begin
    if SingleRun then 
       return(leftStartBounds) 
    else 
       if firstRun and 
          (not RandomMode) and 
          (not complexSearch) then
          // choose a starting point within the search range
          // that is as close to the origin at possible
          // to favour small solutions
          firstRun:= FALSE; // switch off for further runs
          return( [(if leftStartBounds[i] >=0 then
                         leftStartBounds[i]
                    elif rightStartBounds[i] <= 0 then
                         rightStartBounds[i]
                    else float(0)
                    end_if) $ i = 1..m]
                 );
       elif lastRun and
          (not RandomMode) and 
          (not complexSearch) then
          // choose a starting point within the search range
          // that is as close to the the point [1,1,1,...] as possible
          // (this is the heuristics used in the SMT)
          return( [(if leftStartBounds[i] >=1 then
                         leftStartBounds[i]
                    elif rightStartBounds[i] <= 1 then
                         rightStartBounds[i]
                    else float(1)
                    end_if) $ i = 1..m]
                 );
       end_if;
       return(randomvector()) 
    end_if;
  end_proc;
  /*------------------------------------------------------------------- */
  // identify identical solutions found in different Newton searches:         
  EliminateDuplicates:=proc(x)  // insert x into global set outsideSolutions, 
  local xx;                     // if norm(x-xx) is large enough for all xx   
  begin for xx in outsideSolutions do                 // in outsideSolutions  
          if _plus(op(zip(x,xx,abs@_subtract)))<10*macheps
          then userinfo(2,"last solution was found before");
               return();
          end_if;
        end_for;
        outsideSolutions:= outsideSolutions union {x};
  end_proc;

  /*------------------------------------------------------------------- */
  /* Algorithmus ist in den obigen Unter-Prozeduren implementiert ----  */
  /* Nun das Hauptprogramm -------------------------------------------  */
  /*------------------------------------------------------------------- */

  Step:= modifiedNewtonStep;

  if Step=modifiedNewtonStep
     then userinfo(1, "using modified Newton search");
     else userinfo(1, "using standard Newton search");
  end_if;
  if RestrictedMode
     then userinfo(1,"searching inside search range only. No other".
                     " solution will be accepted.");
     else userinfo(1,"searching in and out of search range, any solution".
                     " will be accepted.");
  end_if;

  /* Newton-search until Abbruch (solution found) or Run=maxRun 
    = max number of searches with different startingpoints */

  outsideSolutions:={}; // storage for solutions outside search range 
  insideSolutions:={};  // storage for solutions inside search range 
  maxRun:=  20;         // default max number of random startingpoints 
  maxSearchLength:= 30 + 4.4*ln(1.0*DIGITS); // = 40 for DIGITS = 10 
  if SingleRun then maxRun:= 1;
                    maxSearchLength:= 10*maxSearchLength;
  end_if;
  totalSteps:= 0: Run:= 0; Abbruch:= FALSE; 

  // first while loop runs through various startingpoints 
  firstRun:= TRUE; // the first "random starting point"
                   // is special to prefer small solutions
  lastRun:= FALSE; // the last "random starting point"
                   // is special to comply with the SMTs
                   // heuristics
  while (not Abbruch) and (Run < maxRun) do
    userinfo(2,"starting new Newton sequence");
    x:= startingpoint(); 
    if lastRun then
       Run:= infinity;
    else
       Run:= Run+1; 
    end_if:
    userinfo(2,"1st point : ".expr2text(x)." (random starting point)");

    // next while loop is Newton sequence starting vom x=startingpoint() 
    StepsInThisRun:= 1; // count the Newton steps in each run 

    while (not Abbruch) and (StepsInThisRun < maxSearchLength) do 
      totalSteps:= totalSteps+1; 
      StepsInThisRun:= StepsInThisRun+1;
      if traperror((new_x:= Step(x)))<> 0 then break; end_if;
      if new_x = FAIL then break; // choose next startingpoint 
      else userinfo(3,"    |f'(x)^(-1)f(x)| : ".expr2text(sqrt(lastShift))); 
           x:= new_x; 
           userinfo(3,output::ordinal(StepsInThisRun)." point: ".expr2text(x));
           if lastShift<=machepssqr*(1+Normsqr(x)) then
              x:= map(x, numeric::complexRound, macheps);
              userinfo(2,"");
              userinfo(2,"-------------- solution found: ".expr2text(x));
              newnorm:= Normsqr(F(x));
              userinfo(2,"with residue |f(x)| = ".expr2text(sqrt(newnorm)));
              userinfo(2,"--------------");
              if checkrange(x) 
                 then insideSolutions:= insideSolutions union {x};
                      Abbruch:= TRUE;         // x is the result 
                 else EliminateDuplicates(x); // insert x into outsideSolutions 
                      if not RestrictedMode then Abbruch:= TRUE end_if;
              end_if;
              break;
           end_if;
      end_if;
    end_while;
    SearchLengthTable[Run]:= expr2text(StepsInThisRun-1)." steps";
    if Run = maxRun then 
       Run:= maxRun -1; // do a final round
       lastRun:= TRUE;
    end_if;
  end_while;
  
  // --- Prepare output: ----------------------------------------- 
  // Solutions are stored in x[1],x[2],..,x[n]. The desired output 
  // list of solution equations  [var1 = x[1], var2 = x[2], ... ]  
  // is constructed by zipping the lists [var1,var2,..] and        
  // [x[1],x[2],..] with _equal.                                   
  // ------------------------------------------------------------- 
  userinfo(1,"total number of Newton steps : ".expr2text(totalSteps));
  userinfo(1,"number of search runs : ".expr2text(Run));
  userinfo(1,"of length : ".expr2text(SearchLengthTable));
  userinfo(1,"number of solutions outside search range : ".
                          expr2text(nops(outsideSolutions)));
  userinfo(1,"number of solutions inside search range : ".
                          expr2text(nops(insideSolutions)));

  if (insideSolutions= {}) and (outsideSolutions= {}) then 
     userinfo(1, "no solution found"); 
     return(FAIL);
  end_if;
  if (insideSolutions= {}) and (outsideSolutions<>{}) then 
     userinfo(1,"following solution(s) are not inside search range");
     if ShowAllSolutions then
        return(zip(vars, op(outsideSolutions,i),
                   (var,value)->(var=value)) 
               $ i=1..nops(outsideSolutions) );
     else
        return(FAIL)
     end_if;
  end_if;
  if (insideSolutions<>{}) then 
     if ShowAllSolutions then
        return(zip(vars, op(insideSolutions,i),
                    (var,value)->(var=value)) 
                $ i=1..nops(insideSolutions),
               zip(vars, op(outsideSolutions,i),
                    (var,value)->(var=value)) 
                $ i=1..nops(outsideSolutions) 
              );
     else
        // insideSolutions cannot have more than one element,
        // because the search stopped after one element was
        // found. Nevertheless, use the following construct, in
        // case the functionality is changed in future versions.
        return(zip(vars, op(insideSolutions,i),
                   (var,value)->(var=value)) 
               $ i=1..nops(insideSolutions) );
     end_if;
  end_if;
end_proc:
