/* -----------------------------------------------------------------
 numeric::gaussAGM(a, b) -- the arithmetic/geometric mean of a and b


Calls:       numeric::gaussAGM(a, b)
Parameters:  a, b - arithmetical expression

Details:  

 - gaussAGM computes the iteration 
            [a, b] -> [(a+b)/2, (a+b)*sqrt(a*b/(a+b)^2)]
   which maps a pair [a, b] to its arithmetic/geometric mean.
   This iteration converges quadratically to a value a = b
   called the arithmetic/geometric mean of the starting values
   a and b.
       
 - If a = 0 or b = 0 or a + b = 0, the result 0.0 is returned.

 - If a or b cannot be converted to a real or complex float,
   the symbolic call numeric::gaussAGM is returned.

 - Historical note: Lagrange discovered the arithmetic geometric mean 
   before 1785. Gauss rediscovered it in the 1790s and Gauss and Legendre 
   developed the most complete theory of its use.
 - The following relation to the elliptic integrals hold:

                                  PI/4 * (a + b)
      gaussAGM(a, b) =  ----------------------------------
                        EllipticK((((a-b)/(a+b))^2)^(1/2))

    where ellipticK(m) = PI/2*hypergeom([1/2, 1/2], [1], m^2);
-----------------------------------------------------------------   */

numeric::gaussAGM := proc(a, b)
local fa, fb, aa;
begin
   if iszero(a) or iszero(b) or iszero(a + b) then 
      return(float(0));
   end_if;
   fa:= float(a):
   fb:= float(b):
   if {domtype(fa), domtype(fb)} minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
      return(procname(args()));
   end_if;
   [a, b]:= [fa, fb]:
   if domtype(fa) = DOM_FLOAT and
      domtype(fb) = DOM_FLOAT and
      fa >= 0 and fb >= 0 then
        while specfunc::abs(a - b) > a*10^(-DIGITS) do
          [a, b]:= [(a + b)/2, (a*b)^(1/2)];
        end_while;
   else while specfunc::abs(a - b) > specfunc::abs(a)*10^(-DIGITS) do
          aa:= (a + b)/2;
          b:= aa*(a*b/aa^2)^(1/2);
          a:= aa:
        end_while;
   end_if;
   return(a);
end_proc:
