//------   file  inverse.mu   26.9.97         ---------------------
//-----------------------------------------------------------------
/*  Help file:

 inverse - inverse of a numerical matrix

 Call:

 numeric::inverse( A <, Symbolic> <, hard_soft>  <, Sparse>,
                   <, ReturnType = typ>, <NoWarning>)
                                                                   
 Parameter:

  A         -- a  square matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
  hard_soft -- either 'Hard','HardwareFloats' or 'Soft','SoftwareFloats'
               If DIGITS <=15, the default is HardwareFloats
               If DIGITS  >15, the default is SoftwareFloats
  typ       -- either DOM_ARRAY, DOM_HFARRAY or Dom::Matrix() or Dom::DenseMatrix() 
  NoWarning -- suppress warning when falling back to the symbolic
               mode because of symbolic objects
  Sparse    -- use a sparse algorithm. Recommended input type for A
               is Dom::Matrix
  "dont_use_normal" -- undocumented option to suppress the use of normal

Returns: the inverse of A as an object of type 'typ'.
         If typ is not specified:
         the inverse of an array A is returned as an array. 
         the inverse of an hfarray A is returned as an hfarray. 
         The inverse of a Dom::DenseMatrix(any Ring)
         is returned as a Dom::DenseMatrix().
         The inverse of other matrix objects (Dom::/*Dense*/Matrix(any Ring),
         Dom::SquareMatrix(..), Dom::MatrixGroup(..)
         are returned as Dom::Matrix()-objects.
                                                                   
Synopsis:

  numeric::inverse(A); returns the inverse of A computed by
  Gauss elimination with partial pivoting. The result type
  depends on the type of the input matrix A or the requested 
  return type.
                                                                  
  Non-float numerical entries are admissible, they will be 
  evaluated to floats in the course of the computation.

  The option Symbolic prevents conversion of the matrix entries
  to floats. No pivoting will occur unless necessary. With this
  option non-numeric symbolic entries are admissible. However, 
  their presence will slow down the computation drastically.

  Examples: 

>> A := array( 1..2 , 1..2 , [ [1,2],[3,4] ] ):

>> numeric::inverse(A);
                             +-            -+
                             |  -2.0,  1.0  |
                             |              |
                             |   1.5, -0.5  |
                             +-            -+

>> numeric::inverse(A,Symbolic);

                              +-           -+
                              |   -2,   1   |
                              |             |
                              |  3/2, -1/2  |
                              +-           -+
                                                                  */
//-----------------------------------------------------------------
numeric::inverse:=proc(A)
local B, useHardwareFloats, HardwareFloatsRequested,
      m, n, method, dowarn, specialwarning, returnType, 
      use_normal, normal2, i, j, k, tmp, tmp1, pivot, pivindex,
      dont_use_normal_requested,
      rownorm, density, nontriv, pA, pB, sparsemode,
      rowrangechanged, colrangechanged,
      originalrowrange, originalcolrange;
begin
  if args(0) <1 then error("expecting at least one argument") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  rowrangechanged:= FALSE;
  colrangechanged:= FALSE;
  sparsemode:= FALSE;
  if domtype(args(1)) = DOM_ARRAY or
     domtype(args(1)) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalrowrange:= op(A, [0, 2]):
     originalcolrange:= op(A, [0, 3]):
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       rowrangechanged:= TRUE;
       colrangechanged:= TRUE;
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     originalrowrange:= 1..n:
     originalcolrange:= 1..m:
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then 
     error("expecting a square matrix"); 
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE; 
  else useHardwareFloats:= FALSE; 
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  dowarn:= TRUE;
  specialwarning:= FAIL;
  dont_use_normal_requested:= FALSE;

  for i from 2 to args(0) do
      case args(i)
      of Symbolic do method:= 1:
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Hard do
      of HardwareFloats do
                     HardwareFloatsRequested:= TRUE;
                     if method = 1 then
                        specialwarning:= 
                                "ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'":
                        method:= 1;
                        useHardwareFloats:= FALSE: 
                        HardwareFloatsRequested:= FALSE;
                     else
                        useHardwareFloats:= TRUE: 
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do  
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      of Sparse do   sparsemode:= TRUE;
                     break;
      of "dont_use_normal" do
                     dont_use_normal_requested:= TRUE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, 
                                     Dom::DenseMatrix(), Dom::Matrix()} 
                 <> {} then
                 error("illegal return type ".expr2text(returnType). 
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:
     

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if sparsemode then 
        if not numeric::startHardwareFloats("sparse_m_inverse") then
           userinfo(1, "cannot start the hardware floats interface"):
           if HardwareFloatsRequested and dowarn then
              warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
           end_if;
           useHardwareFloats:= FALSE;
           break;
        end_if;

        userinfo(1, "using hardware floats"):

        if domtype(A) = matrix then
           B:= map(extop(A::dom::mapNonZeroes(A, float), 3), poly2list);
        elif domtype(A) = densematrix then
           B:= map(expr(A), float);
           B:= [[[B[i,j], i] $ i = 1..m] $ j = 1..n]:
        elif domtype(A) = DOM_HFARRAY then
           B:= A;
        elif domtype(A) = DOM_ARRAY then
           B:= map(A, float);
           // convert matrix to a sparse list of columns
           B:= [[[B[i,j], i] $ i = 1..m] $ j = 1..n]:
        else // should not arrive here
           B:= FAIL;
        end_if:

        if traperror((

               B:= hfa::sparse_m_inverse(B, m, n, TRUE)
               // 4-th arg TRUE : the lists B represent columns 
               // 4-th arg FALSE: the lists B represent rows 

              )) <> 0 or
              has(B, RD_NAN) or
              has(B, RD_INF) or
              has(B, RD_NINF) then
                if HardwareFloatsRequested and dowarn then
                   warning("HardwareFloats failed, ".
                           "using 'SoftwareFloats' instead");
                end_if;
                userinfo(1, "'HardwareFloats' failed"):
                useHardwareFloats:= FALSE;
                break;
        end_if;
  
     else  // sparsemode = FALSE

        if not numeric::startHardwareFloats("inverse") then
           userinfo(1, "cannot start the hardware floats interface"):
           if HardwareFloatsRequested and dowarn then
              warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
           end_if;
           useHardwareFloats:= FALSE;
           break;
        end_if;

        userinfo(1, "using hardware floats"):

        if domtype(A) <> DOM_HFARRAY then
           if traperror((B:= hfarray(1..m, 1..n, [op(A)]))) <> 0 or
              has(B, RD_NAN) or
              has(B, RD_INF) or
              has(B, RD_NINF) then
                if HardwareFloatsRequested and dowarn then
                   warning("HardwareFloats failed, ".
                           "using 'SoftwareFloats' instead");
                end_if;
                userinfo(1, "'HardwareFloats' failed"):
                useHardwareFloats:= FALSE;
                break;
           end_if;
        else
           B:= A;
        end_if;

        if traperror((B:= hfa::inverse(B))) <> 0 or
              has(B, RD_NAN) or
              has(B, RD_INF) or
              has(B, RD_NINF) then
                if HardwareFloatsRequested and dowarn then
                   warning("HardwareFloats failed, ".
                           "using 'SoftwareFloats' instead");
                end_if;
                userinfo(1, "'HardwareFloats' failed"):
                useHardwareFloats:= FALSE;
                break;
        end_if;

     end_if:
      
     //------------------------------
     //-------- return --------------
     //------------------------------

     if B = FAIL then
        return(FAIL);
     end_if:

     // proceed to the end of this routine. There, the
     // matrix is converted to the requested return type. 
     // Further, rowrange and colrange are adapted if necessary.
  end_case; 
  //---------------------------------
  //----- end of HardwareFloats -----
  //---------------------------------

  case useHardwareFloats
  of FALSE do
  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  userinfo(1, "using software floats"):
  // Do use domtype(A) <> DOM_ARRAY here to avoid
  // unnecessary loading of the DOMAINS package
  if domtype(A) <> DOM_ARRAY and 
     domtype(A) <> DOM_HFARRAY and
     A::dom::hasProp(Cat::Matrix)=TRUE then  
     // convert to domtype DOM_ARRAY for speed 
     A:= expr(A);
  end_if;
  if domtype(A) <> DOM_ARRAY and
     domtype(A) <> DOM_HFARRAY then
     error("the matrix must be an array, an hfarray, or of category 'Cat::Matrix'");
  end_if:

  if method = 2 then
    if domtype(A) = DOM_ARRAY then
      B:= map(A, float):
      if {op(map(B, domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
        if dowarn then
           warning("symbolic entry found, switching to symbolic mode");
        end_if;
        method := 1;
        if not dont_use_normal_requested then
           use_normal := TRUE
        end_if:
      else 
        A:= B;
      end_if;
    else // A = DOM_HFARRAY
       // convert to DOM_ARRAY for speed (lots of 
       // indexed reading and writing is used below)
       A:= array(1..m, 1..n, [op(A)]):
    end_if;
  end_if;

  use_normal:=FALSE;
  normal2:= x -> normal(x, Expand = FALSE);
  if method=1 then
    if (indets(A)<>{}) 
    or has(map({op(A)},domtype),DOM_NIL)
    or has(map({op(A)},domtype),DOM_EXPR) then 
       if not dont_use_normal_requested then
         use_normal:=TRUE
       end_if;
    end_if;
  end_if;

  if method=1 then
    userinfo(1,"symbolic mode, no pivoting unless necessary");
  end_if;
  if method=2 then 
    userinfo(1,"using column pivoting"); 
  end_if;

  if domtype(A) = DOM_ARRAY then
     B:= array(1..n,1..n,[[0$n]$n]); // initialize inverse
  elif domtype(A) = DOM_HFARRAY then
     B:= hfarray(1..n,1..n,[[0$n]$n]); // initialize inverse
  else
     error("unexpected internal matrix type");
  end_if;
  (B[i,i]:=1) $ i=1..n;    // identity matrix       

  density:=0; // compute how many of the n(n+1)/2 elements in upper    
              // factor U of the LU decomposition of A are non-trivial 
  for j from 1 to n-1 do
  // search for pivot element 
    if method=1 then
      if iszero(A[j,j]) then
         for pivindex from j+1 to n-1 do
             if not iszero(A[pivindex,j]) then break; end_if;
         end_for;
      else pivindex:=j;
      end_if;
    end_if;
    if method=2 then
      pivot:=0; pivindex:=j;
      for i from j to n do 
          rownorm:=max(specfunc::abs(A[i,k]) $ k=j+1..n );
          if not iszero(rownorm) then
             specfunc::abs(A[i,j])/rownorm;
             if pivot<last(1) then pivot:=last(1); pivindex:=i; end_if;
          else pivindex:=i;break;
          end_if;
      end_for;
    end_if;
    if pivindex<>j then
      (tmp:=A[j,k];A[j,k]:=A[pivindex,k];A[pivindex,k]:=tmp;) $ k=1..n;
      (tmp:=B[j,k];B[j,k]:=B[pivindex,k];B[pivindex,k]:=tmp;) $ k=1..n;
    end_if;
    if iszero(A[j,j]) then return(FAIL) end_if;

    // check sparsity of j.th row to reduce cost of elimination step 
    // sparsity of A: 
    pA:=[0$n-j]; // list to store indices of non-trivial entries of jth row in A 
    nontriv:=1;
    for k from j+1 to n do 
        if not iszero(A[j,k]) then pA[nontriv]:=k;nontriv:=nontriv+1;end_if;
    end_for;
    density:=density+nontriv-1;
    // sparsity of B: 
    nontriv:=1;
    pB:=[0$n]; // list to store indices of non-trivial entries of jth row in B 
    for k from 1 to n do 
        if not iszero(B[j,k]) then pB[nontriv]:=k;nontriv:=nontriv+1;end_if;
    end_for;

    // Now do the elimination. 
    // Only need to touch elements in rows pA[1],pA[2],.. and pB[1],pB[2],.. 
    for i from j+1 to n do
      if iszero(A[i,j]) then next end_if;
      tmp:=A[i,j]/A[j,j];
      if use_normal then 
         for k from 1 to n-j do
           if pA[k]=0 then break else A[i,pA[k]]:=normal2(A[i,pA[k]]-tmp*A[j,pA[k]]);end_if;
         end_for;
         for k from 1 to n do
           if pB[k]=0 then break else B[i,pB[k]]:=normal2(B[i,pB[k]]-tmp*B[j,pB[k]]);end_if;
         end_for;
      else 
         for k from 1 to n-j do
           if pA[k]=0 then 
              break 
           else 
              if method = 2 then
                 tmp1:= specfunc::abs(A[i, pA[k]]);
              end_if;
              A[i,pA[k]]:=A[i,pA[k]]-tmp*A[j,pA[k]];
              if method = 2 and specfunc::abs(A[i, pA[k]]) < 10.0^(-DIGITS)*tmp1 then
                 A[i,pA[k]]:= float(0);
              end_if:;
           end_if;
         end_for;
         for k from 1 to n do
           if pB[k]=0 then 
              break 
           else 
              B[i,pB[k]]:=B[i,pB[k]]-tmp*B[j,pB[k]];
           end_if;
         end_for;
      end_if;
    end_for;
  end_for;
  if iszero(A[n,n]) then 
     return(FAIL)
  end_if;

  // Now A is upper triangular. Backsubstitution: 
  if density> 0.4*n^2
  then // original version, does not make use of sparsity in upper triangular A 
       for j from 1 to n do
         for i from n downto 1 do
           B[i,j]:=(B[i,j]-_plus(A[i,k]*B[k,j] $ k=i+1..n))/A[i,i] ;
           if use_normal then B[i,j]:=normal2(B[i,j]) end_if;
         end_for;
       end_for;
  else // try to use sparsity 
       for i from n downto 1 do
         for k from i+1 to n do
           if not iszero(A[i,k]) then  
              (B[i,j]:=B[i,j]-A[i,k]*B[k,j];) $ j=1..n;
           end_if;
         end_for;
         (B[i,j]:=B[i,j]/A[i,i];) $ j=1..n;
         if use_normal then (B[i,j]:=normal2(B[i,j]);)$ j=1..n; end_if;
       end_for;
  end_if;
  end_case; // case useHardwareFloats = FALSE

  //-------------------------
  //------- return ----------
  //-------------------------
  if B::dom = returnType and
     (not rowrangechanged) and
     (not colrangechanged) then
       return(B):
  end_if:
  case returnType
     // mind the ordering: process DOM_{HF}ARRAY first to
     // avoid unnecessary loading of the DOMAIN package
  of DOM_ARRAY do
       if domtype(B) <> DOM_ARRAY or rowrangechanged or colrangechanged then
          // B = array(n1..n2, m1..m2) -> B^(-1) = array(m1..m2, n1..n2) !
          B:= array(originalcolrange, originalrowrange, [op(B)]);
       end_if;
       return(B);
  of DOM_HFARRAY do
       if domtype(B) <> DOM_HFARRAY then
          // B = array(n1..n2, m1..m2) -> B^(-1) = array(m1..m2, n1..n2) !
          if traperror((
              B:= hfarray(originalcolrange, originalrowrange, [op(B)])
             )) <> 0 or
             has(B, RD_NAN) or
             has(B, RD_INF) or
             has(B, RD_NINF) then
               if dowarn then
                  warning("cannot return the inverse as an hfarray. ".
                          "Returning it as an array of type DOM_ARRAY instead");
               end_if:
               return(array(originalcolrange, originalrowrange, [op(B)]));
          end_if:
       elif rowrangechanged or colrangechanged then
        //B:= subsop(B, 0 = (2, originalrowrange, originalcolrange));
          B:= hfarray(originalcolrange, originalrowrange, [op(B)]);
       end_if;
       return(B);
  of Dom::Matrix() do
       return(matrix(B));
  of Dom::DenseMatrix() do
       return(returnType::create(B));
  otherwise
       error("unexpected return type"):
  end_case;
end_proc:
