/*-------------------------------------------------------------
++ kamo, 06.2.94
  Nonrecursive version of the inverse discrete Fast Fourier Transformation.
  The inverse discrete fourier transform is defined by
        A[j] := 1/N* sum(a[k]*exp(2*PI*i*j*k/N), k=0..N-1), for j=0..N-1
  so N sample points will be transformed into N points. 
  The inverse FFT-algorithm needs O(n logn) time.
++ W. Oevel, 11.3.99
   argument m in ifft(A,m) now obsolete (deduced from nops(A))
++ W. Oevel, 12.3.99
   Option Symbolic eingebaut.
   w=exp(I*..) statt w= cos(..)+I*sin(..)
++ W. Oevel, 12.12.99
   Umbenennnung ifft -> invfft
++ W. Oevel, 23.3.00
   auf multi-dim inverse FFT erweitert
   z.B.
   >>  numeric::invfft([L[1],L[2],..,L[2^m]]); // 1-dim inverse FFT via list
   >>  A:= array(1..n[1],1..n[2], .. ,1..n[d], [..data..]) 
   >>  numeric::invfft(A); // d-dim inverse FFT via array
++ W. Oevel, 15.12.01
   HardwareFloats support added
++ W. Oevel, 6.5.06
   Software float code extended to lists of arbitrary length
++ W. Oevel, 30.10.06
   auf hfarrays erweitert, ReturnType und Clean eingefuehrt
++ W. Oevel, 2.1.07
   Bluestein-algorithm for large prime data length
//------------------------------------------------------------

   numeric::fft(L <, mode>, <ReturnType = t>, <Clean>)
   numeric::fft(A <, mode>, <ReturnType = t>, <Clean>)
   numeric::fft(M <, mode>, <ReturnType = t>, <Clean>)

   L -- a list, an array(1..n, [...]) or an hfarray(1..n, [...])
   A -- an array(1..n1, 1..n2, .. , 1..n.d, [...]) or
        an hfarray(1..n1, 1..n2, .. , 1..n.d, [...])
   M -- a matrix of category Cat::Matrix
   mode - Hard, HardwareFloats,
          Soft, SoftwareFloats,
          Symbolic
   t - either DOM_LIST, DOM_ARRAY, DOM_HFARRAY,
       Dom::Matrix() or Dom::DenseMatrix()
   Clean - set small values to 0 and apply numeric::complexRound

   With ReturnType = Dom::Matrix(), only 2-dim arrays and
   hfarray are allowed as input data

   With ReturnType = DOM_LIST, the input data may be
   L, A or M. For higher dimensional FFT, the output list
   is a plain list with all the operands that the output
   array with ReturnType = DOM_ARRAY would have
-------------------------------------------------------------*/

numeric::invfft := proc(A)
local Atype, returnType, symbolic, useHardwareFloats,
      HardwareFloatsRequested, B, doclean, tol,
      dim, nnn, mmm, result, dimfactors,
      n, m, fPI, l, le, le1,
      u, w, j, i, ip, t, nv2, nm1, k,
      M, N, p, r, kk, ln2, sMi, sMj, sMip,
      shift, start;
save DIGITS;
begin
    if args(0)<1 then error("expecting at least one argument"); end_if;
    if args(0)>4 then error("expecting no more than 4 arguments"); end_if;

    Atype:= domtype(A);

    // set the matrix dimensions for ReturnType = Dom::Matrix()
    case Atype
    of DOM_LIST do 
       [dim, mmm, nnn]:= [1, nops(A), 1]: 
       break;
    of DOM_ARRAY do
    of DOM_HFARRAY do
       if op(A, [0, 1]) = 1 then // 1-dim array/hfarray
          [dim, mmm, nnn]:= [1, op(A, [0, 2, 2]), 1]:
       else
          [dim, mmm, nnn]:= [op(A, [0, 1]), op(A, [0, 2, 2]), op(A, [0, 3, 2])]:
       end_if:
    end_case:

    // the funny formulation of the following check serves
    // to avoid unnecessary loading of the domain stuff
    // triggered by a call of Dom::Matrix:
    if Atype<>DOM_LIST and
       Atype<>DOM_ARRAY and
       Atype<>DOM_HFARRAY then
       if A::dom::hasProp(Cat::Matrix)=TRUE then
          if A::dom::constructor = Dom::DenseMatrix then
             returnType:= Dom::DenseMatrix();
          elif A::dom <> Dom::Matrix() then
             A:= Dom::Matrix()(A);
             returnType:= Dom::Matrix();
          else
             returnType:= Dom::Matrix();
          end_if;
          // convert to an array rather than to an hfarry,
          // since the matrix might contain symbolic data
          dim:= 2:
          [mmm, nnn]:= A::dom::matdim(A);
          A:= expr(A);
          Atype:=  domtype(A):
       else
          error("expecting the data as a list, an array, an hfarray, or ".
                "a matrix of category 'Cat::Matrix'"):
       end_if:
    else
       returnType:= Atype:
    end_if;

    //---------------------------------------------
    // Check the options
    //---------------------------------------------
    // set defaults
    symbolic:= FALSE;
    if DIGITS <= 15 then
         useHardwareFloats:= TRUE;
    else useHardwareFloats:= FALSE;
    end_if;
    HardwareFloatsRequested:= FALSE;
    if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
    end_if:
    doclean:= FALSE;

    for i from 2 to args(0) do
      case args(i)
      of Hard do
      of HardwareFloats do
                     useHardwareFloats:= TRUE:
                     HardwareFloatsRequested:= TRUE;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Symbolic do useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     symbolic := TRUE;
                     break;
      of Clean do    doclean:= TRUE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            op(args(i), 1) = ReturnType then
              returnType:= op(args(i), 2);
         else
            error("unexpected argument ".expr2text(args(i)));
         end_if:
      end_case;
    end_for:

    // the funny formulation of the following check serves
    // to avoid unnecessary loading of the domain stuff
    // triggered by a call of Dom::Matrix/Dom::DenseMatrix:
    if (not contains({DOM_LIST, DOM_ARRAY, DOM_HFARRAY}, returnType)) then
       if returnType <> Dom::Matrix() and
          returnType <> Dom::DenseMatrix() then
          error("unexpected return type ".expr2text(returnType));
       else
          if dim > 2 then
             error("cannot return a ".expr2text(dim)." dimensional FFT ".
                   "as a matrix"):
          end_if:
       end_if:
    end_if:

    //--------------------------------
    // ----- use HardwareFloats ------
    //--------------------------------
    case useHardwareFloats // use case because we want to use break below
    of TRUE do
       if DIGITS > 15 then
          if HardwareFloatsRequested then
             warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                     " cannot be achieved with the requested 'HardwareFloats'."):
          end_if;
          useHardwareFloats:= FALSE;
          break;
       end_if;

       if not numeric::startHardwareFloats("fft") then
          userinfo(1, "cannot start the 'HardwareFloats' interface"):
          if HardwareFloatsRequested then
             warning("cannot start the 'HardwareFloats' interface, ".
                     "using 'SoftwareFloats' instead");
          end_if;
          useHardwareFloats := FALSE;
          break;
       end_if;

       userinfo(1, "trying hardware floats"):

       /*-----------------------------------------------------------
       //  call the hfa module
       -----------------------------------------------------------*/
       // The hfa module expects a hfarray as input.
       // If A contains symbols or numerical stuff that
       // cannot be represented by hardware floats, the
       // conversion to an hfarray will fail. Use traperror
       // to detect these cases and fall back to software floats
       if traperror((
          if Atype = DOM_LIST then
             B:= hfarray(1..nops(A), A);
          elif Atype = DOM_ARRAY then
             assert(op(A, [0, 1]) = dim):
             B:= hfarray((1..op(A,[0,i,2])-op(A,[0,i,1])+1)$i=2..1+dim,[op(A)]);
          elif Atype = DOM_HFARRAY then
             assert(op(A, [0, 1]) = dim):
             if {op(A, [0,i,1]) $ i = 2 .. 1 + dim} = {1} then
                B:= A; // no need to waste memory
             else
                B:= hfarray((1..op(A,[0,i,2])-op(A,[0,i,1])+1)$i=2..1+dim,[op(A)]);
             end_if:
          else
             error("unexpected case"):
          end_if;
         )) <> 0 then
         userinfo(1, "'HardwareFloats' failed"):
         useHardwareFloats:= FALSE;
         break;
       end_if;

       assert(op(B, [0, 1]) = dim):
       n:= [op(B, [0, i, 2]) $ i = 2 .. 1 + dim];
       dimfactors:= [Factored::convert_to(factor(n[i]), DOM_LIST) $ i = 1..dim]:

       if traperror((result:= hfa::fft(B, dimfactors, 1))) = 0 then
          if has(result, RD_NAN) or
             has(result, RD_INF) or
             has(result, RD_NINF) then
               userinfo(1, "'HardwareFloats' failed"):
               useHardwareFloats:= FALSE;
          end_if:
          break;
       else // proceed to the branch useHardwareFloats = FALSE
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
       end_if;
    end_case;
    //---------------------------------
    // end of useHardwareFloats = TRUE
    //---------------------------------

    if useHardwareFloats = FALSE then
       //---------------------------------
       //---- use SoftwareFloats ---------
       //---------------------------------
       if HardwareFloatsRequested then
          warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
       end_if;

       userinfo(1, "using software floats"):

       if symbolic then
            fPI:= PI;
       else fPI:= float(PI); 
            if Atype <> DOM_HFARRAY then
               A:= map(A, float):
            end_if;
       end_if;
       ln2:= 0.69314718056: // = ln(2.0)

       // Indexed reading and writing for arrays is
       // as fast as lists. Hfarrays, however, are
       // significantly slower. So: use list or array
       if domtype(A) = DOM_HFARRAY then
          if dim = 1 then // use DOM_LIST
             A:= [op(A)];
          else // use DOM_ARRAY
             A:= array(op(A, [0, i]) $ i = 2 .. 1+dim, [op(A)]);
          end_if;
       end_if;
   
       //---------------------------------------
       // 1-dimensional inverse FFT using a list/array/hfarray
       //---------------------------------------
       if dim = 1 then
         n:= nops(A):
         if n=0 then error("empty data list/array") end_if;
         DIGITS:= DIGITS+10:
         m:= round(ln::float(float(n))/ln2);
         DIGITS:= DIGITS-10:
         if n = 2^m then 
           le1 := n; 
           (le := le1; le1 := le/2; u := 1; 
            w := exp(I*fPI/le1);
            (for i from j to n step le do
                 ip := i+le1;
                 t := A[i]+A[ip];
                 A[ip]:= (A[i]-A[ip])*u; 
                 A[i] := t;
             end_for;
             u :=  u * w;
            ) $ j=1..le1
           ) $ l=1..m;
           nv2 := n/2; nm1 := n-1; j := 1;
           (if i < j then
                t := A[j];  A[j] := A[i];  A[i] := t;
            end_if;
            k := nv2;
            while k < j do  j := j-k; k := k/2 end_while;
            j := j+k
           ) $ i=1..nm1;
           result:= map(A, _mult, 1/n);
         else
           result:= map(numeric::invdft_(A, fPI), _mult, 1/n);
         end_if;
       else // dim > 1
         //---------------------------------
         // d-dimensional inverse FFT of an array
         //---------------------------------
         // expecting A = array/hfarray(1..n[1],1..n[2],...,1..,n[d])
         // d = dimension (stored in dim)
         assert(dim = op(A, [0,1])):

         // check array input
         DIGITS:= DIGITS+10:
         for k from 1 to dim do
            // n[k] = k-th grid size
            n[k]:= op(A, [0, k+1, 2]);
            m[k]:= round(ln::float(float(n[k]))/ln2);
            if op(A, [0, k+1, 1]) <> 1 // indices must start from 1
            then 
               error("array indices must start with 1");
            end_if;
         end_for;
         DIGITS:= DIGITS-10:

         // convert d-dimensional array to a linear list
         A:= [op(A)];

         // Now, the work starts:
         N:= 1;
         M:= _mult(n[k] $ k=1..dim);

         for k from 1 to dim do // transform the k-th index
       
           M:= M/n[k];  // M = n[dim]*n[dim-1]*..*n[k+1]
                        // N = n[k-1]*n[k-2]*..*n[1]
           start:= 0;
           for p from 1 to N do
              for r from 1 to M do
                // start = r + (p-1)*M*n[k]
                start:= start + 1;

                //------------------------------------------------------
                // Either do the 1-dim inv FFT of the k-th index via numeric::invfft
                //------------------------------------------------------
                // // Warning: delete final multiplication with 1/N below,
                // // if you want to use this version !!
                // B:= [0$n[k]]; // initialize
                // (B[j]:= A[start + (j-1)*M];) $ j=1..n[k];
                // B:= numeric::invfft(B);
                // (A[start + (j-1)*M]:= B[j];) $j=1..n[k];
                //------------------------------------------------------

                //---------------------------------------------------
                // or do the direct version of the 1-dim inverse FFT as above
                //---------------------------------------------------
                if n[k] = 2^m[k] then
                   shift:= start - M;
                   le1 := n[k];
                   for l from 1 to m[k] do
                     le := le1;
                     le1 := le/2;
                     u := 1;
                     w := exp(I*fPI/le1);
                     for j from 1 to le1 do
                       for i from j to n[k] step le do
                          sMi:= shift + M*i;
                          sMip := sMi + le1*M;
                          t := A[sMi] + A[sMip];
                          A[sMip]:= (A[sMi] - A[sMip])*u;
                          A[sMi] := t;
                       end_for;
                       u :=  u * w;
                     end_for;
                   end_for;
                   nv2 := n[k]/2;
                   nm1 := n[k]-1;
                   j := 1;
                   (if i < j then
                        sMi:= shift + M*i;
                        sMj:= shift + M*j;
                        [A[sMj], A[sMi]]:= [A[sMi], A[sMj]];
                    end_if;
                    kk := nv2;
                    while kk < j do  j := j-kk; kk := kk/2 end_while;
                    j := j+kk
                   ) $ i=1..nm1;
                else
                  // data length <> 2^n
                  B:= [0$n[k]]; // initialize
                  (B[j]:= A[start + (j-1)*M];) $ j=1..n[k];
                  B:= numeric::invdft_(B, fPI);
                  (A[start + (j-1)*M]:= B[j];) $j=1..n[k];
                end_if;
                //------------------------------------------------------
                // end of 1-dim inverse FFT of k-th index
                //------------------------------------------------------
              end_for: // for r
              start:= start + M*(n[k]-1);
           end_for: // for p
           N:= N*n[k];
         end_for; // for k
         result:= map(A,_mult,1/N);
       end_if; // if dim = 1 or dim > 1
    end_if; // if useHardwareFloats = FALSE

    // depending on the input, the result is now a list,
    // an array, or an hfarray

    if doclean and not symbolic then
       tol:= 10^(-DIGITS)*max(map([op(result)], specfunc::abs)):
       result:= map(result, proc(x)
                            begin
                              if specfunc::abs(x) < tol then
                                 float(0)
                              else
                                 numeric::complexRound(x)
                              end_if
                            end_proc):
    end_if:

    if domtype(result) = returnType then
          // use the result computed above
    elif returnType = DOM_LIST then
         result:= [op(result)];
    elif returnType = DOM_ARRAY then
       if domtype(result) = DOM_LIST then
          if dim = 1 then
            result:= array(1..mmm, result);
          else
            result:= array((1..n[k]) $ k=1..dim, result):
          end_if;
       elif domtype(result) = DOM_HFARRAY then
          if dim = 1 then
            result:= array(1..mmm, [op(result)]);
          else
            result:= array((1..n[k]) $ k=1..dim, [op(result)]):
          end_if;
       else
          error("unexpected case"):
       end_if:
    elif returnType = DOM_HFARRAY then
       if domtype(result) = DOM_LIST then
          if dim = 1 then
            result:= hfarray(1..mmm, result);
          else
            result:= hfarray((1..n[k]) $ k=1..dim, result):
          end_if;
       elif domtype(result) = DOM_ARRAY then
          if dim = 1 then
            result:= hfarray(1..mmm, [op(result)]);
          else
            result:= hfarray((1..n[k]) $ k=1..dim, [op(result)]):
          end_if;
       else
          error("unexpected case"):
       end_if:
    elif returnType = Dom::Matrix() then
       result:= Dom::Matrix()(mmm, nnn, result);
    elif returnType = Dom::DenseMatrix() then
       result:= Dom::DenseMatrix()(mmm, nnn, result);
    else
       error("unexpected case"):
    end_if:
    return(result):
end_proc:

//----------------------------------------------------
// Utility used in numeric::invfft.
// Note: invdft_ implements the inverse DFT without(!)
//       division by data length! This division is
//       done inside invfft.
//----------------------------------------------------
numeric::invdft_:= proc(data, fPI)
local n, m, i, k, kk, f, result, p, r, omega, tmp, tmpp;
begin
   // data = a list
   n:= nops(data);
   if n = 1 then
      return(data);
   end_if;
   // initialize container for the return value
   result:= [0 $ n];
   //----------------------------------------------------------
   // If the data length is even, use divide & conquer.
   // For the factor 2 we use a special implementation
   // that saves a few multiplications
   //----------------------------------------------------------
   if n mod 2 = 0 then
      m:= n/2:
      f:= table():
      f[0] := numeric::invdft_([data[2*k - 1] $ k = 1..m], fPI):
      f[1] := numeric::invdft_([data[2*k] $ k = 1..m], fPI):
      omega:= exp(I*fPI/m);
      tmp:= 1:
      for k from 1 to m do
         tmpp:= tmp*f[1][k]:
         result[ k ]:= f[0][k] + tmpp:
         result[k+m]:= f[0][k] - tmpp:
         tmp:= tmp*omega;
      end_for:
      return(result);
   end_if;
   //----------------------------------------------------------
   // If the data length is not prime, use divide & conquer
   // This is the generic implementation working for any
   // integer p that divides n. It will only be called for
   // values of p that are prime.
   //----------------------------------------------------------
   p:= Factored::convert_to(factor(n), DOM_LIST):
   // Pick the *smallest* factor p[2] of n. This leads to fewer
   // calls to exp (for the cost of more multiplications) comparing
   // with choosing the largest factor p[nops(p)-1].
   p:= p[2];
   assert(n mod p = 0);
   if p < n then
      m:= n/p:
      //---------------
      // DIVIDE the data into p different data lists
      // and compute der DFTs of these lists:
      //---------------
      f:= table():
      for r from 0 to p-1 do
        f[r]:= numeric::invdft_([data[1 + r + p*k] $ k = 0..m-1], fPI):
      end_for:
      //---------------
      // SYNTHESIZE the DFT data of the smaller lists to
      // the DFT data of the entire data list:
      //---------------
      omega:= exp(I*2*fPI/n):
      tmp:= 1;
      for k from 0 to n-1 do
         kk:= 1 + (k mod m);
         // Here, tmp = exp(I*k*2*fPI/n);
         result[1+k]:= f[0][kk] + _plus(tmp^r * f[r][kk] $ r = 1..p-1);
         tmp:= tmp*omega:
      end_for;
      return(result);
   end_if;

   //-------------------------------------------------------------------
   // If the data length is prime and large, use the Bluestein algorithm
   //-------------------------------------------------------------------
   if n > 100 then
      return(numeric::invbluestein(data, ReturnType = DOM_LIST));
   end_if: 

   //----------------------------------------------------------
   // If the data length is prime and small, use the defining DFT formula
   //----------------------------------------------------------
   omega:= table():
   omega[0]:= 1;
   omega[1]:= exp(I*2*fPI/n);

   // generate omega[k]:= exp(I*k*2*fPI/n):
   for k from 2 to n-1 do
     omega[k]:= omega[k-1]*omega[1];
   end_for:

   for k from 0 to n-1 do
      result[k+1] := data[1] + _plus(data[i+1]*omega[(i*k) mod n] $ i = 1..n-1);
   end_for;

   return(result);
end_proc:

/* --------------------------------------------------------
Utility numeric::bluestein for prime data length:

Bluestein's chirp z-transform algorithm (1969) rewrites the
FFT as a convolution, which can be computed efficiently
via standard radix 2 FFT after suitable zero padding of
the sequences. This is useful for doing the FFT of data
of prime length.

F[k] = sum(data[j]*exp(2*PI*i*j*k/N), j = 0..N-1)
     = exp(PI*i*k^2/N) * sum(  data[j]*exp(PI*i*j^2/N)
                                 * exp(-PI*i/N * (k - j)^2),
                                 j = 0..N-1)
     = conjugate(b[k]) * sum( a[j]*b[k-j], j = 0 .. N-1)

with  a[j] = data[j]*exp(i*PI/N * j^2), j = 0 .. N-1
and   b[j] = exp(-i*PI/N*j^2), j = -(N-1) .. N-1

Do zero-padding of a[j] and b[j] to some radix 2 length N
(N needs to be larger than 2*n - 2) and compute the convolution
sum(a[j]*b[k-j], j=0..N-1) via standard radix 2 FFT.
---------------------------------------------------------*/
numeric::invbluestein:= proc(data)
local n, N, fPI, b, bb, j, a;
begin
   n:= nops(data); // original data length
   N:= 2^ceil(ln(float(2*n-1))/ln(2.0)); // data length after zero padding:
   assert(N >= 2*n - 1);
   assert(iszero(frac(log(2, N))));
   fPI:= float(PI);
   b:= [exp(-I*fPI/n*j^2) $ j = 0 .. n-1];
   bb:= [b[j] $ j = 1..n, (0 $ N - 2*n + 1), b[n-j] $ j = 0..n-2];
   bb:= numeric::fft(bb, ReturnType = DOM_LIST);
   b:= map(b, conjugate);
   a:= [(data[j]*b[j] $ j = 1..n), (0 $ N - n)];
   a:= numeric::fft(a, ReturnType = DOM_LIST);
   a:= numeric::invfft(zip(a, bb, _mult), ReturnType = DOM_LIST);
   [b[j]*a[j] $ j = 1..n];
end_proc:

