/* -------------------------------------------------------------------------
Info: The linear least square solution for the linear system A*x = b 
      (A = m x n -Matrix, b = m vector) minimizing norm(A*x - b, 2)
      is the solution of the 'normal equations'  A^H*A*x = A^H*b,
      where A^H is the hermitean transpose of A.

      The solution is unique, if rank(A) = n.

Calls:  numeric::leastSquares(A, b, 
                              <, hard_soft>
                              <, method>, 
                              <, NoWarning>
                              <, ReturnType = d>
                             )

Parameters:  A  -  an m x n matrix (an array or a Dom::Matrix object)
             b  -  a matrix of dimension m x p (a matrix or 
                   a 2-dimensional array(1..m,1..p).
                   Alternatively, a vector passed as a list
                   or as an array(1..m) 
      hard_soft -  Soft, SoftwareFloats, Hard, HardwareFloats
         method -  Symbolic, QRD or SVD. The default is QRD.
      NoWarning -  no warning is given if matlinsolve switches to
                   the 'symbolic mode' due to symbolic parameters
              d -  return the result as matrices of domain type d,
                   with d = DOM_ARRAY, Dom::DenseMatrix(), Dom::Matrix()

Options:
         Hard, HardwareFloats
                  - use compiled code. If this fails, proceed to 
                    SoftwareFloats issuing a warning.
                    This is the default if DIGITS <= 15
         Soft, SoftwareFloats
                  - do not try hardware floats. Proceed directly to 
                    the MuPAD Code
                    This is the default if DIGITS >= 16
         Symbolic - suppresses internal floating point evaluation
                    of the data. 
                    Calls numeric::matlinsolve(A^H*A, A^H*b, Symbolic)
                    to solve the normal equations A^H*A*x = A^H*b.
                    The most general solution is returned.
         QRD      - use a QR decomposition (default). If the solution
                    is not unique, only a special solution is returned.
         SVD      - use a singular value decomposition. If the solution
                    is not unique, only a special solution is returned.
         NoWarning- suppress warning when automatically switching
                    to the symbolic mode

ReturnValue: a list [X, kernel, residues]:
             X is a least squares solution of the matrix equation
             A*X = b: X is an n x p matrix with each column x[j] of X
             minimizing the residue norm(A*x[j] - b[j], 2) where
             b[j] is the j-th column of b.
             The kernel is either 0 or an n x k matrix (k is the
             dimension of the kernel)

             The kernel is only computed with Symbolic.
             In all other cases, the value NIL is returned.

             residues is a list [r[1], .., r[p]] with
             r[i] = norm(A*x[j] - b[j], 2).

             The value [FAIL, NIL] is returned, if no
             solution was found (this can only happen,
             if the option Symbolic is used and no solution
             of the normal equation was found. This may happen,
             if rank(A) < n and if there are floats in A,
             such that the rank deficiency cannot be recognized
             properly).

Details:
 - With the option Symbolic, the normal equations are solved 
   symbolically via numeric::matlinsolve(A^H*A, A^H*b, Symbolic).

   The minimized residue does not contain the free parameters.

 - With the options QRD and SVD, all input data are converted
   to floats. If there are symbolic objects in A that cannot
   be converted to real or complex floats, then leastSquares
   automatically switches to the Symbolic mode, issuing a warning.
   This warning may be suppressed with the option NoWarning.
  
   Exact numerical input data such as PI, sqrt(2) etc. are
   accepted and converted to floats.

   If rank(A) < n (the least squares solution is not unique),
   then *one* least square solution is returned. The results
   produced with QRD and SVD may differ in this case.

 - With the default method QRD, the normal equations are solved
   by a QR decomposition A = Q*R. If the matrix A is ill conditioned,
   a QR decomposition A^H*A = Q*R is used.

 - With the method SVD, as singular value decomposition A = U*D*V^H
   is computed and the equation D^2*V^H*x = D*U^T*b is solved.
   Singular values d[i] of A satisfying d[i] < spectralradius(A)*10^(-DIGITS)
   are ignored. Corresponding components of V^H*x are set to zero. 

   Usually, the method SVD is slower than QRD by a small factor.

   The method SVD is useful for ill conditioned matrices, where
   it produces least square solutions x of small magnitude norm(x, 2).

   The conditioning is given by the ratio of the largest singular
   value of A divided by the smallest singular value of A.
   If this value is large, the problem is badly conditioned.

Examples:

2.0.0 > A:= array(1..3, 1..2, [[1, 2], [1, 2], [1, 2]]): b:= [3, 4, 5]:
2.0.0 > linalg::rank(matrix(A));

                               1

2.0.0 > numeric::leastSquares(A, b, Symbolic);        

                 -- +-   -+  +-    -+       --
                 |  |  4  |  |  -2  |   1/2  |
                 |  |     |, |      |, 2     |
                 |  |  0  |  |   1  |        |
                 -- +-   -+  +-    -+       --


2.0.0 > numeric::leastSquares(A, b);          

               -- +-     -+                   --
               |  |  0.8  |                    |
               |  |       |, NIL, 1.414213562  |
               |  |  1.6  |                    |
               -- +-     -+                   --

2.0.0 > numeric::leastSquares(A, b, QRD);

             -- +-        -+                   --
             |  | 4.0, 0.0 |, NIL, 1.414213562  |
             -- +-        -+                   --

2.0.0 > numeric::leastSquares(A, b, SVD);

             -- +-        -+                   --
             |  | 0.8, 1.6 |, NIL, 1.414213562  |
             -- +-        -+                   --

An ill conditioned problem: 20 equations for 10 unknowns:

2.0.0 > A:= linalg::hilbert(10):  
2.0.0 > A:= A::dom::stackMatrix(A, A):
2.0.0 > b:= [_plus(A[i,j] $ j=1..10) $ i=1..20]:
2.0.0 > numeric::leastSquares(A, b, Symbolic);          

                      -- +-   -+       --
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |, 0, 0  |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      |  |     |        |
                      |  |  1  |        |
                      -- +-   -+       --

2.0.0 > numeric::leastSquares(A, b, QRD);          

        -- +-              -+                       --
        |  |   1.183683426  |                        |
        |  |                |                        |
        |  |  -3.251258874  |                        |
        |  |                |                        |
        |  |   24.22678125  |                        |
        |  |                |                        |
        |  |  -45.07767881  |                        |
        |  |                |                        |
        |  |   32.82370413  |                        |
        |  |                |, NIL, 0.0001278092636  |
        |  |       0.0      |                        |
        |  |                |                        |
        |  |       0.0      |                        |
        |  |                |                        |
        |  |       0.0      |                        |
        |  |                |                        |
        |  |       0.0      |                        |
        |  |                |                        |
        |  |       0.0      |                        |
        -- +-              -+                       --


2.0.0 > numeric::leastSquares(A, b, SVD);          

      -- +-              -+                           --
      |  |   1.000046018  |                            |
      |  |                |                            |
      |  |  0.9991366422  |                            |
      |  |                |                            |
      |  |   1.003248261  |                            |
      |  |                |                            |
      |  |  0.9976945754  |                            |
      |  |                |                            |
      |  |  0.9972347248  |                            |
      |  |                |, NIL, 0.00000004756918602  |
      |  |  0.9998278968  |                            |
      |  |                |                            |
      |  |   1.002330508  |                            |
      |  |                |                            |
      |  |   1.002954011  |                            |
      |  |                |                            |
      |  |   1.001015802  |                            |
      |  |                |                            |
      |  |   0.996459546  |                            |
      -- +-              -+                           --

[dev] > numeric::leastSquares(A, b) // HardwareFloats

        -- +-              -+                        --
        |  |   1.000000005  |                         |
        |  |                |                         |
        |  |  0.9999995747  |                         |
        |  |                |                         |
        |  |   1.000009388  |                         |
        |  |                |                         |
        |  |  0.9999111854  |                         |
        |  |                |                         |
        |  |   1.000345029  |                         |
        |  |                |, NIL, 0.00007642409558  |
        |  |  0.9989584759  |                         |
        |  |                |                         |
        |  |   1.001817793  |                         |
        |  |                |                         |
        |  |  0.9980940074  |                         |
        |  |                |                         |
        |  |   1.000863247  |                         |
        |  |                |                         |
        |  |  0.9998134458  |                         |
        -- +-              -+                        --
-------------------------------------------------------------------------*/

numeric::leastSquares:= proc(A, b)
local m, n, mn, rb, cb, method, B, c, rreal, dowarn, macheps, 
      x, residues, ATA, ATb, kernel, rankproblem, returnType, 
      useHardwareFloats, HardwareFloatsRequested, result,
      i, j, k, h, s, ss, U, d, V, resU, resV, threshold, xx, 
      Amap, Aop, bmap, bop, jj, Axmb, specialwarning, OK,
      colrangeAchanged, originalcolrangeA,
      colrangebchanged, originalcolrangeb;
begin
  if args(0)<2 then error("expecting at least two arguments") end_if;
  if args(0)>6 then error("expecting no more than 6 arguments") end_if;

  colrangeAchanged:= FALSE;
  colrangebchanged:= FALSE;
  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  if domtype(A) = DOM_ARRAY or
     domtype(A) = DOM_HFARRAY then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     originalcolrangeA:= op(A, [0, 3]);
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       colrangeAchanged:= TRUE:
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
     end_if;
     [m, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= A::dom;
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [m, n]:= A::dom::matdim(A):
     originalcolrangeA:= 1..n:
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
          if A::dom <> Dom::DenseMatrix() then
             A:= Dom::DenseMatrix()(A):
          end_if;
     elif A::dom <> Dom::Matrix() then
          returnType:= Dom::Matrix();
          A:= Dom::Matrix()(A);
     else returnType:= Dom::Matrix();
     end_if;
  else
     error("first argument: expecting an array, an hfarray, or ".
           "a matrix of category 'Cat::Matrix'"):
  end_if;
  if A = FAIL then
     error("first argument: could not convert the input matrix"):
  end_if;

  //----------------------------------------------
  // Check the 'right hand side' b
  //----------------------------------------------
  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(b)) then
       case op(b, [0, 1])
       of 1 do
          originalcolrangeb:= 1..1;
          [rb, cb]:= [op(b, [0, 2, 2]) + 1 - op(b, [0, 2, 1]), 1];
          if domtype(b) = DOM_ARRAY then
             b:= subsop(b, 0  = (2, 1..rb, 1..1));
          else
             b:= hfarray(1..rb, 1..1, [op(b)]);
          end_if;
          break;
       of 2 do
          originalcolrangeb:= op(b, [0, 3]);
          [rb, cb]:= [op(b, [0, 2, 2]) + 1 - op(b, [0, 2, 1]),
                      op(b, [0, 3, 2]) + 1 - op(b, [0, 3, 1])];
          if op(b, [0, 2, 1]) <> 1 or op(b, [0, 3, 1]) <> 1 then
             colrangebchanged:= TRUE:
             if domtype(b) = DOM_ARRAY then
                b:= subsop(b, 0 = (2, 1..rb, 1..cb));
             else
                b:= hfarray(1..rb, 1..cb, [op(b)]);
             end_if;
          end_if;
          break;
       otherwise
          error("second argument: expecting a 1-dimensional array/hfarray ".
                "(a vector) or a 2-dimensional array/hfarray (a matrix)");
       end;
  elif domtype(b) = DOM_LIST then
       b:= map(b, expr);
       [rb, cb]:= [nops(b), 1];
       b:=array(1..rb, 1..cb, [[b[i]]$ i=1..rb]);
       originalcolrangeb:= 1..1;
  elif b::dom::hasProp(Cat::Matrix) = TRUE then
       [rb, cb]:= b::dom::matdim(b):
       originalcolrangeb:= 1..cb;
       if b::dom::constructor = Dom::DenseMatrix then
          if b::dom <> Dom::DenseMatrix() then
             b:= Dom::DenseMatrix()(b):
          end_if;
       elif b::dom <> Dom::Matrix() then
            b:= Dom::Matrix()(b);
       end_if;
  else error("second argument: expecting a list, an array, an hfarray, ".
             "or a matrix of category 'Cat::Matrix'"):
  end_if;
  if b = FAIL then
     error("second argument: could not convert the input matrix/vector"):
  end_if;
  if m <> rb  then
     error("Dimensions of matrices/vectors are not compatible. ".
           "First argument: the matrix has ".  expr2text(m). " rows. ".
           "Second argument: the matrix/vector has ". 
            expr2text(nops(b)). " rows");
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;    // floating point method 'QRD'
  dowarn:= TRUE; // default
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:

  specialwarning:= FAIL;
  for i from 3 to args(0) do
      case args(i)
      of Hard do
      of HardwareFloats do
                HardwareFloatsRequested:= TRUE;
                if method = 1 then
                   specialwarning:= 
                           "ignoring the option 'HardwareFloats' ".
                           "because of the option 'Symbolic'":
                   method:= 1; // symbolic method
                   useHardwareFloats:= FALSE:
                   HardwareFloatsRequested:= FALSE;
                else
                   useHardwareFloats:= TRUE:
                end_if;
                break;
      of Soft do
      of SoftwareFloats do
                useHardwareFloats:= FALSE:
                HardwareFloatsRequested:= FALSE;
                break;
      of Symbolic do
                method:=1; // symbolic method
                useHardwareFloats:= FALSE;
                HardwareFloatsRequested:= FALSE;
                break;
      of QRD do method:= 2:
                break;
      of SVD do method:= 3: 
                break;
      of NoWarning do
                dowarn:= FALSE;
                break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if not has({DOM_ARRAY, 
                          DOM_HFARRAY,
                          Dom::DenseMatrix(),  
                          DOM_HFARRAY,
                          Dom::Matrix()}, 
                          returnType) then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between ".
                       "DOM_ARRAY, ".
                       "DOM_HFARRAY, ".
                       "Dom::Matrix(), ".
                       "or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------

  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("leastSquares") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
         assert(nops(A) = m*n);
         OK:= traperror((
                         B:= hfarray(1..m, 1..n, [op(A)]);
                        )) = 0;
     else
        B:= A;
        OK:= TRUE:
     end_if:
     if (not OK) or
        has(B, RD_NAN) or
        has(B, RD_INF) or
        has(B, RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if:

     if domtype(b) <> DOM_HFARRAY then
         assert(nops(b) = rb*cb);
         OK:= traperror(( c:= hfarray(1..rb, 1..cb, [op(b)]); )) = 0;
     else
        c:= b;
        OK:= TRUE:
     end_if:
     if (not OK) or
        has(c, RD_NAN) or
        has(c, RD_INF) or
        has(c, RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if:

     assert(method = 2 or method = 3);
     if traperror((
           result:= hfa::leastSquares(B, c,
                                      if method = 2 then TRUE else FALSE end_if)
        )) = 0 then
         if has(result, RD_NAN) or
            has(result, RD_INF) or
            has(result, RD_NINF) then
               userinfo(1, "'HardwareFloats' failed"):
               useHardwareFloats:= FALSE;
               break;
         end_if:
     else // proceed to the branch useHardwareFloats = FALSE
       userinfo(1, "'HardwareFloats' failed"):
       useHardwareFloats:= FALSE;
       break;
     end_if;

     //squeeze in a NIL representing the kernel (that was
     //not computed in HardwareFloat mode):
     result:= [result[1], NIL, result[2]]:

  end_case; // of useHardwareFloats = TRUE


  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  case useHardwareFloats 
  of FALSE do
    userinfo(1, "not using hardware floats"):

    //-----------------------------------------
    // convert A to domtype DOM_ARRAY for speed 
    //-----------------------------------------
    if A::dom = DOM_HFARRAY then
       A:= array(1..m, 1..n, [op(A)]);
    end_if;
    if A::dom::hasProp(Cat::Matrix)=TRUE and
       A::dom::constructor = Dom::Matrix then
        Amap:= (Dom::Matrix())::mapNonZeroes;
        Aop:= (Dom::Matrix())::nonZeroOperands;
        A:= Dom::Matrix()(A);
        b:= Dom::Matrix()(b);
   else Amap:= map;
        Aop := op;
        A:= expr(A);
   end_if; 
   if domtype(A) <> Dom::Matrix() then
      if domtype(A) <> DOM_ARRAY then
         error("the matrix must be an array or of category 'Cat::Matrix'");
      end_if:
      if op(A,[0,1]) <> 2 then
         error("the first argument is not a matrix");
      end_if;
      // the documentation refers to A as an m x n matrix. Make sure
      // the error messages are consistent with the help page!
      if op(A,[0,2,1]) <> 1 or op(A,[0,3,1]) <> 1 then
         error("the matrix must be of the format array(1..m, 1..n)");
      end_if:
      if testargs() then // expensive, so use testargs()
         if map({Aop(A)}, testtype, Type::Arithmetical) <> {TRUE} then
            error("the matrix components must be arithmetical expressions");
         end_if;
      end_if;
   end_if;

   //-----------------------------------------
   // convert b to domtype DOM_ARRAY for speed 
   //-----------------------------------------
    if b::dom = DOM_HFARRAY then
       b:= array(1..rb, 1..cb, [op(b)]);
    end_if;
   if b::dom = Dom::Matrix() then
        bmap:= b::dom::mapNonZeroes;
        bop:= b::dom::nonZeroOperands;
        b:= bmap(b, expr);
   else bmap:= map;
        bop:= op;
        b:= expr(b);
   end_if;
   if b::dom <> Dom::Matrix() then
      if domtype(b) <> DOM_ARRAY then
         error("the second argument has an illegal type"):
      end_if:
      if testargs() then // expensive, so use testargs()
         if map({bop(b)}, testtype, Type::Arithmetical) <> {TRUE} then
            error("the vector components must be arithmetical expressions");
         end_if;
      end_if;
   end_if;

   //--------------------------------------------
   // fall back to symbolic mode, if the matrix A
   // contains objects that cannot be floated
   //--------------------------------------------

   if method=2 or method=3 then            
     B:= A: // need copies of
     c:= b: // the original data
     A:=Amap(A, float):
     b:=bmap(b, float):
     if map({Aop(A)}, domtype) minus {DOM_FLOAT, DOM_COMPLEX} <> {} then
        if dowarn then
           warning("symbolic entry found, switching to option Symbolic");
        end_if;
        method:=1;  // fall back to the symbolic mode
        A:= B: // restore the original data before float conversion
        b:= c: // restore the original data before float conversion
     end_if:
   end_if;

   //--------------------------------------------
   // Is the matrix A real?
   //--------------------------------------------
   if map({Aop(A)},iszero@Im) = {TRUE}
     then rreal:=TRUE 
     else rreal:=FALSE
   end_if:

  // --------------------------------------------------------------------
  // symbolic method: solve A^H*A*x = A^H*b 
  // via numeric::matlinsolve(.., Symbolic)
  // --------------------------------------------------------------------
  if method=1 then
    userinfo(1,"symbolic mode, calling numeric::matlinsolve(.., Symbolic)");

    if rreal then 
      if domtype(A) = Dom::Matrix() then
         ATA:= A::dom::transpose(A)*A;
         ATb:= A::dom::transpose(A)*b;
      else
         ATA:= array(1..n, 1..n, 
                     [[_plus(A[k,i]*A[k,j] $ k=1..m) $ j=1..n]
                      $ i=1..n]);
         ATb:= array(1..n, 1..cb,
                     [[_plus(A[k,i]*b[k,j] $ k=1..m) $ j=1..cb]
                      $ i=1..n]);
      end_if;
    else 
      if domtype(A) = Dom::Matrix() then
         ATA:= Amap(A::dom::transpose(A), conjugate)*A;
         ATb:= Amap(A::dom::transpose(A), conjugate)*b;
      else
         ATA:= array(1..n, 1..n, 
                     [[_plus(conjugate(A[k,i])*A[k,j] $ k=1..m) $ j=1..n] 
                     $ i=1..n]);
         ATb:= array(1..n, 1..cb,
                     [[_plus(conjugate(A[k,i])*b[k,j] $ k=1..m) $ j=1..cb]
                      $ i=1..n]);
      end_if;
    end_if;

    [x, kernel]:= numeric::matlinsolve(ATA, ATb, Symbolic):

    if x = FAIL then 
       // This should not happen: A^H*A*x = A^H*b always has a solution.
       // However, it may happen because of round-off,
       // if A or b contain floats
       return([FAIL, NIL, NIL]);
    end_if;

    // compute residues = norm(A*x - b, 2).
    residues:= [0 $ cb];
    if domtype(A) = Dom::Matrix() then
         Axmb:= A*x - b;
         for j from 1 to cb do
           residues[j]:= _plus(abs(Axmb[i,j])^2 $ i=1..m)^(1/2);
         end_for;
    else
         for j from 1 to cb do
            residues[j]:= _plus(abs(_plus(A[i,k]*x[k,j] $k=1..n)-b[i,j])^2 
                                $i=1..m)^(1/2):
         end_for;
    end_if;

    result:= [x, kernel, residues];
  end_if;
  // --------------------------------------------------------------------
  // end of Symbolic mode
  // --------------------------------------------------------------------

  // Above, input matrices of type Dom::Matrix() still survived,
  // because in the symbolic mode sparse multiplication A^T*A, A^T*b
  // was used. The code down below presently does not use any sparsity.
  // Hence, it is time to convert Matrix -> DOM_ARRAY to speed
  // up the code below.

  if A::dom::hasProp(Cat::Matrix)=TRUE then
     A:= expr(A);
  end_if;

  // --------------------------------------------------------------------
  // numerical method 2: solve A^H*A*X = A^H*b via QR factorization
  // A = QR. The resulting equation R^H*R*x = R^H*Q^H*b is solved
  // by backsubstitution of the upper part of R*x = Q^H*b.
  // --------------------------------------------------------------------
  macheps := 10.0^(-DIGITS); // machine precision
  if method=2 then            

    userinfo(1,"numerical mode, computing a QR factorization ".
               "via Householder transformations");


    mn:= min(n,m);
    if n > m then
         rankproblem:= TRUE; // we definitely know that rank(A) < n
    else rankproblem:= FALSE;
    end_if;
 
    if not rankproblem then
      // Extend the j loop to mn, not just to mn - 1!
      // In principle, it would suffice to run through mn - 1
      // to transform A to upper triangular form, but this
      // more stable.

      for j from 1 to mn do
  
        // Householder step from the left to obtain upper   
        // triangular form. Householder data are stored in  
        // lower triangular part of A.                      
  
        s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..m);
        if not iszero(s) then
          ss:=s+specfunc::abs(A[j,j])^2;
          s:=specfunc::sqrt(ss):
          h := 1 / (ss + specfunc::abs(A[j,j]) * s);
          if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
          A[j,j]:=s+A[j,j];
          ss:= s; // save s to reassign A[j, j]:=-s below.
          for i from j+1 to n do
            if rreal then  s:=_plus(A[k,j] *A[k,i]$ k=j..m);
            else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..m);
            end_if;
            s:=s*h;
            (A[k,i]:=A[k,i]-s*A[k,j]) $ k=j..m ;
          end_for;
  
          for jj from 1 to cb do
            // Householder step for right hand side of A*x = b
            if rreal then  s:=_plus(A[k,j] *b[k, jj]$ k=j..m);
            else s:=_plus(conjugate(A[k,j])*b[k, jj]$ k=j..m);
            end_if;
            s:=s*h;
            for k from j to m do
               b[k, jj]:= b[k, jj]-s*A[k,j]
            end_for;
          end_for;
          A[j, j]:= - ss;
        end_if; // of not iszero(s)
      end_for;

      //---------------------------------------------------------
      // check whether rank(A) < n, or if
      // the matrix was ill conditioned:
      //---------------------------------------------------------
      threshold:= max(specfunc::abs(A[i, i]) $ i=1..mn)*macheps;
      for i from 1 to mn do
        if specfunc::abs(A[i,i]) <= threshold then
           rankproblem:= TRUE:
           break;
        end_if;
      end_for:
    end_if; // of if not rankproblem

    //---------------------------------------------------------
    // Backsubstitution for R*x = Q^H*b
    //---------------------------------------------------------
    if not rankproblem then

      userinfo(1,"matrix is well conditioned, proceed ..."):

      // Now, A = Q*R, where R is stored in the upper diagonal
      // part of A. The vector Q^H*b is stored in b.
      // To solve A^H*A*x = A^H*b --> R^H*R*x = R^H*Q^H*b one
      // only has to solve R*x = Q^H*b, restricted to the upper
      // mn elements of the equations, provided R is not singular.
  
      // Backsubstitution for R*x = Q^H*b
      x:= array(1..n, 1..cb, [[float(0) $ cb] $ n]):
      for i from mn downto 1 do
        if not iszero(A[i,i]) then 
           for j from 1 to cb do
            x[i, j] := (b[i, j] - _plus(A[i,k]*x[k, j] $ k = i+1..mn)) / A[i,i];
           end_for;
        else return([FAIL, NIL, NIL]): // should not happen
        end_if;
      end_for: 

      // -----------------------------------------------------
      // compute residues = norm(A*x - b, 2). The original data
      // are stored in B and c:
      // -----------------------------------------------------
      A:= map(B, float): // the original m x n matrix
      b:= map(c, float): // the original m vector
      residues:= [0 $ cb];
      for j from 1 to cb do
         residues[j]:= _plus(specfunc::abs(_plus(A[i,k]*x[k,j] $k=1..n)-b[i,j])^2 $i=1..m)^(1/2):
      end_for:
      result:= [x, NIL, residues];
    end_if; // of if not rankproblem
    
    //--------------------------------------------------------
    //--------------------------------------------------------
    if rankproblem then
       userinfo(1,"matrix is ill conditioned: rank < column dimension"):

       // Instead of solving A^H*A*x = A^H*b we tried to solve
       // A*x = b by solving the first m equations of R*x = Q^H*b 
       // (where A=Q*R). It turned out that rank(R) < m. 
       // Now, we try to solve R^H*R*x = R^H*Q^H*b:

       ATA:= array(1..n,1..n):
       ATb:= array(1..n,1..cb):
       if rreal then
          if n <= m then // A was replaced by the upper tringular R above
             for i from 1 to n do
               for j from 1 to n do
                 ATA[i,j]:= _plus(A[k,i]*A[k,j] $ k=1..min(i,j));
               end_for:
             end_for:
             for i from 1 to n do
               for j from 1 to cb do
                 ATb[i, j]:= _plus(A[k, i]*b[k, j] $ k=1..i):
               end_for:
             end_for:
          else // A is still the original m x n matrix
             for i from 1 to n do
               for j from 1 to n do
                 ATA[i,j]:= _plus(A[k,i]*A[k,j] $ k=1..m);
               end_for:
             end_for:
             for i from 1 to n do
               for j from 1 to cb do
                 ATb[i, j]:= _plus(A[k, i]*b[k, j] $ k=1..m):
               end_for:
             end_for:
        end_if;
       else 
        if n <= m then // A was replaced by the upper tringular R above
             for i from 1 to n do
               for j from 1 to n do
                 ATA[i,j]:= _plus(conjugate(A[k,i])*A[k,j] $ k=1..min(i,j));
               end_for:
             end_for:
             for i from 1 to n do
               for j from 1 to cb do
                 ATb[i, j]:= _plus(conjugate(A[k, i])*b[k, j] $ k=1..i):
               end_for:
             end_for:
        else // A is still the original m x n matrix
             for i from 1 to n do
               for j from 1 to n do
                 ATA[i,j]:= _plus(A[k,i]*A[k,j] $ k=1..m);
               end_for:
             end_for:
             for i from 1 to n do
               for j from 1 to cb do
                 ATb[i, j]:= _plus(A[k, i]*b[k, j] $ k=1..m):
               end_for:
             end_for:
        end_if;
       end_if; // of if rreal

       // Solve R^H*R*x = R^H*Q^H*b by a QR factorization of R^H*R:
       
       A := ATA:
       b := ATb:
       for j from 1 to n do

        // Householder step from the left to obtain upper   
        // triangular form. Householder data are stored in  
        // lower triangular part of A.                      

        s:= _plus(specfunc::abs(A[i,j])^2 $ i=j+1..n);
        if not iszero(s) then
          ss:=s+specfunc::abs(A[j,j])^2;
          s:=specfunc::sqrt(ss):
          h := 1 / (ss + specfunc::abs(A[j,j]) * s);
          if not iszero(A[j,j]) then s:=sign(A[j,j])*s;end_if:
          A[j,j]:=s+A[j,j];
          ss:= s; // save s to reassign A[j, j]:=-s below.
          for i from j+1 to n do
            if rreal then  s:=_plus(A[k,j] *A[k,i]$ k=j..n);
            else s:=_plus(conjugate(A[k,j])*A[k,i]$ k=j..n);
            end_if;
            s:=s*h;
            (A[k,i]:=A[k,i]-s*A[k,j]) $ k=j..n ;
          end_for;
  
          // Householder step for right hand side of A*x = b
          for jj from 1 to cb do
            if rreal then  s:=_plus(A[k,j] *b[k, jj]$ k=j..n);
            else s:=_plus(conjugate(A[k,j])*b[k, jj]$ k=j..n);
            end_if;
            s:=s*h;
            for k from j to n do
             b[k, jj]:= b[k, jj]-s*A[k,j]
            end_for;
          end_for;

          A[j, j]:= - ss;
        end_if; // of not iszero(s)
      end_for;

      // -----------------------------------------------------
      // Backsubstitution for R*x = Q^H*b
      // Ignore all equations with small pivots: these are
      // likely to be pure round-off. Assuming they correspond
      // to kernel vectors, we just put x[i, j] := 0.0.
      // -----------------------------------------------------
      x:= array(1..n, 1..cb, [[float(0) $ cb] $ n]):
      threshold:= max(specfunc::abs(A[i, i]) $ i=1..n)*macheps;
      for i from n downto 1 do
        if specfunc::abs(A[i,i]) > threshold then 
           for j from 1 to cb do
                x[i, j] := (b[i, j] - _plus(A[i,k]*x[k, j] $ k = i+1..n)) / A[i,i];
           end_for;
        else 
           for j from 1 to cb do
             x[i, j] := float(0):
           end_for;
        end_if;
      end_for: 
  
      // -----------------------------------------------------
      // compute residues = norm(A*x - b, 2). The original data
      // are stored in B and c:
      // -----------------------------------------------------
      A:= map(B, float): // the original m x n matrix
      b:= map(c, float): // the original m vector
      residues:= [0 $ cb];
      for j from 1 to cb do
        residues[j]:= _plus(specfunc::abs(_plus(A[i,k]*x[k,j] $k=1..n)-b[i,j])^2 $i=1..m)^(1/2);
      end_for:

      result:= [x, NIL, residues];
    end_if: // of if rankproblem

  end_if; // of method = 2 (float method QRD)
  // --------------------------------------------------------------------
  // end of QRD mode
  // --------------------------------------------------------------------


  if method=3 then
  // --------------------------------------------------------------------
  // numerical method 3 (SVD): solve A^H*A*X = A^H*b via singular value
  // decomposition A = U D V^H, where D = diag(d1, d2, ..) with the
  // singular values d1, d2, .. of A. The resulting equation
  // V*D*(D*V^H*x - U^H*b) = 0 is solved by solving D*V^H*x = U^H*b.
  // If small singular values are encountered, the corresponding
  // equations are ignored (these singular values are set to zero).
  // --------------------------------------------------------------------

    userinfo(1,"numerical mode, computing a singular value decomposition"):

    // --------------------------------------------------- 
    // This version computes U. This is rather expensive:
    //
    // [U,d,V,resU,resV] := numeric::singularvectors(A, NoErrors);
    // x:= array(1..m, 1..cb):
    // // b ->  x = U^H*b
    // for i from 1 to n do
    //  for j from 1 to cb do
    //   x[i, j] := _plus(U[k, i]*b[k, j] $ k=1..m);
    //  end_for:
    // end_for:
    // ---------------------------------------------------
  
    // Instead, use NoLeftVectors to avoid computing U:
    [U,d,V,resU,resV] := numeric::singularvectors(A, NoLeftVectors, NoErrors);

    // We have to contruct U^H*b = D^(-1)*V^H*V*D*U^H*b = D^(-1)*V^H* A^H*b.
    // Computing U via a SVD is rather expensive (U is of dimension m x m,
    // where, typically, m >> n). Instead, we use D*U^H*b = V^H*A^H*b:

    // compute x = A^H*b
    // Initialize container x = array(1..n) with zero entries.
    // If m < n, the initialized zeroes x[i, j] = 0.0, i = m+1,..n
    // will survive the following:
    x:= array(1..n, 1..cb, [[float(0) $ cb] $ n]);
    if rreal then
         for i from 1 to n do
          for j from 1 to cb do
           x[i, j] := _plus(A[k, i]*b[k, j] $ k=1..m);
          end_for:
         end_for:
    else for i from 1 to n do
          for j from 1 to cb do
           x[i, j] := _plus(conjugate(A[k, i])*b[k, j] $ k=1..m);
          end_for:
         end_for:
    end_if:

    // -----------------------
    // compute  x = V^H*A^H*b
    // -----------------------
    xx:= x;
    for i from 1 to n do
     for j from 1 to cb do
      x[i, j] := _plus(conjugate(V[k, i])*xx[k, j] $ k=1..n);
     end_for:
    end_for:

    // ---------------------------------------------------
    // Edit the singular values stored in the array d: 
    // all singularvalues < 10^(-DIGITS/2)*spectralradius 
    // are set to 0.0.
    // The spectral radius is the maximal singular value:
    // ---------------------------------------------------

    threshold := max(op(d))*10^2*macheps;
    d:= map(d, x -> if x <= threshold then float(0) else x end_if);

    // ---------------------------------------------------
    // Start to solve D^2*V^H*x = D*U^H*b. D*U^H*b is stored in x.
    // x -> D^(-1)*x; store the result in x:
    // ---------------------------------------------------
    for i from 1 to min(n,m) do
      if iszero(d[i]) then
           // We have to solve D*D*x = D*U^H*b.
           // If a singular value is zero,
           // the corresponding component of the solution
           // is an arbitrary parameter. Set it to zero:
           for j from 1 to cb do
             x[i, j]:= float(0);
           end_for:
      else 
           for j from 1 to cb do
              x[i, j]:= x[i, j]/d[i]^2;
           end_for:
      end_if;
    end_for:
    // ---------------------------------------------------
    // x -> V*x; store the result in x:
    // ---------------------------------------------------
    xx := x:
    for i from 1 to n do
      for j from 1 to cb do
        x[i, j] := _plus(V[i, k]*xx[k, j] $ k=1..n);
      end_for:
    end_for:

    // ---------------------------------------------------
    // compute residues = norm(A*x - b, 2).
    // ---------------------------------------------------
    residues:= [0 $ cb];
    for j from 1 to cb do
      residues[j]:= _plus(specfunc::abs(_plus(A[i,k]*x[k,j] $k=1..n)-b[i,j])^2 $i=1..m)^(1/2):
    end_for:

    result:= [x, NIL, residues];

  end_if; // of method = 3 (float method via SVD)
  // --------------------------------------------------------------------
  // end of SVD mode
  // --------------------------------------------------------------------

  end_case; // end of case useHardwareFloats 

  //-------------------------
  //------- return ----------
  //-------------------------
  if domtype(result[1]) = returnType and
    (not colrangeAchanged) and
    (result[2] = 0 or
     result[2] = NIL or
     (domtype(result[2]) = returnType and
      not colrangebchanged)
    ) then
     return(result);
  end_if:
  case returnType
  of Dom::Matrix() do
       if result[1] <> FAIL then
          result[1]:= returnType(result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType(result[2]);
       end_if;
       break;
  of Dom::DenseMatrix() do
       if result[1] <> FAIL then
          result[1]:= returnType::create(result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType::create(result[2]);
       end_if;
       break;
  of DOM_ARRAY do
       if domtype(result[1]) = DOM_ARRAY then
          result[1]:= subsop(result[1], 0 = (2, originalcolrangeA, originalcolrangeb));
       elif domtype(result[1]) = DOM_HFARRAY then
          result[1]:= array(originalcolrangeA, originalcolrangeb, [op(result[1])]);
       end_if;
       if domtype(result[2]) = DOM_ARRAY then
          result[2]:= subsop(result[2], 0 = (2, originalcolrangeA, op(result[2], [0, 3])));
       elif domtype(result[2]) = DOM_HFARRAY then
          result[2]:= array(originalcolrangeA, op(result[2], [0, 3]), [op(result[2])]);
       end_if;
       break;
  of DOM_HFARRAY do
       result[1]:= hfarray(originalcolrangeA, originalcolrangeb, [op(result[1])]);
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= hfarray(originalcolrangeA, op(result[2], [0, 3]), [op(result[2])]);
       end_if:
       break;
  otherwise
       error("unexpected return type"):
  end_case;

  return(result):
end_proc:

// end of file
