/* ---------------------------------------------------------------
Utility: 

   the following string represents scilab code for solving
   the linear system A*x = b with a *dense* matrix A.
   This scilab code is used by numeric::linsolve and 
   numeric::matlinsolve, calling it in the form

    f:= scilab::func(["A", "b"],
                     ["x", "kern", "kernindices", "info"],
                     numeric::linsolveSciDense);
    [x, kern, kernindices, Info] 
       = f(A, b, ReturnTypes = [..., ..., Default, Default]);

    On input of arbitrary matrices A (m x na) and b (m x nb),
    it returns a special solution x (na x nb)  of A*x = b
    together with the kernel (na x kerneldim) of A.

    kernindices is a list of the non-characteristic column 
    indices of A.

    the string Info is
    "EmptyKernel"  (there is a unique solution, the kernel is empty)
    "OK",   (there is a solution, the kernel is not empty)
    "FAIL"  (there is no solution)

----------------------------------------------------------------*/

numeric::linsolveSciDense:=
["function [x, kern, kernindices, info] = mylinsolveDense(A,b); ".
   "[m, n] = size(A); ".
   "[bm, bn] = size(b); ".
   "x = 0; ".  // important: proper initialization 
   "kern = 0; ".  // important: proper initialization
   "kernindices = list(); ". // important: proper initialization
   "info = 'OK';". // return "OK", "EmptyKernel" or "FAIL"

   // Normalize the rows to 1. This is essential, because
   // scilab has a bizarre way of 'cleaning' elements that
   // are much smaller than the max entry in the matrix.
   // If inconsistency of the equations 0 = b can be detected
   // on input, return an immediate info = "FAIL":
   "A = A'; ".   // transpose and normalize the columns.
   "b = b'; ".   // This is faster than working row-wise
   "for i = 1:m, ".
   "  nn = norm(A(:, i)); ".
   "  if nn <> 0 then ".
   "     A(:, i) = A(:, i)/nn; ".
   "     b(:, i) = b(:, i)/nn; ".
   "  else ".   // norm(A:, i) = 0
   "     if norm(b(:, i)) <> 0 then ".
   "       info = 'FAIL'; ".
   "       return; ".
   "     end; ".
   "  end; ".
   "end; ".
   "A = A'; ".
   "b = b'; ".

   // compute the row reduced echelon form (rref) of A | 1
   "if type(A) == 5 then ".  // A is a sparse matrix
   "   A = full(A); ".
   "end; ".
   "A = rref([A, eye(m, m)]); ".
   "L = A(:,n + 1:n + m); ".
   "A = A(:,1:n); ".
   "x = L*b; ".  // x = the rhs of the new equation A*x = L*b.
   // It still needs to be padded with zeroes
   // or entries must be deleted ('Entzerrung').
   // This turns x into the special solution 
   // (because, up to kernel columns, A = rref(A) is the identity).

   // Find the kernel columns by looking for 
   // non-characteristic column indices:
   "kernindex = 1; ".
   "i = 1; ".  // consider row i
   "for j = 1:n, ". // consider column j
   "  if i <= m then ".
   "    if A(i, j) <> 0 then ".
   "      i = i + 1; ".
   "    else ".
   "      kernindices(kernindex) = j;  ".
   "      kernindex = kernindex + 1; ".
   "    end; ".
   "  else ".
   "    kernindices(kernindex) = j;  ".
   "    kernindex = kernindex + 1; ".
   "  end; ".
   "end; ".

   // check if the system is solvable:
   // i is the index where the zero rows 
   // in rref(A) begin:
   "xmax = max(abs(x))/10^10; ".
   "for k = i:m, ".
   "  if norm(x(i:i, 1:bn)) > xmax then ".
   "    info = 'FAIL'; ".
   "    return; ".
   "  end; ".
   "end; ". 

   // delete the zeros that were inserted
   // in the beginning:
   "if m > n then ".
   "  x = x(1:n,1:bn); ".
   "end; ".
   "if n > m then ".
   "  x = [x; zeros(n - m, bn)]; ".
   "end; ".

   // check the kernel
   "kdim = size(kernindices); ".
   "if kdim == 0 then ".
   "  info = 'EmptyKernel'; ".
   "  kern = 0; ".
   "  return; ".
   "end; ".

   // Entzerre A and rhs x:
   "if m > n then ".
   "  A = A(1:n,1:n); ".
   "end; ".
   "if n > m then ".
   "  A = [A; zeros(n - m, n)]; ".
   "end; ".
   "A = A'; ". // entzerre A
   "x = x'; ". // entzerre rhs x
   "for j = 1:kdim, ".
   "  i = kernindices(j); ".
   "  for k = 0:(n - i - 1), ".
   "    A(:,n-k) = A(:,n-k-1); ".
   "    x(:,n-k) = x(:,n-k-1); ".
   "  end; ".
   "  A(:,i) = zeros(n, 1); ".
   "  A(i,i) = -1; ".
   "  x(:,i) = zeros(bn, 1); ".
   "end; ".
   "A = A'; ".
   "x = x'; ".
   "kern = zeros(n, kdim); ".
   "for j = 1:kdim, ".
   "  kern(:, j) = -A(:, kernindices(j)); ".
   "end; ".
 "endfunction;".

 "[x, kern, kernindices, info] = mylinsolveDense(A, b);"
]:
