//       
// 
// W.Oevel, 01/04/98 

//------   file  matlinsolve.mu   1.4.98  ------------------------
//-------------------  W. Oevel  ---------------------------------
/*  Help file:

 matlinsolve - solve a system of linear equations numerically

 Call:

 numeric::matlinsolve(A, B 
                      <hard_soft>, 
                      <Sparse>,
                      <Symbolic>, 
                      <ShowAssumptions>,
                      <NoWarning>,
                      <ReturnType = d>
                     )
                                                                   
 Parameter:

  A          --  an m x n matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
  B          --  an m x p matrix (DOM_ARRAY, DOM_HFARRAY, or category Cat::Matrix)
                 For p=1 the vector B may also be a 1-dimensional array/hfarray
                 or a list of length m.
  hard_soft  --  Soft, SoftwareFloats, Hard, HardwareFloats
  Symbolic   --  do not convert input data to floats
  NoWarning  --  no warning is given if matlinsolve switches to
                 the 'symbolic mode' due to symbolic parameters
  "dont_use_normal" -- undocument option to suppress the use of normal
  d          --  return the result as matrices of domain type d,
                 with d = DOM_ARRAY, Dom::DenseMatrix(), Dom::Matrix()
                                                                   
Synopsis:

  numeric::matlinsolve(A,B,<Symbolic>); 
      returns the list [X,Kernelbasis].

  numeric::matlinsolve(A,B,<Symbolic>, ShowAssumptions);
      returns the list [X,Kernelbasis,Constraints, PivotAssumptions].

  (*) X is a special solution of the linear system A*X=B. It is
      an n x p array.
  (*) Kernelbasis represents the general solution of the homogeneous
      linear system A*X=0. It is an n x d array,
      where d is the dimension of the kernel of A. The columns of 
      Kernelbasis form a basis of the kernel. If the kernel is empty,
      then the returned value of Kernelbasis is the null vector,
      represented by array(1..n,1..1,[[0]$n]).
  (*) Constraints is a list of equations for symbolic parameters in A and B,
      which are necessary and sufficient to grant solvability of 
      the linear system A*X=B. If no such contraints arise, then the
      returned value of Constraints is the empty list.
  (*) PivotAssumptions is a list of inequalities [piv1 <>0, piv2<>0, ...].
      These assumptions were implicitly used by matlinsolve to compute
      the solution and the kernel.

  By default, all entries in A and B are converted to floats. Non-numeric
  symbolic entries in A are not admissible, unless the option Symbolic
  is used. In the symbolic mode no conversion to floats will occur. We note 
  that the presence of symbolic parameters in the matrix A will slow down 
  the computation drastically.

  Due to roundoff errors, some or all basis vectors in the kernel of A may be
  missed in the numeric mode.

  matlinsolve returns [FAIL,NIL], if no solution of A*X=B exists. If B contains
  symbolic parameters, then Constraints is the set of linear equations
  for the entries of B needed to grant solvability.

  matlinsolve uses Gaussian Elimination with row pivoting. Internally
  a sparse representation of the matrices is used.
  
  Examples: 

>> A:=array(1..2,1..2,[[1,2],[3,4]]): B:=[3,7]:
>> numeric::matlinsolve(A,B);

                       -- +-     -+  +-   -+ --
                       |  |  1.0  |  |  0  |  |
                       |  |       |, |     |  |
                       |  |  1.0  |  |  0  |  |
                       -- +-     -+  +-   -+ --

>> numeric::matlinsolve(A,B,Symbolic);
 
                    -- +-   -+  +-   -+ --
                    |  |  1  |  |  0  |  |
                    |  |     |, |     |  |
                    |  |  1  |  |  0  |  |
                    -- +-   -+  +-   -+ --

>> A:=array(1..3,1..3,[ [1,1,0],[0,1,1],[0,0,1]]):
>> B:=array(1..3,1..3,[ [1,0,0],[0,1,0],[0,0,1]]):
>> InverseOfA:=numeric::matlinsolve(A,B,Symbolic)[1];

                              +-           -+
                              |  1, -1,  1  |
                              |             |
                              |  0,  1, -1  |
                              |             |
                              |  0,  0,  1  |
                              +-           -+

>> A:=Dom::Matrix()(3,3,[[2,2,3],[1,1,2],[3,3,5]]):
>> B:=Dom::Matrix()(3,1,[sin(y)^2,cos(y)^2,x]):
>> ([X,Kernel,Constraints,PivAssupmt]):=
       numeric::matlinsolve(A,B,Symbolic,ShowAssumptions);

-- +-                 -+
|  |          2        |  +-    -+
|  |  5 sin(y)  - 3 x  |  |  -1  |
|  |                   |  |      |         2             2
|  |         0         |, |   1  |, [cos(y)  - x + sin(y)  = 0]
|  |                   |  |      |
|  |                2  |  |   0  |
|  |  2 x - 3 sin(y)   |  +-    -+
-- +-                 -+

        --
         |
         |
         |
   , []  |
         |
         |
         |
        --

>> check_the_result:= A*Dom::Matrix()(X)-B,
                      A*Dom::Matrix()(Kernel); 

                  +-                       -+
                  |           0             |  +-   -+
                  |                         |  |  0  |
                  |            2         2  |  |     |
                  |  x - cos(y)  - sin(y)   |, |  0  |
                  |                         |  |     |
                  |           0             |  |  0  |
                  +-                       -+  +-   -+

 See also:  linalg::matlinsolve, linsolve, numeric::linsolve
                                                                  */
//-----------------------------------------------------------------

numeric::matlinsolve:= proc(A,b)
local returnType, rb, useHardwareFloats, HardwareFloatsRequested,
      AA, bb, OK, sparsemode, specialwarning, furtherargs,
      m, neqs,n,method,coefftypes,use_normal,i,j,tmp,B,cb,check,
      dont_use_normal_requested, normal2,
      Amap, bmap, Aop, L, list, showassumptions, PivotAssumptions,dowarn,
      x,X,X0,Xvars,y,nTerms,koeff,np2,np2mj,p,kdim,Kernel,
      piv,pivot,pivindex,rownorm,constraints,pnorm,Bnorm,
      lowerbands,jpl,minlength, count, result, triv, kernindex,
      colrangeAchanged, originalcolrangeA,
      colrangebchanged, originalcolrangeb;
begin
  if args(0)<2 then error("expecting at least two arguments"); end_if;
  if args(0)>8 then error("expecting no more than eight arguments"); end_if;

  colrangeAchanged:= FALSE;
  colrangebchanged:= FALSE;
  sparsemode:= FALSE;
  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(A)) then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array or hfarray");
     end_if;
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       colrangeAchanged:= TRUE:
       originalcolrangeA:= op(A, [0, 3]);
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
          A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                      1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if:
       [neqs, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     else
       [neqs, n]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
       originalcolrangeA:= 1..n;
     end_if;
     returnType:= domtype(A);
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [neqs, n]:= A::dom::matdim(A):
     originalcolrangeA:= 1..n;
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("first argument: expecting an array, an hfarray, or ".
           "a matrix of category 'Cat::Matrix'"):
  end_if;

  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(b)) then
       case op(b, [0, 1])
       of 1 do 
          [rb, cb]:= [op(b, [0, 2, 2]) + 1 - op(b, [0, 2, 1]), 1];
          if domtype(b) = DOM_ARRAY then
             b:= subsop(b, 0  = (2, 1..rb, 1..1));
          else
             b:= hfarray(1..rb, 1..1, [op(b)]);
          end_if:
          break;
       of 2 do
          [rb, cb]:= [op(b, [0, 2, 2]) + 1 - op(b, [0, 2, 1]), 
                      op(b, [0, 3, 2]) + 1 - op(b, [0, 3, 1])];
          if op(b, [0, 2, 1]) <> 1 or op(b, [0, 3, 1]) <> 1 then
             colrangebchanged:= TRUE:
             originalcolrangeb:= op(b, [0, 3]);
             if domtype(b) = DOM_ARRAY then
                b:= subsop(b, 0 = (2, 1..rb, 1..cb));
             else
                b:= hfarray(1..rb, 1..cb, [op(b)]);
             end_if;
          end_if;
          break;
       otherwise
          error("second argument: expecting a 1-dimensional array/hfarray ".
                "(a vector) or a 2-dimensional array/hfarray (a matrix)");
       end;
  elif domtype(b) = DOM_LIST then
       [rb, cb]:= [nops(b), 1];
       b:=array(1 .. rb, 1..1,[[b[i]]$ i=1..rb]);
  elif b::dom::hasProp(Cat::Matrix) = TRUE then
       [rb, cb]:= b::dom::matdim(b):
       if b::dom <> Dom::DenseMatrix() then
            b:= Dom::DenseMatrix()(b);
       end_if;
  else error("second argument: expecting a list, an array, an hfarray, or ".
             "a matrix of category 'Cat::Matrix'"):
  end_if;

  if neqs <> rb  then 
     error("dimensions of matrices/vectors not compatible"); 
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  method:= 2;        // floating point method
  showassumptions:= FALSE; // default
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  dowarn:= TRUE; // default
  specialwarning:= FAIL;
  dont_use_normal_requested:= FALSE;

  for i from 3 to args(0) do
      case args(i)
      of Hard do
      of HardwareFloats do
                     HardwareFloatsRequested:= TRUE;
                     if method = 1 then
                        specialwarning:= 
                                "ignoring the option 'HardwareFloats' ".
                                "because of the option 'Symbolic'":
                        method:= 1; // symbolic method
                        useHardwareFloats:= FALSE:
                        HardwareFloatsRequested:= FALSE;
                     else
                        useHardwareFloats:= TRUE:
                     end_if;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of Symbolic do method:=1; // symbolic method
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     break;
      of ShowAssumptions do
                     useHardwareFloats:= FALSE;
                     HardwareFloatsRequested:= FALSE;
                     showassumptions:= TRUE; 
                     break;
      of NoWarning do 
                     dowarn:= FALSE;
                     break;
      of Sparse do   sparsemode:= TRUE;
                     break;
      of "dont_use_normal" do
                     dont_use_normal_requested:= TRUE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if not has({DOM_ARRAY, 
                          DOM_HFARRAY,
                          Dom::DenseMatrix(), 
                          Dom::Matrix()}, 
                          returnType) then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between ".
                       "DOM_ARRAY, ".
                       "DOM_HFARRAY, ".
                       "Dom::Matrix(), ".
                       "or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:
  if specialwarning <> FAIL and dowarn then
     warning(specialwarning);
  end_if:

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if sparsemode then 

        if not numeric::startHardwareFloats("sparse_matlinsolve") then
           userinfo(1, "cannot start the hardware floats interface"):
           if HardwareFloatsRequested and dowarn then
              warning("cannot start the 'HardwareFloats' interface, ".
                      "using 'SoftwareFloats' instead");
           end_if;
           useHardwareFloats:= FALSE;
           break;
        end_if;

        userinfo(1, "trying hardware floats"):

        if domtype(A) = matrix then
           AA:= map(extop(A::dom::mapNonZeroes(A, float), 3), poly2list);
        elif domtype(A) = densematrix then
           AA:= map(expr(A), float);
           AA:= [[[AA[i,j], i] $ i = 1..neqs] $ j = 1..n]:
        elif domtype(A) = DOM_HFARRAY then
           AA:= A;
        elif domtype(A) = DOM_ARRAY then
           AA:= map(A, float);
           // convert matrix to a sparse list of columns
           AA:= [[[AA[i,j], i] $ i = 1..neqs] $ j = 1..n]:
        else // should not arrive here
           AA:= FAIL;
        end_if:

        if domtype(b) = matrix then
           bb:= map(extop(b::dom::mapNonZeroes(b, float), 3), poly2list);
        elif domtype(b) = densematrix then
           bb:= map(expr(b), float);
           bb:= [[[bb[i,j], i] $ i = 1..rb] $ j = 1..cb]:
        elif domtype(b) = DOM_HFARRAY then
           bb:= b;
        elif domtype(b) = DOM_ARRAY then
           bb:= map(b, float);
           // convert matrix to a sparse list of columns
           bb:= [[[bb[i,j], i] $ i = 1..rb] $ j = 1..cb]:
        else // should not arrive here
           bb:= FAIL;
        end_if:

        /* The legal input types for the call
              hfa::sparse_matlinsolve(AA, bb)
           are:
              hfa::sparse_matlinsolve(HFA, HFA)
              hfa::sparse_matlinsolve(HFA, List, m, n)
              hfa::sparse_matlinsolve(List, HFA, m, n)
              hfa::sparse_matlinsolve(List1, List2, m, n1, n2)
           
           wher
          (List = [ col1, col2, ..., col.n ] mit col.j = [A[1,j], .. , A[m, j]]
        */

        case [domtype(AA), domtype(bb)] 
        of [DOM_HFARRAY, DOM_HFARRAY] do
           furtherargs:= null():
           break;
        of [DOM_LIST, DOM_HFARRAY] do
           furtherargs:= neqs, n:
           break;
        of [DOM_HFARRAY, DOM_LIST] do
           furtherargs:= neqs, cb:
           break;
        of [DOM_LIST, DOM_LIST] do
           furtherargs:= neqs, n, cb:
           break;
        otherwise
           error("unexpected case");
        end_case;

        if traperror((
              result:= hfa::sparse_matlinsolve(AA, bb, furtherargs)
           )) = 0 then
            if has(result, RD_NAN) or
               has(result, RD_INF) or
               has(result, RD_NINF) then
                  userinfo(1, "'HardwareFloats' failed"):
                  useHardwareFloats:= FALSE;
                  break;
            end_if:
        else // proceed to the branch useHardwareFloats = FALSE
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
        end_if;
        // The kernel dimension kdim may be needed at the
        // end of this routine for conversion to other types
        kdim:= op(result[2], [0, 3, 2]);

     else // sparsemode = FALSE

        if not numeric::startHardwareFloats("matlinsolve") then
           userinfo(1, "cannot start the hardware floats interface"):
           if HardwareFloatsRequested and dowarn then
              warning("cannot start the 'HardwareFloats' interface, ".
                      "using 'SoftwareFloats' instead");
           end_if;
           useHardwareFloats:= FALSE;
           break;
        end_if;

        userinfo(1, "trying hardware floats"):

        if domtype(A) <> DOM_HFARRAY then
            assert(nops(A) = neqs*n);
            OK:= traperror((
                            AA:= hfarray(1..neqs, 1..n, [op(A)]);
                           )) = 0;
        else
           AA:= A;
           OK:= TRUE:
        end_if:
        if (not OK) or
           has(AA, RD_NAN) or
           has(AA, RD_INF) or
           has(AA, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
        end_if:

        if domtype(b) <> DOM_HFARRAY then
            assert(nops(b) = rb*cb);
            OK:= traperror((
                            bb:= hfarray(1..rb, 1..cb, [op(b)]);
                           )) = 0;
        else
           bb:= b;
           OK:= TRUE:
        end_if:
        if (not OK) or
           has(bb, RD_NAN) or
           has(bb, RD_INF) or
           has(bb, RD_NINF) then
             userinfo(1, "'HardwareFloats' failed"):
             useHardwareFloats:= FALSE;
             break;
        end_if:

        if traperror((
              result:= hfa::matlinsolve(AA, bb)
           )) = 0 then
            if has(result, RD_NAN) or
               has(result, RD_INF) or
               has(result, RD_NINF) then
                  userinfo(1, "'HardwareFloats' failed"):
                  useHardwareFloats:= FALSE;
                  break;
            end_if:
        else // proceed to the branch useHardwareFloats = FALSE
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
        end_if;
        // The kernel dimension kdim may be needed at the
        // end of this routine for conversion to other types
        kdim:= op(result[2], [0, 3, 2]);

     end_if;  // end of if sparsemode
  end_case; 
  //--------------------------------
  // end of useHardwareFloats = TRUE
  //--------------------------------

  // use case instead of if useHardwareFloats = FALSE,
  // because we want to use break
  case useHardwareFloats
  of FALSE do
    //---------------------------------
    //---- use SoftwareFloats ---------
    //---------------------------------
    if HardwareFloatsRequested and dowarn then
       warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
    end_if;

    userinfo(1, "using software floats"):

    if A::dom::hasProp(Cat::Matrix)=TRUE and
       A::dom::constructor = Dom::Matrix then
         Amap:= (Dom::Matrix())::mapNonZeroes;
         Aop := (Dom::Matrix())::nonZeroOperands;
    else Amap:= map;
         Aop := op;
    end_if;
    // convert dense matrices to domtype DOM_ARRAY for speed 
    if A::dom::hasProp(Cat::Matrix)=TRUE then 
       if A::dom::constructor = Dom::Matrix then
          if A::dom <> Dom::Matrix() then
             A:= Dom::Matrix()(A);
          end_if;
       else A:= expr(A)
       end_if;
    end_if; 

    if b::dom::hasProp(Cat::Matrix)=TRUE and
       b::dom::constructor = Dom::Matrix then
         bmap:= (Dom::Matrix())::mapNonZeroes;
    else bmap:= map;
    end_if;
    // convert dense rhs to domtype DOM_ARRAY for speed 
    if b::dom::hasProp(Cat::Matrix)=TRUE then
       if b::dom::constructor = Dom::Matrix then
          if b::dom <> Dom::Matrix() then
             b:= (Dom::Matrix())(b);
          end_if;
       else b:= expr(b);
            if not contains({1,2},op(b,[0,1])) then 
               error("second argument: wrong format");
            end_if;
       end_if;
    end_if;

    // force evaluation of A and b (both are arrays now) 
    A:= Amap(A, eval):
    b:= bmap(b, eval);

    use_normal:=FALSE;  // default 

    if method=2 then
      tmp:= Amap(A, float):
      B:= bmap(b, float):
      coefftypes := map({Aop(tmp)}, domtype);
      if coefftypes // union {op(map(B,domtype))} 
          minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT} <> {} then
          if dowarn then
            warning("symbolic coefficient found, switching to symbolic mode");
          end_if;
          method:=1;B:=NIL;
      else A:=tmp; b:=B; B:=NIL:
      end_if;  
    end_if;

    use_normal := FALSE;
    if method=1 and not dont_use_normal_requested then
      coefftypes := map({op(A)}, domtype);
      if coefftypes minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_RAT} <> {}
        then use_normal:=TRUE
      end_if;
    end_if;

    userinfo(10,
             if method = 2 then
                  "using numerical mode"
             elif use_normal then
                  "using symbolic mode with normalization"
             else "using symbolic mode without normalization"
             end_if
            );

    normal2:= x -> normal(x, Expand = FALSE);
    if method=1 then
      userinfo(1,"symbolic mode: no pivoting unless necessary");
    end_if;
    if method=2 then userinfo(1,"numerical mode: using column pivoting"); end_if;

  // -- initialization done --- 

  kdim:=0;  // dimension of kernel 
  np2:=n+2;
  lowerbands:=0:       // count sudiagonal bands and store in lowerbands: 
  x:=genident("x"):    // free identifiers as unknown in polynomials to   
  y:=genident("y"):    // avoid conflicts with global variables           

  //-----------------------------------------------
  // convert A from polys = columns to polys = rows.
  // Use p as container. Store the j-th element in p[j]
  // as the coefficient of x^(n+2-j) (j=1.. n )
  //-----------------------------------------------
  m:= neqs; // dimension(A) = m x n
  // table p = container for the rows
  // Initialize as a table
  p := table();
  if A::dom::hasProp(Cat::Matrix)=TRUE and
     A::dom::constructor = Dom::Matrix then
      // In analogy to (Dom::Matrix())::transpose:
      // For sparse matrices the version using L:= poly2list(x[j])
      // is faster than the version using coeff(x[i], j).
      // For dense matrices it is the other way round.
      // Regard a matrix as 'reasonably sparse', if at most
      // half its entries are nonzero:

      if A::dom::nonZeroes(A) < m*n/2
      then //sparse case
        A:= extop(A, 3);
        for j from 1 to n do
          L:= poly2list(A[j]);
          for list in L do
            // fill the rows
            p[list[2]][j]:= [list[1], np2 - j];
            if list[2] - j >lowerbands then 
               lowerbands:=  list[2] - j; 
            end_if;
          end_for;
        end_for;
        // So far, the row p[i] is a table created implicitly by
        // p[list[2]][j]:= [list[1], np2 - j];
        // Convert the rows to polynomials.
        for i from 1 to m do
          if type(p[i]) = "_index" then
             // p[i] was not implictly created. This row is empty
             p[i] := poly(0, [x]);
          else
             // convert table p[i] to a list p[i]:
             p[i]:= map([op(p[i])], op, 2);
             // convert list p[i] to a poly p[i]:
             p[i]:= poly(p[i], [x]);
             if use_normal then p[i]:=mapcoeffs(p[i], normal2);end_if;
          end_if;
        end_for;
      else //dense case
        // build up row for row to avoid doubling of memory
        // due to doubly nested sequences
        A:= extop(A, 3);
        for i from 1 to m do
            p[i] := poly([[coeff(A[j],i), np2-j] $j=1..n],[x]);
            if use_normal then p[i]:=mapcoeffs(p[i], normal2);end_if;
        end_for;
        lowerbands:= m - 1;
      end_if;
  else // A is a DOM_ARRAY
      for i from 1 to neqs do // convert equations to univariate polynomials 
        tmp:= 0:
        for j from 1 to n do  // x[j] -> x^(n+2-j),j=1.. n 
           if not iszero(A[i,j]) then 
                  tmp:=tmp+A[i,j]*x^(np2-j): 
                  if i-j>lowerbands then lowerbands:=i-j; end_if;
           end_if;
        end_for;
        p[i]:=poly(tmp,[x]):
        if use_normal then p[i]:=mapcoeffs(p[i],normal2);end_if;
      end_for;
  end_if;

  //-----------------------------------------------
  // convert b from polys = columns to polys = rows.
  // Use B as container. Store the j-th element in B[j]
  // as the coefficient of y^(j-1) (j=1.. n )
  //-----------------------------------------------
  // B = container for the rows of the rhs
  B := table();
  if b::dom::hasProp(Cat::Matrix)=TRUE and
     b::dom::constructor = Dom::Matrix then
      if b::dom::nonZeroes(b) < m*cb/2
      then //sparse case
        b:= extop(b, 3);
        for j from 1 to cb do
          L:= poly2list(b[j]);
          for list in L do
            // fill the rows
            B[list[2]][j]:= [list[1], j-1];
          end_for;
        end_for;
        // So far, the row B[i] is an implicitly created table.
        // Convert the rows to polynomials.
        for i from 1 to m do
          if type(B[i]) = "_index" then
             // B[i] was not implictly created. This row is empty
             B[i] := poly(0, [y]);
          else
             // convert table B[i] to a list B[i]:
             B[i]:= map([op(B[i])], op, 2);
             B[i]:= poly(B[i], [y]);
          end_if;
        end_for;
      else //dense case
        // build up row for row to avoid doubling of memory
        // due to doubly nested sequences
        b:= extop(b, 3);
        for i from 1 to m do
            B[i] := poly([[coeff(b[j],i), j-1] $j=1..cb],[y]);
        end_for;
      end_if;
  else // b is a DOM_ARRAY
      for i from 1 to neqs do  // convert equations to univariate polynomials 
        tmp:= b[i,1];
        for j from 2 to cb do           // store rows of b as poly in y 
           if not iszero(b[i,j]) then tmp:=tmp+b[i,j]*y^(j-1); end_if;
        end_for;
        B[i]:=poly(tmp,[y]):
      end_for;
  end_if;

  //----------------------------------------------------
  // Row compression: get rid of all trivial equations
  //----------------------------------------------------
  count:= 0;
  m:= neqs;
  for i from 1 to neqs do
      if iszero(p[i]) and iszero(B[i]) then
         m:= m - 1;
      else
         count:= count + 1;
         p[count]:= p[i];
         B[count]:= B[i];
      end_if;
  end_for;
  for i from m+1 to neqs do
      delete p[i], B[i];
  end_for;
  neqs:= m;

  //----------------------------------------------------
  // check = utility to check rhs of equations 0 = b:
  //----------------------------------------------------
  constraints:= {};

  check:=proc(b) // routine to check right hand side of equations 0=b 
  local koeff, tmp;
  begin //b:=subs(b,float(0)=0); 
        if iszero(b) then return(null()) end_if;
        if showassumptions=FALSE then return(FAIL); end_if;
        // Now showassumptions = TRUE and iszero(b) = FALSE.
        // Investigabe, whether b has symbolic parameters.
        // If b is numeric, then return FAIL (the system is not solvable),
        // otherwise insert contraint 0=b into constraints.
        koeff:=[coeff(b)]:     // b is a polynomial in y 
        for tmp in koeff do    
          if not iszero(tmp) then
            if hastype(tmp,{DOM_EXPR,DOM_IDENT,"_index"})
               then constraints:=constraints union {numer(tmp)};
               else return(FAIL):
            end_if;
          end_if;
        end_for;
        return(b);
  end_proc;

 //  Conversion of matrices to sparse univariate poly representation done 

  PivotAssumptions:= {};
  triv:= table();
  kernindex:= table();

/*
  //--------------------------------------
  // Pre-conditioning for the numerical
  // computation: normalize alls rows to 1
  //--------------------------------------
  if method = 2 then
     for i from 1 to neqs do
        tmp:= norm(p[i]);
        if not iszero(tmp) then
           p[i]:= mapcoeffs(p[i], _mult, 1/tmp):
           B[i]:= mapcoeffs(B[i], _mult, 1/tmp):
        end_if;
        B0[i]:= norm(B[i]);
     end_for:
  end_if;
*/

  //------------------------------
  // Start Gauss-Elimination 
  //------------------------------
   
  //for j from 1 to n do // eliminate column j
  j:= 0;
  while j < min(neqs, n) do
    j:= j + 1;
    np2mj:=np2-j;
    jpl:=min(j+lowerbands,neqs):

   // ------------------- search for pivot element ------------------ 
   // ------- initialization: collect piv[j],..,piv[jpl] -------- 

    pivot:=0;pivindex:=j; minlength:=RD_INF;
    (piv[i]:= 0;) $ i=j..max(neqs,n):
    /* store element j from p[i] = i.th row in piv[i] and
      truncate p[i] to become the remainder of the row */
    for i from j to jpl do
        piv[i]:= coeff(p[i],np2mj);
        // Make sure, vanishing piv[i] are identified as 0:
        if use_normal and j=1 and not iszero(piv[i])
           then piv[i]:=normal2(piv[i]);
//              mapcoeffs(p[i], normal2);
        end_if;
        /* Must cut piv[i] from p[i], even if iszero(piv[i]). Note that
          piv[i]=float(0)<>0 might happen. */
        if not iszero(piv[i]) then p[i]:=lmonomial(p[i],Rem)[2]: end_if;
    end_for;

    // ------- search for best pivot element----------------- 
    // piv[j],..,piv[jpl] are candidates for the pivot elements.
    for i from j to jpl do

        if method=1 then  /*symbolic mode: take the "simplest" nonsymbolic
                           element<>0 as pivot */
           if (not iszero(piv[i])) then
             // look for non-symbolic Pivot elements only 
             if indets(piv[i],PolyExpr)={} then
               if iszero(p[i])
               then /*remaining row is trivial. This piv[i] is the ideal
                     candidate for a pivot element: accept it! */
                    pivindex:=i; break;   //stop pivot search 
               else /*remaining row is nontrivial. Look for rows
                     with pivot element of smallest complexity */
                    length(piv[i]):
                    if last(1)<minlength
                       then minlength:=last(1); pivindex:=i:
                    end_if;
               end_if;
             end_if;
           end_if;
        end_if;

        if method=2 then //numeric mode: do row pivoting 
           if not iszero(piv[i]) then
              tmp:= [coeff(p[i])];
              if nops(tmp) = 0 or iszero((
                  rownorm:=max(op(map([coeff(p[i])],specfunc::abs)))
                   )) then //remaining row is trivial 
                   pivindex:=i; break;  //stop pivot search 
              else specfunc::abs(piv[i])/rownorm:
                   if pivot<last(1) then pivot:=last(1);pivindex:=i;end_if;
              end_if;
           end_if;
        end_if;

    end_for;

    /* so far we tried to avoid a symbolic pivot element. If no such
      pivot element exists, we have to start again. This time we must
      accept symbolic elements. */
    if method=1 and iszero(piv[pivindex]) then
      pivot:=0; pivindex:=j; minlength:=RD_INF;
      for i from j to jpl do
        if not iszero(piv[i]) then
          if iszero(p[i]) then pivindex:=i; break;
          else length(piv[i]):
               if last(1)<minlength
                  then minlength:=last(1); pivindex:=i:
               end_if;
          end_if;
        end_if;
      end_for;
    end_if;

    //------------------------------------------------
    // optimal Pivot element found, now exchange rows 
    //------------------------------------------------

    if j<>pivindex then
      ([p[j],p[pivindex]]):=[p[pivindex],p[j]]:
      ([B[j],B[pivindex]]):=[B[pivindex],B[j]]:
      ([piv[j],piv[pivindex]]):=[piv[pivindex],piv[j]]:
    end_if;

/* ----- if symbolic pivot must be used, this is assumed to be <>0.
        Store this assumption in PivotAssumptions: 
--------*/

//  if method=1 and showassumptions=TRUE and
//     indets(piv[j],PolyExpr)<>{} then
//     tmp:= numer(piv[j]);
//     if indets(float(tmp),PolyExpr)<>{} then
//        PivotAssumptions:= PivotAssumptions union {tmp};
//     end_if;
//  end_if;
    if method=1 and showassumptions=TRUE then 
       if // indets(piv[j],PolyExpr)<>{} or
          numeric::isnonzero(piv[j]) = UNKNOWN then
          tmp:= numer(piv[j]);
          if indets(float(tmp),PolyExpr)<>{} or
             numeric::isnonzero(tmp) <> TRUE then
             PivotAssumptions:= PivotAssumptions union {tmp};
          end_if;
       end_if;
    end_if;

// ----- check if non-trivial pivot element was found ------- 

    if iszero(piv[j]) then
         kdim:= kdim + 1;
         triv[j]:= TRUE;
         kernindex[j]:= kdim;

         // Entzerrungsschritt: fuege eine weitere Gleichung ein 
         // j may have become larger than the number of
         // input equations, i.e., p[j] and B[j] have not
         // been set before:

         if not(iszero(p[j]) and iszero(B[j])) then
            neqs:=neqs+1; 
            piv[neqs]:=0;
            p[neqs]:=p[j]:
            B[neqs]:=B[j];
            // the number of lower bands was increased 
            lowerbands:=max(lowerbands,neqs-j-1);
            jpl:=min(j+lowerbands,neqs):
         end_if;
    else // do the elimination. Column j is stored in piv[j],..,piv[neqs]
         for i from j+1 to jpl do
             if not iszero(piv[i]) then
                tmp:=piv[i]/piv[j]:

                if method = 2 then
                   pnorm:= norm(p[i]);  
                   //Beware: B[i] may contain symbols. In this case,
                   // norm(B[i]) would produce an error.
                   if indets([coeff(B[i])]) = {} 
                   then 
                      Bnorm:= norm(B[i]);
                   else                      
                      Bnorm:= 0;
                   end_if: 
                end_if;

                if iszero(tmp-1) or 
                   (method = 2 and specfunc::abs(tmp - 1) < 10^(-DIGITS)) then
                   p[i]:=p[i]-p[j];
                   B[i]:=B[i]-B[j];
                elif iszero(tmp+1) or 
                   (method = 2 and specfunc::abs(tmp + 1) < 10^(-DIGITS)) then
                   p[i]:=p[i]+p[j];
                   B[i]:=B[i]+B[j];
                else
                   p[i]:=p[i]-mapcoeffs(p[j],_mult,tmp):
                   B[i]:=B[i]-mapcoeffs(B[j],_mult,tmp):
                end_if;

                //Beware: the original B[j] may have introduced symbols
                // in B[i]. Set Bnorm = 0. In this case, norm(B[i]) will
                // not be computed further down below.
                if method = 2 and
                   indets([coeff(B[i])]) <> {} then 
                   Bnorm:= 0;
                end_if:

                // increase the chance of finding a kernel of the
                // matrix: if a row nearly vanishes, then regard it
                // as a zero row. This will generate a kernel vector.
                if method = 2 and
                   norm(p[i]) <= 10^(-DIGITS)*pnorm then
                   p[i]:= poly(0, [x]);
                   if not iszero(Bnorm) and
                      norm(B[i]) <= 10^(-DIGITS)*Bnorm then
                      B[i]:= poly(0, [y]);
                   else
                      // keep the B[i] without cleaning it
                   end_if;
                end_if;

                if use_normal then 
                    p[i]:=mapcoeffs(p[i],normal2):
                    B[i]:=mapcoeffs(B[i],normal2):
                end_if;
             end_if;
         end_for:
    end_if;
  end_while; // while j < min(neqs, n)

  //-------------------------------------------------
  // we now have neqs equations for n unknowns, where 
  // the first n equations are upper triangular. 
  // Check that the remaining equations are trivial
  //-------------------------------------------------

  for j from n+1 to neqs do
    if check(B[j])=FAIL then 
      if showassumptions=FALSE then 
           return([FAIL, NIL]);
      else return([FAIL, NIL, [], []]);
      end_if;
    end_if;
  end_for;
  constraints:=[op(constraints)];

  //-----------------------------
  // Fill in trivial equations to
  // obtain a quadratic scheme:
  //-----------------------------
  for j from neqs + 1 to n do
      triv[j]:= TRUE;
      piv[j]:= 0:
      p[j]:= poly(0, [x]):
      B[j]:= poly(0, [y]):
      kdim:= kdim + 1;
      kernindex[j]:= kdim;
  end_for;

/*--------------------
  // this is a little utility needed during
  // the design of this function. It checks
  // that above no unexpected indexed calls to
  // rows occured that were not properly
  // initialized
// if nops(select(p, hastype, DOM_TABLE)) <> 0 then 
//    warning("p has DOM_TABLE"): 
// end_if;
// if nops(select(B, hastype, DOM_TABLE)) <> 0 then 
//    warning("B has DOM_TABLE"): end_if;
// if nops(select(p, hastype, DOM_TABLE)) <> 0 or
//    nops(select(B, hastype, DOM_TABLE)) <> 0 then
//   print("-------------"): 
//   print(p, B);
//   error(" p or B has DOM_TABLE"): 
// end_if;
------------------------*/

  //--------------------------------------------------------
  // Backsubstitution:
  // Now the system is in upper triangular form. The diagonal
  // is stored in piv[1],..,piv[n]
  //--------------------------------------------------------

  // convert univariate polynomials to expressions
  // for fast sparse backsubstition:  
  // coeff*x^(n+2-i) <-> coeff*x[i] 
  if triv[n] = TRUE then
       X[n]:=y^(cb-1+kernindex[n]);
  else X[n]:= op(B[n],1)/piv[n];
       if use_normal then X[n]:=normal2(X[n]): end_if:
  end_if;
  for i from n-1 downto 1 do
     if triv[i] = TRUE then
          X[i]:=y^(cb-1+kernindex[i]);
     else // pick out only non-trivial coefficients of the poly:
          nTerms:=nterms(p[i]):
          Xvars:=[X[np2-op(nthterm(p[i],j),[1,2])]$ j=1..nTerms]:
          koeff:=[coeff(p[i])]:
          p[i]:=_plus( koeff[j]*Xvars[j] $ j=1..nTerms):
          if piv[i]=1 
            then X[i]:=(op(B[i],1)-p[i]);
            else X[i]:=(op(B[i],1)-p[i])/piv[i]:
          end_if;
     end_if;
  end_for;

  //-------------------------------------------------------------
  // The backsubstitution is finished.
  // Now, X[i] = polynomial expression in y. We need to pick out the
  // various y powers. Reconvert expressions to polys in y for
  // faster access via coeff. 
  // Do not normalize X here, because poly would undo normal   
  //-------------------------------------------------------------

  for i from 1 to n do 
      X[i]:=poly(X[i], [y]);
  end_for;

  //-------------------------------------------------------------
  // The polynomials X[i] store the special solution X0[i, j]
  // (i=1..n, j=1..cb) as the coefficients of y^0, .. , y^{cb -1}
  // and the kernel elements as the coefficients of
  // y^cb, .. ,y^cb + kdim -1
  /* Dense extraction:
     X0:=array(1..n,1..cb,[[coeff(X[i],j-1)$ j=1..cb]$ i=1..n]);
     Kernel:=array(1..n,1..kdim,
           [[coeff(X[i],cb-1+j)$ j=1..kdim]$ i=1..n]);
  */
  //-------------------------------------------------------------

  // sparse extraction:
  if returnType = Dom::Matrix() then
       // use a table (sparse structure!) to write into
       // when extracting the coeffs of the row polys.
       // Further down, the table is passed to
       // (Dom::Matrix())::new for conversion
       // to a sparse matrix
       X0:= table();
       if kdim > 0 then
            Kernel:= table();
       else Kernel:= 0; // array(1..n,1..1,[[0]$n]) 
       end_if;
  else // if arrays or Dom::DenseMatrices are to be returned,
       // it is the cheapest to create arrays right here
       // Note that array -> Dom::DenseMatrix is extremely
       // cheap via (Dom::DenseMatrix())::create)
       X0:=array(1..n,1..cb, [[0$cb]$n]);
       if kdim > 0 then
            Kernel:=array(1..n,1..kdim, [[0$kdim]$n]);
       else Kernel:= 0; // array(1..n,1..1,[[0]$n]) 
       end_if;
  end_if;

  // sparse extraction:
  for i from 1 to n do
     L:= poly2list(X[i]);
     for list in L do
       if list[2] < cb then
            X0[i, list[2] + 1]:= list[1];
       else Kernel[i, list[2] + 1 - cb]:= list[1];
      end_if;
     end_for; 
  end_for;
  if method = 2 and kdim > 0 then
     // for kdim = 0, return 0, not float(0)!
     Kernel:= map(Kernel, float);
  end_if;
  X:= X0:

  if method = 2 then
     X:= map(X, float);
  end_if;
   
  //----------------------------------
  // That's it. The special solution X
  // and the kernel 'Kernel' are extracted.
  // Final simplifications: 
  //----------------------------------
  if use_normal then 
     X:=map(X,normal2); 
     Kernel:=map(Kernel,normal2);
  end_if:
   
  constraints:=map(constraints,
                   proc(x) begin
                    if iszero(x) then null() else x end_if
                   end_proc);

  if showassumptions=FALSE then 
       result:= [X, Kernel];
  else constraints:=map([op(constraints)],
             proc() begin args(1)=0 end_proc);
       PivotAssumptions:=map([op(PivotAssumptions)],
             proc() begin args(1)<>0 end_proc);
       result:= [X, Kernel, constraints, PivotAssumptions];
  end_if;

  end_case; // of useHardwareFloats = FALSE

  //-------------------------
  //------- return ----------
  //-------------------------
  case returnType
  of Dom::Matrix() do
       if result[1] <> FAIL then
          result[1]:= returnType(n, cb, result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType(n, kdim, result[2]);
       end_if;
       break;
  of Dom::DenseMatrix() do
       if result[1] <> FAIL then
          result[1]:= returnType::create(result[1]);
       end_if;
       if result[2] <> 0 and result[2] <> NIL then
          result[2]:= returnType::create(result[2]);
       end_if;
       break;
  of DOM_ARRAY do
       if result[1] <> FAIL then
          if domtype(result[1]) <> returnType then
             result[1]:= array(originalcolrangeA, op(result[1], [0,3]), [op(result[1])]);
          elif colrangeAchanged then
             result[1]:= subsop(result[1], 0 = (2, originalcolrangeA, op(result[1], [0,3])));
          end_if;
          if colrangebchanged then
             result[1]:= subsop(result[1], 0 = (2, op(result[1], [0, 2]), originalcolrangeb));
          end_if;
       end_if;
       if result[2] <> 0 and 
          result[2] <> NIL then
          if domtype(result[2]) <> returnType then
             result[2]:= array(originalcolrangeA, op(result[2], [0,3]), [op(result[2])]);
          elif colrangeAchanged then
             result[2]:= subsop(result[2], 0 = (2, originalcolrangeA, op(result[2], [0,3])));
          end_if;
       end_if:
       break;
  of DOM_HFARRAY do
       if result[1] <> FAIL then
          if traperror((
            if domtype(result[1]) <> returnType then
               result[1]:= hfarray(originalcolrangeA, op(result[1], [0,3]), [op(result[1])]);
            elif colrangeAchanged then
               result[1]:= hfarray(originalcolrangeA, op(result[1], [0,3]), [op(result[1])]);
            end_if;
            if colrangebchanged then
              result[1]:= hfarray(op(result[1], [0,2]), originalcolrangeb, [op(result[1])]);
            end_if;
          )) <> 0 then
             if dowarn then
                warning("cannot return the special solution as an hfarray. ".
                        "Returning an array instead");
             end_if;
             if domtype(result[1]) <> DOM_ARRAY then
                result[1]:= array(originalcolrangeA, op(result[1], [0,3]), [op(result[1])]);
             elif colrangeAchanged then
                result[1]:= array(originalcolrangeA, op(result[1], [0,3]), [op(result[1])]);
             end_if;
             if colrangebchanged then
                result[1]:= array(op(result[1], [0,2]), originalcolrangeb, [op(result[1])]);
             end_if;
           end_if:
       end_if;
       if result[2] <> 0 and 
          result[2] <> NIL then
          if domtype(result[2]) <> returnType then
             result[2]:= hfarray(originalcolrangeA, op(result[2], [0,3]), [op(result[2])]);
          elif colrangeAchanged then
             result[2]:= hfarray(originalcolrangeA, op(result[2], [0,3]), [op(result[2])]);
          end_if;
       end_if:
       break;
  otherwise
       error("unexpected return type"):
  end_case;

  return(result):
end_proc:
