// 

// iterative remez best approximation

// numeric::minimax(f, x=a..b, m, n)
//
// computes P_m/Q_n, a rational best approximation for f(x)
// with x in [a,b].
// Currently, only works over the reals.

// ToDo: find algorithm for complex area
// ToDo: document
// ToDo: option ContFrac to return in the form
//       suggested on p. 151 of Cheney's book.

/*
 An example where the returned mu value is really just
 a lower bound on the error bound:
 
 numeric::minimax(gamma, x=2..3, 0, 1)
 */


// utility: approximate the infinity norm of err(x), returning
// an x maximizing abs(err(x)) in addition to the estimate.
// See Cheney's book "Introduction to approximation theory",
// p. 97, for why we need this.
numeric::_infnorm :=
proc(err, errp, a, b, imax = 50, reclevel = 2)
  local i, l, r, x, x0, infhere, heap;
begin
  heap := adt::Heap();
  heap::insert(RD_INF, [a, b, reclevel+1]);
  while heap::min_element()[3] > 0 do
    // further recursion
    [a, b, reclevel] := heap::delete_min();
    r := a;
    for i from 1 to imax do
      [l, r] := [r, ((imax-i)*a+i*b)/imax];
      x := l...r;
      x0 := (l+r)/2.0;
      // slope evaluation
      infhere := op(hull(abs(//err(x0)
                           //+ errp(x)*(x-x0)
                          //intersect
                          err(x))), 2);
      heap::insert(-infhere, [l, r, reclevel-1]);
    end_for;
  end_while;
  [-heap::min_pair()[1],
   heap::min_element()[1], heap::min_element()[2]];
end_proc:

// best approximations are characterized (in the nondegenerate
// case) by the equioscillation property.  To check for this,
// we need to count the number of sign changes in a list:
numeric::_count_signchanges :=
proc(l)
  local signs, i;
begin
  signs := map(l, sign);
  _plus(if signs[i] = signs[i+1] then 0 else 1 end_if
       $ i = 1..nops(signs)-1);
end_proc:

// The following routine tries to find alternations; to be more
// precise, it returns [x[0], x[1], ..., x[n]] s.t. 
// sign(f(x[i])) * sign(f(x[i+1])) = -1.
numeric::_find_crit :=
proc(f, df, x, a, b, N)
  local L, heap, fL, i, imax, l, r, 
       infhere, iv, x0, findplace, inserted_sth;
begin
  findplace := // return largest i s.t. L[i] <= x
  proc(x)
    local i;
  begin
    i := 0;
    while i < nops(L) and L[i+1] <= x do
      i := i+1;
    end_while;
    i;
  end_proc:
  imax := 5:
  
  heap := adt::Heap();
  L := [float(a), float(b)];
  fL := [float(f(a)), float(f(b))];
  heap::insert(RD_INF, L);
  while numeric::_count_signchanges(fL) < N and
       heap::nops() > 0 do
    inserted_sth := FALSE;
    while not inserted_sth and
         heap::nops() > 0 do
      [a, b] := heap::delete_min();

      i := findplace(b);
      if not iszero(i) and
        not 0 in f(a...b...L[i]) then
       next;
      end_if;

      x0 := (a+b)/2.0;
      i := findplace(x0);
      
      if iszero(i) then
       if not 0 in f(x0...L[1]) then
         next;
       end_if;
       L := [x0].L;
       fL := [float(f(x0))].fL;
      else
       if numeric::realroot(f(x), x=L[i]..x0) = FAIL then
         next;
       end_if;
       if i < nops(L) and
          numeric::realroot(f(x), x=x0..L[i+1]) = FAIL then
         next;
       end_if;
       L[i..i] := [L[i], x0];
       fL[i..i] := [fL[i], f(x0)];
      end_if;
      
      if abs(b-a) < max(abs(b),abs(a)) * 10.0^(-DIGITS)
       then next; end; // too close for current DIGITS?
      
      r := a;
      for i from 1 to imax do
       [l, r] := [r, ((imax-i)*a+i*b)/imax];
       iv := l...r;
       x0 := (l+r)/2.0;
      // slope evaluation
       infhere := op(hull(abs(f(x0)
                            + df(iv)*(iv-x0))), 2);
       heap::insert(-infhere, [l, r]);
      end_for;
      inserted_sth := TRUE;
    end_while;
  end_while;
  // now remove superfluous x from L
  for i from 1 to nops(L) do
    while nops(L) > i and
         sign(fL[i]*fL[i+1]) = 1 do
      if abs(fL[i]) > abs(fL[i+1]) then
       delete fL[i+1], L[i+1];
      else
       delete fL[i], L[i];
      end_if;
    end_while;
  end_for;
  return(L);
end_proc:
  
numeric::minimax :=
proc(f, xr, m, n)
  local rettype, x, a, b, r, ret,
        num, den, errtype, err, i, other;
begin
  // TODO: check the input
  
  [x, a] := [op(xr)];
  [a, b] := [op(a)];
  
  if not testtype(f, Type::Function) then
    f := fp::unapply(f, x);
  end_if;
  
  other := [args(5..args(0))];
  rettype := Normal;
  i := contains(other, hold(Contfrac));
  if i>0 then
    rettype := hold(Contfrac);
    delete other[i];
  end_if;
  
  errtype := AbsoluteError;
  i := contains(other, AbsoluteError);
  if i>0 then delete other[i]; end_if;
  i := contains(other, RelativeError);
  if i>0 then
    if numeric::realroot(f(x), xr) <> FAIL then
      error("RelativeError only supported for functions without zeroes");
    end_if;
    
    errtype := RelativeError;
    delete other[i];
  end_if;
  
  if FALSE and iszero(n) then
    [num, err] := numeric::minimax::_poly(f, x, a, b, m, errtype);
    den := 1;
  else
    [num, den, err] :=
    numeric::minimax::_rat(f, x, a, b, m, n, errtype);
  end_if;
  
  if rettype = Normal then
    num := num / lcoeff(den);
    den := den / lcoeff(den);
    [expr(num)/expr(den), err];
  else
    r := normal(expr(num)/expr(den));
    num := numer(r); den := denom(r);
    ret := table();
    while not iszero(den) do
      [num, r, den] := [den, divide(num, den)];
      ret[nops(ret)] := poly(r, [x])(x); // Horner scheme
    end_while;
    r := ret[nops(ret) - 1];
    for i from nops(ret) - 2 downto 0 do
      r := ret[i]+1/r;
    end_for;
    [r, err];
  end_if;
end_proc:

// for userinfo, the other two cases are slots below numeric::minimax:
numeric::minimax := funcenv(numeric::minimax):

// Computing a minimax polynomial is taken from
// W. Fraser: A Survey of Methods of Computing Minimax and
// Near-Minimax Polynomial Approximations for Functions
// of a Single Independent Variable, JACM, 12, 3, pp. 295-314
numeric::minimax::_poly :=
proc(f, x, a, b, m, errtype)
  local k, i, j, xk, fxk, cj, r, sol, err, errp, errpp, errs,
       infnorm, maxi, xmax, xmaxl, xmaxr,
       lambda, newinfnorm, rnew, iterations;
begin
  a := float(a);
  b := float(b);
  // start with a Chebychev approximation
  xk := [float((a+b)/2+(b-a)/2*specfunc::cos(PI*(k-1/2)/(m+1)))
         $ k = 1..m+1];
  fxk := map(xk, float@f);
  cj := [2/(m+1)*_plus(fxk[k]*specfunc::cos(PI*j*(k-1/2)/(m+1))$k=1..m+1)
         $j=0..m];
  r := poly(_plus(cj[j+1]*expr(orthpoly::chebyshev1(j,
                                                    (x-(b+a)/2)/((b-a)/2)))
                  $ j=0..m) - cj[1]/2, [x]);
  
//  plotfunc2d(r(x) - f(x), x=a..b, LegendVisible=FALSE);

  if errtype = RelativeError then
    err := (f - r)/f;
  else
    err := f - r;
  end_if;
  errp := err';

  xk := numeric::_find_crit(err, errp, x, a, b, m+2);
  
  errs := map(xk, float@err);
    
  if nops(xk) > m+2 then
      // find i s.t. abs(errs[i]) is maximal
    maxi := 1;
    for i from 2 to nops(errs) do
      if abs(errs[i]) > abs(errs[maxi]) then
        maxi := i;
      end_if;
    end_for;
    
      // now, find a window of m+2 values around maxi
    maxi := max(1, maxi-floor((m+2)/2));
    maxi := min(maxi, nops(errs) - m - 1);
    
    xk   := xk  [maxi..maxi+m+1];
    errs := errs[maxi..maxi+m+1];
  end_if;
  
  infnorm := numeric::_infnorm(err, errp, a, b)[1];
  
  // improve with Remez
  iterations := 0;
  repeat
    iterations := iterations + 1;
    sol := numeric::matlinsolve(array(1..nops(xk), 1..m+2,
                                      [[float(xk[i]^j)$j=0..m, float((-1)^i)]
                                       $i=1..nops(xk)]),
                                array(1..nops(xk), 1..1,
                                      [[float(f(xk[i]))]
                                       $i=1..nops(xk)]))[1];
    // sol[m+2,1] is an approximation to the error
    // sol[i+1,1] are our new coefficient candidates
    rnew := poly([[sol[i+1,1], i]$i=0..m], [x]);
//    plotfunc2d(r(x), rnew(x), x=a..b, LegendVisible=FALSE);
    lambda := float(1);
    while (newinfnorm :=
           numeric::_infnorm((err := (f - (lambda * rnew
                                          + (lambda - 1) * r)) /
                                    (if errtype = RelativeError 
                                       then f else 1 end)),
                             (errp := err'), a, b)[1]) > (1+1e-5)*infnorm do
      lambda := lambda/4;
      if lambda < 10.0^(3-DIGITS) then
        warning("insufficient convergence, check output carefully");
        return([r, infnorm]);
      end_if;
    end_while;
    
    infnorm := newinfnorm;
    r := lambda * rnew + (lambda - 1) * r;
    
//  plotfunc2d(r(x) - f(x), x=a..b, LegendVisible=FALSE);
    
    errpp := errp';
    
    // adjust xk with a few Newton steps
    for i from 1 to 3 do
      xk := map(xk, proc(x0)
                    begin
                      if traperror((x0 := x0-errp(x0)/errpp(x0))) <> 0
                        then x0 := x0 + stats::normalRandom(0,1/100)()*(b-a);
                      end_if;
                      min(max(x0, a), b);
                    end_proc);
    end_for;
    
    if xk = [] or
       sign(err(a)*err(xk[1])) = -1 and
       err(a)/err(xk[1]) > 10.0^(-DIGITS) then
      xk := [a].xk;
    end_if;
  
    if xk = [] or
       sign(err(b)*err(xk[-1])) = -1 and
       err(b)/err(xk[-1]) > 10.0^(-DIGITS) then
      xk := xk.[b];
    end_if;
    
    errs := map(xk, float@err);
    
    for i from 1 to nops(xk)-1 do
      while nops(errs) > i and
            sign(errs[i]*errs[i+1]) = 1 do
        if abs(errs[i]) > abs(errs[i+1]) then
          delete xk[i+1];
          delete errs[i+1];
        else
          delete xk[i];
          delete errs[i];
        end_if;
      end_while;
    end_for;
    
    if nops(xk) > m+2 then
      // find i s.t. abs(errs[i]) is maximal
      maxi := 1;
      for i from 2 to nops(errs) do
        if abs(errs[i]) > abs(errs[maxi]) then
          maxi := i;
        end_if;
      end_for;
      
      // now, find a window of m+2 values around maxi
      maxi := max(1, maxi-floor((m+2)/2));
      maxi := min(maxi, nops(errs) - m - 1);
            
      xk   := xk  [maxi..maxi+m+1];
      errs := errs[maxi..maxi+m+1];
    end_if;

    // Cheney, p. 97
    [infnorm, xmaxl, xmaxr] :=
    numeric::_infnorm(err, errp, a, b);
    if infnorm > 1.01 * max(map(errs, abs)) then
      xmax := (xmaxl + xmaxr)/2;
      if xmax < xk[1] then
       xk := [xmax].xk[1..-2];
       errs := [err(xmax)].errs[1..-2];
      else
       // find max i s.t. xk[i] < xmax
       for i from nops(xk) downto 1 do
         if xk[i] < xmax then break; end_if;
       end_for;
       
       if sign(err(xmax)) = sign(errs[i]) then
         xk[i] := xmax;
         errs[i] := err(xmax);
       elif i = nops(xk) then
         xk := xk[2..-1].[xmax];
         errs := errs[2..-1].[err(xmax)];
       else
         xk[i+1] := xmax;
         errs[i+1] := err(xmax);
        end_if
      end_if;
    end_if;
    
  until max(map(errs, abs))
        < 1.05 * min(map(errs, abs)) or
       iterations > 10*m end_repeat;
  
  [r(x), infnorm];
end_proc:

// rational approximations are pretty similar,
// with the initital guess taken from an L_2 approximation
// on discrete points, refined by the algorithm of Fraser
// and Hart.
numeric::minimax::_rat :=
proc(f, x, a, b, m, n, errtype)
  local num, den, newpol, err, errp, errpp, errs, rnd,
        i, j, k, xk, fxk, sol, N, maxerr, infnorm,
       lambda, newinfnorm, iterations, p, lastxk, lastNum, lastDen,
       trans, evalC, chebvals, errs2, maxi, newmaxerr;
begin
  a := float(a);
  b := float(b);
  
  rnd := frandom(77162);

  // transform [a,b] to [-1, 1] for orthpoly::chebyshev1:
  trans := x -> (2*x-(a+b))/(b-a);

  // evaluate sum of Chebychev polynomials:
  evalC := proc(coeffs, x)
            local j;
          begin
            _plus(coeffs[j]*expr(orthpoly::chebyshev1(j-1, trans(x)))
                 $ j = 1..nops(coeffs));
          end_proc:
  
  p := min(m, n)-1;
  
  N := 3*(m+n-2*p+1);
  
  xk := [(a+b)/2 + (a-b)/2*specfunc::cos(PI*(k-1/2)/N)
        $ k = 1..N]: // zeroes of Chebyshev poly

  fxk := map(xk, f);
  
  sol := numeric::leastSquares(matrix([[orthpoly::chebyshev1(j, trans(xk[i]))
                                   $ j = 0..m-p,
                                   -fxk[i]
                                   * orthpoly::chebyshev1(j, trans(xk[i]))
                                   $ j=1..n-p]
                                   $ i = 1..nops(xk)]),
                            matrix([[fxk[i]]  $ i = 1..nops(xk)]))[1];
  
  num := [sol[k+1] $ k=0..m-p];
  newpol := [1, sol[k+m-p] $ k=1..n-p];

  den := poly([[1,0]], [x]);
  lambda := float(1);
  while numeric::realroot((lambda*evalC(newpol, x))+
                       (1-lambda), x=a..b) <> FAIL do
    lambda := lambda/2;
    if lambda < 10.0^(3-DIGITS) then
      warning("insufficient convergence, check output carefully");
      return([evalC(num, x), evalC(den, x), infnorm]);
    end_if;
  end_while;
  den := map(newpol, _mult, lambda);
  den[1] := 1;
  
//  p := p+1;
  p := 1;
  while p > 0 do
    p := p - 1;
    num := num.[0$m-p+1-nops(num)];
    den := den.[0$n-p+1-nops(den)];
    userinfo(1, "Now finding best approximation of degree ".expr2text(m-p).
            " over ".expr2text(n-p));
    N := m+n-2*p+2;
  
  // initial guess computed, now refine via Remez algorithm
    iterations := 0;
    repeat
      lastxk := xk;
      lastNum := num; lastDen := den;
      if errtype = RelativeError then
        err := (f - (x -> evalC(num,x)/evalC(den,x)))/f;
      else
        err   := f - (x -> evalC(num,x)/evalC(den,x));
      end_if;
      errp  := err';
      errpp := errp';
      
      if iterations mod 10 = 0 then
       xk := numeric::_find_crit(err, errp, x, a, b, N);
      else
        // adjust xk with a few Newton steps
       for i from 1 to 3 do
         xk := map(xk, proc(x0)
                       local x1;
                     begin
                       if traperror((x1 := x0-errp(x0)/errpp(x0))) <> 0
                         then x1 := x0 + stats::normalRandom(0,1/100)()*(b-a);
                       end_if;
                       min(max(x1, a), b);
                     end_proc);
       end_for;
      end_if;
      
      xk := listlib::removeDupSorted(sort(xk.[a,b]));
  
      if nops(xk) < N or
        numeric::_count_signchanges(map(xk, float@err)) < N-1 then
        // retry to find values
       xk := numeric::_find_crit(err, errp, x, a, b, N);
        
       // still too few points?
        if nops(xk) < N then
         xk := sort(xk.
                   [a + (b-a)*rnd()
                    $ i=1..N-nops(xk)]);
       end_if;
      end_if;
      
//      // add the point of maximal error
//      [infnorm, xmaxl, xmaxr] :=
//      numeric::_infnorm(err, errp, a, b);
//      xk := listlib::removeDupSorted(sort(xk.[a,b,
//                                         (xmaxl+xmaxr)/2]));
      
      xk := listlib::removeDupSorted(sort(xk.[a,b]));
      
      errs := map(xk, float@err);
      
      for i from 1 to nops(xk) do
       while nops(xk) > N and
             nops(xk) > i and
             sign(errs[i]*errs[i+1]) = 1 do
         if abs(errs[i]) > abs(errs[i+1]) then
           delete errs[i+1], xk[i+1];
         else
           delete errs[i], xk[i];
         end_if;
       end_while;
      end_for;

      if nops(xk) > N then
         // find i s.t. abs(errs[i]) is maximal
       maxi := 1;
       for i from 2 to nops(errs) do
         if abs(errs[i]) > abs(errs[maxi]) then
           maxi := i;
         end_if;
       end_for;
    
       // now, find a window of N values around maxi
       maxi := max(1, maxi-floor(N/2));
       maxi := min(maxi, nops(errs) - N + 1);
    
       xk   := xk  [maxi..maxi+N-1];
       errs := errs[maxi..maxi+N-1];
      end_if;
      
      newmaxerr := - max(map(errs, abs)) * sign(errs[1]);
      infnorm := abs(newmaxerr);
      fxk := map(xk, float@f);

      chebvals := [[orthpoly::chebyshev1(j, trans(xk[i]))
                  $ i = 1..nops(xk)] $ j = 0..max(m-p, n-p)];
      
      repeat
        maxerr := newmaxerr;

        sol := numeric::matlinsolve(
                 array(1..nops(xk), 1..N,
                       [[chebvals[j][i] $ j = 1..m-p+1,
                         ((-1)^i*maxerr-fxk[i])*
                         chebvals[j+1][i] $ j = 1..n-p,
                         (-1)^i]
                        $ i = 1..nops(xk)]),
                 array(1..nops(xk), 1..1,
                       [[fxk[i]]$i=1..nops(xk)]))[1];
      until abs((newmaxerr := sol[N, 1]))
           > 0.9*abs(maxerr) or
           abs(newmaxerr) < 0.1*infnorm
      end_repeat;
           
      newpol := [1, sol[j,1] $ j = m-p+2..N-1];
      lambda := float(1);
      while numeric::realroot((lambda*evalC(newpol,x))+
                           (1-lambda)*evalC(den,x), x=a..b) <> FAIL do
        lambda := lambda/2;
        if lambda < 10.0^(3-DIGITS) then
         warning("insufficient convergence, check output carefully");
         return([evalC(num, x), evalC(den, x), infnorm]);
        end_if;
      end_while;
      err := (f - fp::unapply(expand(lambda*
                            evalC([sol[i,1]$i=1..m-p+1],x)/
                            evalC(newpol, x) +
                            (1-lambda)*
                            evalC(num, x)/
                            evalC(den, x)), x)) /
          (if errtype = RelativeError then f else 1 end_if);
      while (newinfnorm :=
            max(op(map((errs := map(xk, float@err)), abs)),
               // numeric::_infnorm(err, err', a, b)[1],
               null()))
           > (1+1e-5)*infnorm and FALSE do
       lambda := lambda/2;
       if lambda < 10.0^(3-DIGITS) then
         warning("insufficient convergence, check output carefully");
         return([evalC(num, x), evalC(den, x), infnorm]);
       end_if;
       err := (f - fp::unapply(expand(lambda*
                                   evalC([sol[i,1]$i=1..m-p+1],x)/
                                   evalC(newpol, x) +
                                   (1-lambda)*
                                   evalC(num, x)/
                                   evalC(den, x)), x)) /
            (if errtype = RelativeError then f else 1 end_if);
      end_while;
      
      userinfo(Text, 3, "lambda = ", lambda);
      userinfo(Text, 2, "new max error = ", newinfnorm, "  (", infnorm, ")");
      den := [1, lambda*sol[m-p+j, 1] + (1-lambda)*den[j]
             $ j = 2..n-p+1];
      num := [lambda * sol[j, 1] + (1-lambda) * num[j]
             $ j = 1..m-p+1];

      errs2 := sort(map(errs, abs));
      if nops(errs2) >= N then
        userinfo(Text, 3, "error ratio: ", errs2[-1]/errs2[-N]);
      end_if;
      userinfo(Text, 5, "errors: ", errs);
      iterations := iterations + 1;
    until iterations > 10*N or
         (nops(errs2) >= N and
          errs2[-1] < 1.0005*errs2[-N] and
          infnorm < (1+10.0^(-DIGITS/4)) * newinfnorm and
          (numeric::_count_signchanges(errs) >= N-1 or
           (numeric::_count_signchanges(errs) >= N-3 and
            infnorm < (1+10.0^(1-DIGITS)) * newinfnorm))) or
         _lazy_and(op(zip(lastxk, xk,
                        (a, b) -> abs(a-b) < 10.0^(-DIGITS/3)*abs(b))),
                  op(zip(lastNum, num,
                        (a, b) -> abs(a-b) < 10.0^(-DIGITS/3)*abs(b))),
                  op(zip(lastDen, den,
                        (a, b) -> abs(a-b) < 10.0^(-DIGITS/3)*abs(b))))
    end_repeat;
    userinfo(Text, 1, "best approx = ", evalC(num, x)/evalC(den, x));
    userinfo(Text, 1..1, "with max error ", rdfloat2text(newinfnorm, 5, 0));
  end_while;
  
  [evalC(num, x), evalC(den, x), newinfnorm];
end_proc:
