/*


Tips & Tricks (einbauen ????)
 weitere Aufrufsyntax:

 numeric::ode2vectorfield(IVP)

 Finde die 'fields' automatisch im IVP:

 misc::subExpressions( y''(x) + u(x)*v'(x), "D")
                                       {D(v), D(y), D(D(y))}
 map(%, field -> op(field)(x))
                                       {v(x), y(x), D(y)(x)}
*/

/* ------- help page for numeric::ode2vectorfield------------------------
numeric::ode2vectorfield -- 
      convert a set of quasi-linear differential equations
      into input data for numeric::odesolve2

Call:   numeric::ode2vectorfield(IVP, fields)

Parameter:  IVP -- a set or list of quasi-linear differential equations
                   and linear initial conditions. Similar syntax as in
                   other systems, e.g.
                    {u''(x) + x = v(x), v'(x)=u'(x)^2,
                     u(0) = 1, v(0) = 2, u'(0) = u(0) + v(0) }

                   u'(x), u''(x) may also be specified by
                   diff(u(x),x), diff(u(x),x,x) etc. 

            fields -- a list of all unknown functions (minus the
                      highest derivatives) occuring in IVP, e.g.
                       [u(x), u'(x), v(x)]

Return Value:  the sequence f, t0, Y0.
               f is a procedure as required by numeric::odesolve2,
               t0 is the initial time, Y0 is a list of initial
               values for the fields specified in 'fields'.

               This sequence can be directly passed 
               to numeric::odesolve2:

               Y:= numeric::odesolve2(ode2vectorfield(IVP, fields))

               is the numerical solution of IVP. 

Details: 
   *) "quasi-linear" means, that the highest derivative of each
      dependent function must enter the differential equations
      linearly. Lower derivatives may enter arbitrarily. E.g.
        IVP:= {u''(x) + x = exp(-v(x))*v'(x), v'(x)=u'(x)^2,
               u(0) = 1, u'(0) = 1, v(0) = 2}
        fields:= [u(x), u'(x), v(x)]
      is quasi-linear: the dependent fields are u(x) and v(x),
      their highest derivatives u''(x) and v'(x) enter linearly
      in the odes.

      Note that
         {u''(x) + x = v'(x)^2, v'(x)=u'(x)^2, u(0) = 1, ...}
      is not quasi-linear, because of v'(x)^2.

   *) The set of differential equations must be solvable
      for the highest derivatives. E.g.,
         {u''(x) + v'(x) = 1,
          u''(x) + v'(x) = u(x)*v(x), 
          u(0)=.., u'(0)=.., v(0) = ..}
      will lead to a crash, because you cannot solve uniquely
      for u''(x) and v'(x).

   *) The list of fields must be "complete": if an n-th
      derivative of a dependent function u(x), say, occurs in the
      odes, then u(x), u'(x), .. (up to the (n-1)-st derivative)
      must be specified in the list of fields. E.g., for the
       IVP:= {u''(x) + x = exp(-v(x))*v'(x), v'(x)=u'(x)^2,
              u(0) = 1, u'(0) = 1, v(0) = 2};
      the fields must contain [u(x), u'(x), v(x)].

   *) The specification of 'fields' determines the ordering
      in the output of Y generated by
       Y:= numeric::odesolve2(ode2vectorfield(IVP, fields));
      E.g., all 3 versions

        IVP:= {u''(x) + x = exp(-v(x))*v'(x), v'(x)=u'(x)^2,
               u(0) = 1, u'(0) = 1, v(0) = 2}:
        fields:= [u(x),u'(x),v(x)]:
        Y:= numeric::odesolve2(numeric::ode2vectorfield(IVP,fields)):
        Y(1);
               [1.880383662, 0.5740374221, 2.792010328]

        fields:= [u'(x),v(x),u(x)]:
        Y:= numeric::odesolve2(numeric::ode2vectorfield(IVP,fields)):
        Y(1);

               [0.5740374221, 2.792010328, 1.880383662]

        fields:= [v(x),u(x),u'(x)]:
        Y:= numeric::odesolve2(numeric::ode2vectorfield(IVP,fields)):
        Y(1);

              [2.792010328, 1.880383662, 0.5740374221]

      work, only the ordering of the output 
        Y(some_time) --> [value1, value2, value3]
      changes.

   *) The initial conditions may be arbitrary *linear*
      equations in the initial values of the fields (as specified 
      by the list 'fields'). 
      It must be complete: the set of initial conditions must be 
      solvable for the initial values of the fields. E.g.,

       IVP:=  {u''(x) + x = v'(x)^2, v'(x)=u'(x)^2,
               u'(0) + v(0) = 1, 3*u(0) + v(0) = 0, v(0) =5 };
       fields:= [u(x), u'(x), v(x)];
       Y:= numeric::odesolve2(numeric::ode2vectorfield(IVP,fields)):
       Y(1);

           [-8.758075809, -11.59140914, 60.01371628]

       is ok. In the following an error occurs:

       IVP:=  {u''(x) + x = v'(x)^2, v'(x)=u'(x)^2,
               u'(0) + v(0) = 1, 3*u(0) + v(0) = 0 };
       fields:= [u(x), u'(x), v(x)];
       numeric::ode2vectorfield(IVP, fields);

       Error: not enough initial conditions [numeric::ode2vectorfield]

      Reason: you need values for u(0), u'(0) and v(0), but
      there are not enough data in the IVP.
         
Example 1:

    IVP:= {y'(x) = x*y(x), y(0) = 1}:
    data:= numeric::ode2vectorfield(IVP, [y(x)]):
    Y:= numeric::odesolve2(data):  // Y(x) = [y(x)]
    Y(0), Y(1), Y(2);

              [1.0], [1.64872127], [7.389056098]

Example 2:

    IVP:= { y'(x) = y(x)*v'(x) + u(x),
            y'(x) = u(x) - v(x),
            u(x)*u'(x) = x*v'(x) + y(x),
            y(0) = 1, u(0) = 2, v(0) = 3 }:
    data:= numeric::ode2vectorfield(IVP, [u(x), v(x), y(x)]);
    Y:= numeric::odesolve2(data):  // Y(x) = [u(x), v(x), y(x)]
    Y(0), Y(PI), Y(5);

       [2.0, 3.0, 1.0],
          [3.990084413, 0.4552957825, 5.930257615],
          [7.286256466, 0.3738541775, 15.45772073]

Example 3:
     
     IVP:= {u''(x) + x = v(x), 
            v'(x)=u'(x)^2,
            u(0) = 1,
            v(0) = 2, 
            u'(0) = u(0) + v(0) }:
     data:= numeric::ode2vectorfield(IVP, [u(x), v(x), u'(x)]);
     Y:= numeric::odesolve2(data):  // Y(x) = [u(x), v(x), u'(x)]
     Y(0), Y(0.5), Y(0.9), Y(1.5);

          [1.0, 2.0, 3.0],
             [2.968152207, 9.994168236, 5.469441207],
             [6.402418659, 41.58531552, 13.63665918],
             [90.23154525, 44466.04691, 1436.723553]

Examples 4:
   IVP:= {y'''(x)= y(x) - y''(x)^2, y(0)=1, y'(0)=0, y''(0)=0}
   fields:= [ y(x), y'(x), y''(x)]
   Y:= numeric::odesolve2(numeric::ode2vectorfield(IVP, fields)):
   Y(1);
           [1.153739773, 0.4403812337, 0.7920372396]


Further examples: see tests at the end of this file

----- end of help page for numeric::ode2vectorfield ----- */


numeric::ode2vectorfield:= proc(IVP, fields: DOM_LIST)
option escape;
local n, diff_eqs, initial_conditions, vectorfield,
      f, T, TT, t0, Y0, names, namesdot, i, j, tmp,
      ffields;
begin

   n:= nops(fields); // the number of unknown functions

   //-------------------------------------------------------------
   // are the fields=[y1(t), y2(t), ..] ok ?
   //-------------------------------------------------------------
   // convert parameters in the fields such as y1(t, PI) to
   // floats:
   fields:= map(fields, float):

   // convert [y(t), diff(y(t), t), ..] to [y(t), D(y)(t), ..]
   if has(fields, diff) then
      fields:= map(fields, rewrite, D);
   end_if;
   if map({op(fields)}, type) <> {"function"} then
      error("illegal second argument ".expr2text(fields).
            ". The unknown fields must be specified as ".
            "function calls such as '[y1(t), y2(t), ..]'")
   end_if;
   //-------------------------------------------------------------
   // check the time parameter:
   //-------------------------------------------------------------
   // the set of first arguments of the unknown fields:
   T:= map({op(fields)}, op, 1):
   if nops(T) <> 1 then
      error("wrong specification of the unknown fields to solve for: ".
            "all functions must depend on the same symbolic 'time variable'");
   end_if;
// if T minus Type::ConstantIdents <> T then
//    error("illegal symbolic 'time parameter' ".expr2text(op(T)));
// end_if;
   // TT = the set of *all* arguments of the unknown fields
   // (they may be passed as multivariate expressions y(t1, t2)):
   TT:= map({op(fields)}, op); 
   if T <> TT then
      error("the unknowns to solve for seem to depend ".
            "on more than one parameter"):
   end_if;
            
   // the single symbolic time variable:
   T:= op(T, 1);
   if domtype(T) <> DOM_IDENT and type(T) <> "_index" then 
      error("the 'time variable' must be an identifier ".
            "or an indexed identifier");
   end_if;

   //-------------------------------------------------------------
   // generate symbolic names for the fields
   //-------------------------------------------------------------
   names:=    [genident() $ i=1..n];
   namesdot:= [genident() $ i=1..n];

   //-------------------------------------------------------------
   // is the IVP ok ?
   //-------------------------------------------------------------
   if map({op(IVP)}, type) <> {"_equal"} then
      error("the 1st argument must be a set or list of ".
            "differential equations and initial conditions")
   end_if;

   //-------------------------------------------------------------
   // process the equations
   //-------------------------------------------------------------

   if domtype(IVP) = DOM_LIST then
      IVP:= {op(IVP)};
   end_if;
   // convert diff(y(x), x) etc. to D(y)(x):

   if has(IVP, diff) then
      IVP:= map(IVP, rewrite, D);
   end_if;

   // check that 'fields' is a *complete* list fields for
   // the IVP.
   // E.g.:
   // misc::subExpressions( y''(x) + u(x)*v'(x), "D")
   //
   //                                  {D(v), D(y), D(D(y))}
   // map(%, field -> op(field)(x))
   //                                  {v(x), y(x), D(y)(x)}
   ffields:=  misc::subExpressions(IVP, "D");
   ffields:=  map(ffields, field -> op(field)(T));
   if {op(fields)} <> ffields then
      error("the specified differential equations are equivalent ".
            "to a system of first order equations in the field(s) ".
            expr2text(ffields). ". The equations cannot be converted ".
            "to a dynamical system in the specified field(s) ". 
            expr2text(fields).
            ".  You must specify an ordering of the elements in ".
            expr2text(ffields). " by passing a corresponding list ".
            "as second argument to numeric::ode2vectorfield. ".
            "This list determines the ordering of the numerical ".
            "values returned by numeric::odesolve2.");
   end_if;

   // substitute fields by their symbolic name: D(y)(x) -> y1 etc.:
   IVP:= subs(IVP, [fields[i]=names[i] $ i=1..n]);
   // do it again for the initial conditions: D(y)(0) -> y1 etc.:
   IVP:= subs(IVP, [op(fields[i],0)=names[i] $ i=1..n]);

   // Add further differential equations, if fields contain
   // dervaties. E.g., if fields = [y(x), y'(x)],
   // then add D(y)(x) = y'(x) to the equations:

   for i from 1 to n do
     for j from 1 to n do
        if fields[i] = D(op(fields[j],0))(T)
           then IVP:= IVP union { D(names[j])(T) = names[i] };
        end_if;
     end_for;
   end_for;

   //-------------------------------------------------------------
   // split IVP into differential equations and initial conditions:
   //-------------------------------------------------------------
   // the highest order terms should remain as D(y)(t) = ...
   // Distinguish between odes and initial conditions by checking
   // the appearance of the time paramter t.

   diff_eqs:= select(IVP, has, T);
   initial_conditions:= IVP minus diff_eqs;

   // replace remaining diffs by symbolic names:
   diff_eqs:= subs(diff_eqs, [D(names[i])(T) = namesdot[i] $ i=1..n]);

   // is the system quasi-linear?
   tmp:= map(diff_eqs, eq -> if type(eq) = "_equal" then
                       lhs(eq) - rhs(eq) else eq end_if);
   for i from 1 to n do
       if not has(tmp, namesdot[i]) then
          error("the system does not seem to contain a ".
                "first order differential equation for ".
                "the field ". expr2text(fields[i]));
       end_if;
       if has(map(tmp, diff, namesdot[i]), namesdot[i]) then
          error("the system must be quasi-linear: highest ".
                "derivatives must enter the differential ".
                "equations linearly. The system contains ".
                "a nonlinear equation in ".
                expr2text(D(op(fields[i], 0))(T)));
       end_if;
   end_for:

   if n > nops(diff_eqs) then 
      error("more unknowns than differential equationns")
   end_if;

   //-------------------------------------------------------------
   // convert diff equations to vector field procedure
   //-------------------------------------------------------------
   
   // solve for the first derivatives:
   diff_eqs:= numeric::linsolve(diff_eqs, [namesdot[i] $ i=1..n], Symbolic);
   if diff_eqs=FAIL then
      error("could not convert the initial value problem ".
            "to an equivalent dynamical system. Either the ".
            "differential equations cannot be solved for the ".
            "highest derivatives, or inappropriate initial ".
            "conditions were specified.");
   end_if;
   if nops(diff_eqs) <> n then
      error("number of odes and number of unknowns do not match")
   end_if;

   // sort rhs of 'diff_eqs' in the correct order and store in a
   // list 'vectorfield'.
   // Using numeric::linsolve this should not be necessary
   // (because 'vectorfield' = 'diff_eqs'), but I do it anyway
   // in case you want to replace numeric::linsolve by linsolve or solve.

   vectorfield:= [ NIL $ n ];
   for i from 1 to n do
      for j from 1 to n do
        if op(diff_eqs, [i, 1]) = namesdot[j]
           then vectorfield[j]:= op(diff_eqs, [i, 2]);
                break;
        end_if;
      end_for;
   end_for;
   if has(vectorfield, NIL) then
      error("not enough equations for the unknowns");
   end_if;
  
   // Generate a procedure f representing the vector field:
   // note that 'vectorfield', 'names' and 'n' are in the scope of
   // the following procedure f:
   f:= subsop(proc(t, Y)
		local i;
	      begin
		float(subs(`#vectorfield`, `#substs`))
	      end_proc,
	      [4,1,1] = vectorfield,
	      [4,1,2] = [T=DOM_VAR(0,3), (names[i]=hold(_index)(DOM_VAR(0,4),i)) $ i=1..n],
	     Unsimplified);

   //-------------------------------------------------------------
   // are the initial_conditions = {y1(t0)=y10, y2(t0)=y20, ..} ok ?
   //-------------------------------------------------------------
   if map(initial_conditions, type) <> {"_equal"} then
      error("the initial conditions must be equations")
   end_if;

   t0:= {}:

   // let all calls names(t0) in initial_conditions write
   // their argument to the set t0:
   eval(
   subs(initial_conditions, 
        [(names[i] = proc() begin 
                      t0:= t0 union {args()};
                      return(args());
                     end_proc ) $ i=1..n])
   );

   if nops(t0) <> 1 then
      error("all initial conditions must refer to the same initial 'time'"):
   end_if;
   t0:= op(t0, 1);

   // solve the set of linear equations for the initial conditions:
   initial_conditions:= 
     numeric::linsolve(initial_conditions, [names[i](t0) $ i=1..n], Symbolic);
   if has(initial_conditions, FAIL) then
      error("illegal specification of initial conditions");
   end_if;

   //-------------------------------------------------------------
   // Generate list of initial values [y10,y20,..] from the
   // the initial_conditions {y1(t0) = y10, y2(t0)=y20, ..}
   //-------------------------------------------------------------
   Y0:= [ NIL $ n ]:
   for i from 1 to n do
      for j from 1 to n do
        if op(initial_conditions, [i, 1, 0]) = names[j]
           then Y0[j]:= op(initial_conditions, [i, 2]);
                break;
        end_if;
      end_for;
   end_for;
   if has(Y0, NIL) then
      for j from 1 to n do
          if Y0[j] = NIL then break; end_if;
      end_for;
      error("there is no initial condition for the field ".
            expr2text(op(fields[j], 0)(T)));
   end_if;
/*
   if map({op(Y0)}, domtype@float) minus {DOM_FLOAT, DOM_COMPLEX} <> {}
      then error("illegal symbolic initial conditions")
   end_if;
*/

   //-------------------------------------------------------------
   // Done. The return sequence f, t0, Y0 can be
   // directly passed to numeric::odesolve2
   //-------------------------------------------------------------

   subsop(f, 6=NIL), t0, Y0;
end_proc:
