//----------------------------------------------------------------
/* numerical roots of a univariate polynomial

call:  numeric::polyroots(p, <FixedPrecision>, <Factor>, <SquareFree>, <NoWarning>  )

       p -- a univariate polynomial expression
(or    p -- a univariate polynomial of domain type DOM_POLY)

Synopsis: 
     numeric::polyroots(p) returns a list of numerical approximations of
     all (different) real and complex roots of p. Multiple roots are listed
     according to their multiplicities, i.e, the length of the returned
     list coincides with the degree of p.

     The list is sorted from small real parts to large real parts.

     The coefficients may be arbitrary real or complex numbers.
     Also symbolic coefficients are admissible, if they can be
     converted to floats.

     The trivial polynomial p=0 results in an error message
     (the infinite solution set cannot be represented).

     Constant polynomials p<>0 without roots result in [] (no roots exists).

     The working precision is set by the global variable DIGITS.
     The roots should be exact within this precision.

     The indeterminate in p is determined automatically.

     The roots of a factored input polynomial such as 
     p = sqrt(2)*(x-PI)^2*(x^2-2) are computed factor by factor.

     Warning: in numerically ill-conditioned cases
     a warning is given, if the precision goal DIGITS
     was not achieved for all roots. In this case
     less accurate approximations of the roots are returned.
     In extreme case numeric::polyroots may return FAIL.

     Warning: if the polynomial has float entries, then
     all floats a rationalized by numeric::rationalize,
     before any factorization or numerical root search
     starts. The call 
     >>  numeric::polyroots(p, <Options>)
     is always equivalent to
     >> numeric::polyroots(numeric::rationalize(p, Minimize), <Options>);

     Reference:
        W.H. Press, B.P. Flannery, S.A. Teukolsky, W.T. Vetterling
        Numerical Recipes in C, Cambridge University Press, 1988
        Chapter 9.5 "Laguerre's Method" (page 279).


Option:  With <FixedPrecision> a straightforward internal
         numerical search with fixed precision of 2*DIGITS
         is launched.
         This is the fastest way to obtain roots (it will
         usually be at least twice as fast as without <FixedPrecision>).

         However, degenerate or badly separated roots cannot be
         computed to DIGITS decimal places with fixed precision.

         Example:
         >> p:= poly((x^2-PI/7)^3, [x]):
         >> numeric::polyroots(p, FixedPrecision);

          [- 0.6702279933 - 0.0005250124712 I,
           - 0.6702279932 + 0.00052501249 I,
           - 0.6693177698,
             0.6692820588,
             0.6702458261 - 0.0005560571073 I,
             0.6702458713 + 0.0005560570885 I]

         Without <FixedPrecision> the DIGITS are increased internally:

         >> numeric::polyroots(p);

          [-0.6699245857, -0.6699245857, -0.6699245857, 
            0.6699245857,  0.6699245857,  0.6699245857]

Option: With <Factor> the input polynomial is factored symbolically
        (via factor) before the numerical search starts.
        The numerical algorithm is applied factor by factor.
        Note that this option may be costly. 

Option: With <SquareFree> a squarefree factorization of the input polynomial
        (by polylib::squarefree) is performed before the numerical search
        start. The numerical algorithm is applied factor by factor.
        For polynomials with real rational coefficients this squarefree
        factorization is always performed!
        Note that for other types of coefficients this option may be costly. 
        
Examples:

     >> DIGITS:=16:
     >> numeric::polyroots(0);

        Error: trivial polynomial, solution set is infinite [numeric::polyroots]

     >> numeric::polyroots(1);

                                []

     >> numeric::polyroots(x-sqrt(2)*I);

                         [1.414213562373095 I]

     >> numeric::polyroots(3*y^2-PI);

                [-1.023326707946488, 1.023326707946488]

     >> numeric::polyroots(z^4+I*z+0.1);

        [- 0.8677397208014193 - 0.532129323624284 I, 
         0.9641582450344705 I, 0.1001004022140976 I,
         0.8677397208014193 - 0.532129323624284 I]

     >> numeric::polyroots( poly( 10^(-12)*x^3 + 1) );

        [-10000.0, 5000.0 - 8660.254037844386 I, 
                   5000.0 + 8660.254037844386 I]

     >> numeric::polyroots( poly( 10^12*x^3-2*I*x+28) );

         [- 0.0003036588971875662 - 0.000000002195445853389855 I,
            0.0001518275472819015 - 0.0002629752213266727 I,
            0.0001518313499056647 + 0.0002629774167725261 I]

     >> numeric::polyroots( poly( 10^(-40)*x^3-2*I*x+28) );

        [- 100000000000000001387.0 - 100000000000000001380.0 I,
         - 14.0 I,
           100000000000000001387.0 + 100000000000000001394.0 I]


     //------------------------------------------------------------
     // The following polynomial has exact coefficients.
     >> numeric::polyroots( expand((x-1)*(x-1/7)^3) );

           [0.1428571428, 0.1428571428, 0.1428571428, 1.0]

     //------------------------------------------------------------
     // The same polynomial as above, but now with float coefficients,
     // which are rationalized internally.
     // The multiple root 1/7 is found with the reduced precision of about 
     //5 digits. This is optimal! 
     // It roughly corresponds to the theoretical precision
     // of DIGITS/multiplicity = 2*10/4 = 5 digits for the multiple root. 
     // Note that no purely numerical algorithm can do better than
     // this, because any roundoff error in the coefficients of the
     // expanded polynomial will change the location of the multiple
     // root by this amount!

     >> DIGItS:=10:
     >> numeric::polyroots( expand((x-1)*(x-1.0/7)^3) );

         [0.142818799 + 0.00006626263295 I, 
          0.142818799 - 0.00006626263295 I,
          0.1429338306, 0.9999999992]

     //------------------------------------------------------------
     >> numeric::polyroots( expand((x-I)*(x-1/7)^3) );

            [I, 0.1428571428, 0.1428571428, 0.1428571428]

     // with coefficients marred by roundoff:
     >> numeric::polyroots( expand((x-1.0*I)*(x-1.0/7)^3) );

        [1.0 I, 
         0.1427703098,
         0.1429005593 - 0.00007529587514 I,
         0.1429005593 + 0.00007529587514 I]

     //------------------------------------------------------------
     // extreme example:
     >> setuserinfo(numeric::polyroots, 1):
     >> numeric::polyroots( expand((x-sqrt(2))^2*(x-1/3)^10) );

        ....

        increasing working precision to DIGITS=20
        increasing working precision to DIGITS=40
        increasing working precision to DIGITS=80
        increasing working precision to DIGITS=160
        increasing working precision to DIGITS=320
        accepting last approximation

        [0.3333333333, 0.3333333333, 0.3333333333, 0.3333333333, 
         0.3333333333, 0.3333333333, 0.3333333333, 0.3333333333, 
         0.3333333333, 0.3333333333, 1.414213562, 1.414213562]
*/

/* Some timings:
    DIGITS:=16:
    p:= orthpoly::legendre(100,x):numeric::polyroots(p);  -->  111 sec
    p:= orthpoly::legendre(100,x):numeric::polyroots2(p); --> 1159 sec
    MV.5:
    Digits:=16: fsolve(orthopoly[P](100,x));              -->  745 sec
*/

//-----------------------------------------
numeric::polyroots:= proc(p)
local X,                                      // the unknown
      macheps,                                // precision goal
      fixedPrecision, doSquareFree, doFactor, // activate options
      doWarn,
      compare, NewtonCheck, prepareOutput,    // auxiliary subroutines
      opt, sqrf, result, roots, roots2, R, lc,// auxiliary variables
      extratypes, i, j, oldDIGITS, n, r, r0;  // auxiliary variables
save DIGITS;
begin
   if args(0)<1 then error("expecting at least one argument"); end_if;

   if p::dom::hasProp(Cat::Matrix)=TRUE then
     p:= expr(p):
   end_if:
   case domtype(p)
   of DOM_SET do
   of DOM_ARRAY do
      if nops(p) <> 1 then
         error("expecting one polynomial");
      end_if:
      p:= op(p, 1);
   end_case;
   p:= subs(p, float(0)=0);
   if iszero(p) then error("trivial polynomial, solution set is infinite") end_if; 

   // investigate options
   fixedPrecision:=FALSE; // default
   doSquareFree:= FALSE;  // default
   doFactor:= FALSE;      // default
   doWarn:= TRUE;         // default
   for opt in [args(2..args(0))] do
     case opt
       of FixedPrecision do fixedPrecision:= TRUE; break;
       of SquareFree do doSquareFree:= TRUE; break;
       of Factor do doFactor:= TRUE; break;
       of NoWarning do doWarn:= FALSE; break;
       otherwise error("unknown option");
     end_case;
   end_for;

   macheps:= 10^(-DIGITS): //relative machine precision = precision goal

   // Search for float entries in p. If such an entry exists,
   // then rationalize the float. Otherwise it does not make
   // sense to increase DIGITS lateron, which would change
   // the float.
   // Do the rationalization before conversion to DOM_POLY
   // to make sure that 
   // numeric::polyroots(p) = numeric::polyroots(numeric::rationalize(p),Minimize).
   // Note that poly(..) will expand the expression with 
   // round off effects.

   // Replace I by some dummy. Then you only need to search
   // for DOM_FLOAT, not for DOM_COMPLEX !
   if hastype(subs(expr(p),I=genident("I")), DOM_FLOAT)
   then p:= numeric::rationalize(p,Minimize);
        userinfo(1,"float found, rationalizing polynomial");
        userinfo(2,"to ".expr2text(p));
   end_if;

   // now the input has exact coefficients: (complex) rational
   // or stuff like exp(5), sqrt(3), sin(exp(2)-4) etc.

//-------------------------------------------------------
// Prepare output: omit small real or imaginary numerical trash, then sort:
prepareOutput:= proc(result_list,macheps)
begin numeric::sort(map(result_list, numeric::complexRound, macheps))
end_proc:
//-------------------------------------------------------

   R:= Expr; // initialization needed in case p = DOM_EXPR
   case domtype(p)
     of DOM_POLY do
        if degree(p)<1 then return([]); end_if;
        if nops(op(p,2))<>1 then error("input must be a univariate polynomial") end_if;
        X:= op(op(p,2));
        R:= op(p, 3): // the coefficient ring
        if R <> Expr then 
           // Convert all coefficients rings to Expr for speed
           p:= poly(poly2list(p),[X]);
        end_if;
        if {op(map(poly2list(p),domtype@float@op,1))} minus 
            {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_INTERVAL}<>{}
           then error("non-numerical coefficients not allowed");
        end_if;
        break;
     otherwise 
        if type(p) = "_equal" then 
           p:= subs(op(p,1)-op(p,2), float(0)=0);
        end_if;
        X:=numeric::indets(p);
        if nops(X)<1 then return([]) end_if;
        if nops(X)>1 then error("argument is not a univariate polynomial") end_if;
        X:=op(X);
        // make use of factored input:" 
        if type(p)="_power" and domtype(op(p,2))=DOM_INT then
           userinfo(2, "input is a power, investigating each factor");
           result:= [op(numeric::polyroots(op(p,1), args(2..args(0)))) $ op(p,2)];
           return(prepareOutput(result,macheps));
        end_if;
        // make use of factored input:" 
        if type(p)="_mult" then
           userinfo(1, "input is a product, investigating each factor");
           p:= select([op(p)], has, X); // eleminate constant factors
           result:= [op(numeric::polyroots(p[i], args(2..args(0)))) $ i=1..nops(p)];
           return(prepareOutput(result,macheps));
        end_if:
        if testtype(p,Type::PolyExpr(X))
           then p:=poly(p,[X]);
           else error("argument is not a univariate polynomial"); 
        end_if;
   end_case;

   // Now p ist DOM_POLY with exact coefficients: either
   // (complex) rationals or numerical expressions (PI, sqrt(2), etc.)

   // Factor out zero solutions:
   i:=0; while iszero(coeff(p,i)) do i:= i+1; end_while;
   if i>0 then p:= p/poly(X^i,[X]); end_if;
   // store zero solutions in result sequence. This will be
   // filled with further solutions below:
   result:= float(0) $ i;

   // make p monic
   lc:= lcoeff(p):
   if float(lc)<>float(1) then 
      if R = Dom::FloatIV then
           // split into lmonomial + remainder
           p:= lmonomial(p, Rem);
           p:= mapcoeffs(lterm(p[1]), _mult, 1...1) +  
               mapcoeffs(p[2], _mult, 1/lc);
      else p:= mapcoeffs(p, _mult, 1/lc) ;
      end_if;
   end_if;

//-------------------------------------------------------

  // Symbolic preprocessing: factorization

  sqrf:= [1,p,1]; // default: no factorization
  if doFactor
  then userinfo(1, "factorizing");
       sqrf:= factor(p);
       sqrf:= Factored::convert_to(sqrf,DOM_LIST);
       userinfo(2, "factorization: ".expr2text(sqrf));
  else // always use squarefree for real integer/rational coefficients. 
       // All other types must be dealt with separately.
       extratypes:= {op(map(poly2list(p),domtype@op,1))} minus {DOM_INT, DOM_RAT};
       if extratypes = {}
         then //real rational coefficients, always use sqrfree
              userinfo(1, "starting squarefree factorization");
              sqrf:= Factored::convert_to(
                  polylib::sqrfree(p),DOM_LIST
              );
              userinfo(2, "squarefree factorization: ".expr2text(sqrf));
     //  elif extratypes minus {DOM_COMPLEX} ={} 
     //       then // complex rational coefficients
     //            if doSquareFree then 
     //               sqrf:=Factored::convert_to( 
     //                   polylib::sqrfree(p),DOM_LIST
     //               );
     //               userinfo(2, "squarefree factorization: ".expr2text(sqrf));
     //            end_if;
     //       else // symbolic coefficients, PI, sqrt(2) etc.
     //            if doSquareFree then 
     //            // sqrf:=polylib::sqrfree(p); 
     //            // userinfo(2, "squarefree factorization: ".expr2text(sqrf));
     //            end_if;
         elif doSquareFree then 
              userinfo(1, "starting squarefree factorization");
              sqrf:= Factored::convert_to(
                  polylib::sqrfree(p),DOM_LIST
              );
              userinfo(2, "squarefree factorization: ".expr2text(sqrf));
       end_if;
  end_if;

//-------------------------------------------------------
   //auxiliary function: compare 2 lists of solutions. Note that the 
   //ordering of the roots may be different!
   compare:= proc(a,b,macheps) local i,j,mindist; 
   begin
     for i from 1 to nops(b) do
       mindist:= min(op(map(a,specfunc::abs@_subtract,b[i]))); 
       for j from 1 to nops(a) do
         if specfunc::abs(a[j]-b[i])=mindist then
            if specfunc::abs(a[j]-b[i])>=macheps*specfunc::abs(b[i])
               then return(FALSE)
               else delete a[j]: 
                    break;
            end_if;
         end_if;
       end_for:
     end_for:
     TRUE;
   end_proc;
//-------------------------------------------------------
// auxiliary function: check, whether a Newton-Step x->x-p(x)/p'(x)
// would satisfy the standard stopping criterion:
   NewtonCheck:= proc(p,roots,macheps) local X, float_p,float_p1, root;
   begin
      X:= op(p,[2,1]):
      // Need to float p again, because DIGITS may have increased!
      // Important, if coefficients contain PI, sqrt(2) etc.
      float_p:=mapcoeffs(p,float);
      float_p1:=diff(float_p,X);
      for root in roots do
          if specfunc::abs(evalp(float_p,X=root)) >
             specfunc::abs(evalp(float_p1,X=root))
            *specfunc::abs(root)*macheps
          then return(FALSE); end_if;
      end_for;
      TRUE;
   end_proc;
//-------------------------------------------------------

// We are ready to go!

   oldDIGITS:= DIGITS;

   // process each factor of the polynomial:
   for i from 1 to (nops(sqrf)-1)/2 do
      userinfo(1, "computing roots of factor ".expr2text(sqrf[2*i]));
      n:= degree(sqrf[2*i]);

      // special case: p = x^n - c
      if nterms(sqrf[2*i]) = 2 then
         DIGITS:= 2*oldDIGITS: 
         r:= (-float(tcoeff(sqrf[2*i]))/float(lcoeff(sqrf[2*i])))^(1/n);
         r0:= exp(2*float(PI)*I/n);
         roots:= [r, 0 $ n-1]; // initialize container for roots
         for j from 2 to n do
             r:= r*r0;
             roots[j]:= r;
         end_for;
         userinfo(2, "roots = ".expr2text(roots));
         result:=result, (op(roots)$ sqrf[2*i+1]);
         next; // that's it, no further checks,
               // doubling of DIGITS etc. necessary
      end_if;

      // For large DIGITS, __laguerre may fail because maxit is
      // fixed to 200. (Experiments have shown that it is dangerous
      // to increase maxit depending on DIGITS, because the run time
      // may grow enormously due to bad convergence of the first steps
      // of the Laguerre method).
      // Strategy: compute first approximations of the roots with
      // DIGITS = 10. Use them as starting points for the
      // search with the actual DIGITS.
      // Sort the starting values from large absolute value to small
      // absolute value to favor convergence towards small roots.
      // This increases the numerical stability of the deflation
      // inside __LaguerreSearch (note that the last roots of the
      // list are deflated first, so sort from large to small).

      roots:= [0 $ n];
      DIGITS:= 10:
      while 2*DIGITS < oldDIGITS do
           userinfo(1, "computing approximations with ".expr2text(DIGITS)." DIGITS"):
           roots:=numeric::__LaguerreSearch(sqrf[2*i],10.0^(-DIGITS),roots, FALSE);
           if roots <> FAIL
           then //deflation process in LaguerreSearch: last values will be polished 
                //first, then deflation. So make sure that last values are small:
                roots:= sort(roots, 
                        ()->bool(specfunc::abs(args(1))>specfunc::abs(args(2))));
                userinfo(2, "approximation: roots = ".expr2text(roots));
                DIGITS:= 2*DIGITS;
           else // no roots found with low DIGITS.
                userinfo(2, "failed to compute roots with DIGITS = ".expr2text(DIGITS));
                // Try again below with the actual DIGITS, use
                // zero starting points for numerical stability
                roots:= [0$n];
                break;
           end_if;
      end_while;

      // First run with double precision
      DIGITS:= 2*oldDIGITS: 
      userinfo(1, "increasing working precision to DIGITS=".expr2text(DIGITS));
      // First approximation of roots with initial values [float(0)$n]
      // to favor convergence to small roots roots. This enhances numerical
      // stability of deflation inside LaguerreSearch.
      // For DIGITS > 10, use the previously computed roots as starting points.
      roots:=numeric::__LaguerreSearch(sqrf[2*i],macheps,roots, FALSE);
      if roots = FAIL then
         userinfo(1, "failed to compute roots, calling numeric::polyroots2");
         DIGITS:= oldDIGITS;
         roots:= numeric::polyroots2(sqrf[2*i]);
         result:=result,(op(roots)$ sqrf[2*i+1]);
         next; // process next square free factor
      end_if;
      userinfo(2, "first approximation: roots = ".expr2text(roots));
      // No need for refinement for linear and quadratic factors.
      // Otherwise, double precision and compare refined roots to
      // single precision roots;
      if not fixedPrecision and degree(sqrf[2*i])>2 then
        // now double precision again, until roots become stationary:
        while TRUE do //double precison
          if NewtonCheck(sqrf[2*i],roots,macheps) then 
             userinfo(1, "accepting last approximation");
             break;
          end_if;
          DIGITS:= 2*DIGITS;
          userinfo(1, "increasing working precision to DIGITS=".expr2text(DIGITS));
          //sort roots from large to small absolute value to increase stability
          //of deflation process in LaguerreSearch: last values will be polished 
          //first, then deflation. So make sure that last values are small:
          roots:= sort(roots, 
                       ()->bool(specfunc::abs(args(1))>specfunc::abs(args(2))));
          roots2:=numeric::__LaguerreSearch(sqrf[2*i],10^(-DIGITS/2),roots, doWarn);
          if roots2<>FAIL then
             userinfo(2, "next approximation: roots = ".expr2text(roots2));
             if compare(roots,roots2,macheps)
                then roots:= roots2;
                     userinfo(1, "accepting last approximation");
                     break;
                else roots:= roots2;
             end_if;
          end_if;
          if roots2=FAIL then //return old roots
             userinfo(1, "failed to compute roots, returning last approximation");
             if doWarn then
                warning("precision goal may not have been achieved");
             end_if;
             break;
          end_if;
          if DIGITS>oldDIGITS*4*n then
             userinfo(1, "limit for DIGITS reached, returning last approximation");
             if doWarn then
                warning("precision goal may not have been achieved");
             end_if;
             break;
          end_if;
        end_while;
      end_if;
      result:=result,(op(roots)$ sqrf[2*i+1]);
   end_for;
   prepareOutput([result],macheps);
end_proc:
//-----------------------------------------
// numerically stable float solution of a quadratic 
// polynomial p = a*x^2 +b*x +c.
numeric::__solveDegree2:= proc(p)
local a,b,c,d;
begin a:=coeff(p,2); b:=-coeff(p,1)/a/2; c:=coeff(p,0)/a;
      d:= (b^2-c)^(1/2); a:= b+d; b:= b-d;
      if specfunc::abs(a)>specfunc::abs(b) then return([a,c/a]) end_if;
      if specfunc::abs(a)=specfunc::abs(b) then return([a, b ]) end_if;
      if specfunc::abs(a)<specfunc::abs(b) then return([b,c/b]) end_if;
end_proc:
//-----------------------------------------
// Control routine for Laguerre search of all roots of polynomial p.
// It finds the first root via __laguerre, then deflates 
// the polynomial, then finds the next root via __laguerre etc.
numeric::__LaguerreSearch:= proc(p,macheps, roots, warning_flag)
local n, p0, j, x;
begin
   p:=subs(mapcoeffs(p,float), float(0)=0);
   n:=degree(p);
   if n=0 then
      if iszero(p) then return(FAIL) else return([]) end_if;
   end_if;
   if n=1 then return([float(-coeff(p,0)/coeff(p,1))]); end_if;
   if n=2 then return(numeric::__solveDegree2(p)); end_if;
   // Make p monic. Note that p is a squarefree factor of the
   // original p. Note that polylib::sqrfree will in general
   // produce non-monic factors.
   if lcoeff(p)<>float(1) then 
       p:= mapcoeffs(p,_mult,1/lcoeff(p)) 
   end_if;
   p0:= p; // copy for undeflated polishing
   for j from n downto 3 do 
       // Find a single root x within relative precision macheps.
       // Starting point should be small to favour convergence towards
       // smallest root (better numerical stability of deflation process)
       x:= numeric::__laguerre(p,j,roots[j],macheps,0); 
       if x<>FAIL then roots[j]:=x else return(FAIL) end_if;
       // Now polish roots using original polynomial p0:
       x:= numeric::__laguerre(p0,n,roots[j],macheps,1);
       if x<>FAIL then roots[j]:=x; 
       else //keep unpolished root with warning
            if warning_flag then
                warning("failed to polish root ".expr2text(roots[j]));
            end_if;
       end_if;
       // root found. Now do forward deflation:
       p:= divide(p, poly([[1,1],[-roots[j],0]],op(p,2)),Quo);
   end_for;
   // The 2 remaining roots by exact formula:
   ([roots[2],roots[1]]):= numeric::__solveDegree2(p);
   for j from 2 downto 1 do // polish last 2 roots
       x:= numeric::__laguerre(p0,n,roots[j],macheps,1);
       if x<>FAIL then roots[j]:=x; 
       else //keep unpolished root with warning
            if warning_flag then
                warning("failed to polish root ".expr2text(roots[j]));
            end_if;
       end_if;
   end_for;
   roots;
end_proc:
//-----------------------------------------
// Utility: fractions used in numeric::__laguerre to break
// limit cycles (see the scheme of 'laguer' in Numerical Recipes)
numeric::polyroots_fractions:= table(
    // 0 = 1/float(PI), 
    0 = 1/2,
    1 = 1/2,  2 =  1/4, 3 = 3/4,  4 = 0.13,
    5 = 0.38, 6 = 0.62, 7 = 0.88, 8 = 1/2):

//-----------------------------------------
// Laguerre method to find *one* root of monic p(X)= a0+a1*X+..+X^n
// within relative precision macheps with initial approximation x of the root.
// The coefficients of p are DOM_FLOAT or DOM_FLOAT + DOM_FLOAT*I.
numeric::__laguerre:= proc(p,n,x,macheps,polish) 
    // p= poly(a0+a1*X+..+X^n,[X])
    // n = degree(p)
    // x = starting point of laguerre iteration
    // polish = 0: stop after some rough approximation is achieved
    // polish = 1: input x is a rough approximation of a root.
    //             Try to improve it by some more Laguerre steps.
local maxit, absx, absdx, absdxold, X, p1, p2, absp, i, frand,
      iter, b, err, d, f, g, g2, h, sq, gp, gm, dx, rootbound, try;
begin
   assert(n = degree(p));
   frand:= frandom(1234):
  //maxit:= max(10*DIGITS*n,200);// max number of Laguerre steps
   maxit:= 200;// max number of Laguerre steps
   absx:= specfunc::abs(x);
   absdxold:= absx;
   X:= op(p,[2,1]);
   p:= subs(p, float(0)=0);
   p1:= diff(p, X);
   p2:= diff(p1, X);
   absp:= mapcoeffs(p, specfunc::abs);
   // all roots satisfy  2*abs(root) <= rootbound:
   rootbound:= 2* max(specfunc::abs(coeff(p,0)), 
                    1+specfunc::abs(coeff(p,i)) $ i=1..n-1);
   for iter from 1 to maxit do
     b:= evalp(p,X=x);
     d:= evalp(p1,X=x);
     // Check, whether a Newton Step x -> x-p(x)/p'(x) = x-b/d would
     // satisfy a stopping criterion:
     if specfunc::abs(b) < macheps*absx*specfunc::abs(d) then return(x) end_if;
     err:= evalp(absp,X=absx); // estimate of absolute error when evaluating p
     // err = absolute roundoff error,
     // err*macheps = relative roundoff error
     // stopping criterion: if abs(b)=abs(p(x))<=relative roundoff,
     // then b is dominated by roundoff. We can't get much better
     // than this: the root precision has reached its theoretical
     // numerical optimum. Return x.
     if specfunc::abs(b)<= 10*err*macheps then return(x) end_if;
     //-------- begin Laguerre step:
     f:= evalp(p2,X=x);
     g:= d/b; 
     g2:= g^2; 
     h:= g2 - f/b;
     sq:= ( (n-1) * (n*h-g2) )^(1/2);
     gp:= g+sq; 
     gm:= g-sq;
     if specfunc::abs(gp) < specfunc::abs(gm) then gp:= gm; end_if;

     //-------------------------------------------------------
     // Break non-converging limit cycles
     //-------------------------------------------------------
     // Ever so often, try to break limit cycles of the Laguerre
     // iteration by taking a fractional step. Here, we follow
     // the scheme of 'laguer' in Numerical Recipes. The funny
     // parameters (do a fractional step every 19-th step, choose 
     // iter div 20 mod 9 to pick the fraction) were chosen to give 
     // best performance in the bench mark tests in TEST/polyrts4.tst
     if iter mod 19 = 0 and not iszero(gp) then
        x:= x - numeric::polyroots_fractions[(iter div 20) mod 9]*n/gp;
        next;
     end_if;

     //-------------------------------------------------------
     // dx=n/gp below should not lead x:=x-dx out of circle for roots.
     if specfunc::abs(gp)*rootbound<n 
        // or (iter mod 2*DIGITS = 0) // replaced by 'break limit cycle' above
        // Note, iszero(gp) happens, when p'(x)=p''(x) = 0
        // Essentially for  p = (x-x0)^n; x close to x0;
        //   (another case is p = x^n - c ; x close to 0)
        // This motivates heuristic choice of new starting point:
        then
          if polish=0 then 
             try:= -(1 + 0.5*frand())*(coeff(p,0))^(1/n);
             if specfunc::abs(x-try)>macheps*specfunc::abs(try)
             then x:= try;
                  next; 
             else return(FAIL);
             end_if;
          else return(FAIL) 
          end_if;
     end_if;
     dx:= n/gp; absdx:= specfunc::abs(dx);
     if absdx<10^(-DIGITS)*absx then return(x) end_if;
     // make sure that you do not drift towards a solution different from x
     // when polishing an approximation of x
     if polish=1 and (absdx >= absdxold) then return(x) end_if;
     // check, whether stopping criterion is satisfied
     x:= x - dx; absx:= specfunc::abs(x);
     if absdx<= macheps * absx then return(x); end_if;
     absdxold:= absdx;
     //-------- end of Laguerre step
   end_for;
   FAIL;
end_proc:
//-----------------------------------------
/* end of file */
